#!/usr/bin/perl

=head1 METADATA

<crontab>
    params_postfix: 2>&1 | tail -1000
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-bs
        <leases>
            mem: 800
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:       checks.direct.yandex.ru
    sharded:    1
    ttl:        20m
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: CRIT
        status: OK
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

<crontab>
    params_postfix: 2>&1 | tail -1000
    time: */3 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.bsExportMaster.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.bsExportMaster.working.sandbox.shard_$shard
    vars:           shard=1
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    bsExportMaster.pl - скрипт, который заполняем таблицу bs_export_queue

=head1 DESCRIPTION

    Опции:
    --help - справка
    --once - отработать только одну итерацию
    --heavy-shake-interval=N - раз в сколько секунд вызывать перетряхивать тяжёлую очередь
    --no-resync — не добавлять объекты из очереди ленивой переотрпавки
    --cid — работать только с указанной кампанией (можно использовать несколько раз)
            оптимизация для бет: если использовать только с одной кампанией, будет брать лок только по этой кампании
            в продакшене и для нескольких кампаний будет брать общий лок

=head1 RUNNING

    LOG_TEE=1 ./protected/bsExportMaster.pl --shard-id 1 --once --no-resync --cid 263

=cut

use strict;
use warnings;
use utf8;


use Yandex::Advmon;
use Yandex::DBTools;

use my_inc "..";

use Settings;
use ScriptHelper script_timer => undef, get_file_lock => undef, sharded => 1, 'Yandex::Log' => 'messages';

use BS::Export;
use BS::ExportMaster;
use EnvTools ();
use LockTools;
use Property;


# время между итерациями 
my $ITERATION_INTERVAL = 60;
# время в секундах между перетряхиванием тяжёлой очереди
my $HEAVY_SHAKE_INTERVAL = 20 * 60;

my ($ONCE, @ONLY_CIDS);
my $RESYNC = 1;
extract_script_params(
    'once' => \$ONCE,
    'heavy-shake-interval=i' => \$HEAVY_SHAKE_INTERVAL,
    'cid=i@' => \@ONLY_CIDS,
    'resync!' => \$RESYNC,
);

my $script = get_script_name(short => 1);

if (!@ONLY_CIDS) {
    my $script_name = get_script_name();
    get_file_lock('dont_die', "$script_name.lock");
}

BS::ExportMaster::init(shardid => $SHARD);
my $lasttime_heavy_shake = time;
my $shake_enabled_prop = Property->new('BS_EXPORT_HEAVY_SHAKE_ENABLED');
while(1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    my $trace = Yandex::Trace->new(service => 'direct.script', method => $script, tags => "shard_$SHARD");

    my $lock_timeout = $ScriptHelper::is_beta ? 1 : 1800;
    my (@sql_lock_guards, @lock_names);
    if (scalar(@ONLY_CIDS) == 1) {
        push @lock_names, map { "BS_EXPORT_MASTER_CID_$_" } @ONLY_CIDS;
    } else {
        push @lock_names, $BS::ExportMaster::LOCK_NAME;
    }
    for my $lock_name (@lock_names) {
        my $sql_lock_guard = eval { sql_lock_guard(PPC(shard => $SHARD), $lock_name, $lock_timeout) };
        if (! $sql_lock_guard) {
            $log->die("Can't obtain mysql lock $lock_name with timeout $lock_timeout: $@");
        }
        push @sql_lock_guards, $sql_lock_guard;
    }

    $log->out("start iteration");

    if (@ONLY_CIDS) {
        $log->out('Working only on cids:', \@ONLY_CIDS);
    }

    my $iter_start_time = time();

    # апдейтим во всех табличках statusBsSynced: No -> Sending
    $log->out("Updating statusBsSynced");
    BS::ExportMaster::update_status_bs_synced($log, \@ONLY_CIDS);

    # переносим заказы в тяжёлую очередь
    $log->out("Moving campaigns into heavy queue");
    BS::ExportMaster::move_heavy_campaigns($log, \@ONLY_CIDS);

    # не чаще, чем раз в N секунд перетряхиваем тяжёлую очередь
    # чтобы не тратить ресурсы понапрасну
    if ($shake_enabled_prop->get(60) && time - $lasttime_heavy_shake >= $HEAVY_SHAKE_INTERVAL) {
        $log->out("Shaking heavy queue");
        BS::ExportMaster::shake_heavy_queue($log, \@ONLY_CIDS);
        $lasttime_heavy_shake = time;
    }

    # добавляем данные из resync
    my $resync_stat;
    if ($RESYNC) {
        # TODO - протащить понимание "можно или нельзя досыпать" из bsExportMonitor
        $log->out("Checking queue stats for resync");
        my $can_resync_queue = BS::ExportMaster::can_resync_queue();

        if ($can_resync_queue || BS::ExportMaster::exists_resync_priority($SHARD)) {

            $resync_stat = BS::ExportMaster::add_resync_items(!$can_resync_queue, $log);
            my $stat_str = %$resync_stat ? join(", ", map {"$_: $resync_stat->{$_}"} sort keys %$resync_stat) : 'nothing';
            my $is_priority_str = $can_resync_queue ? 'both normal and priority' : 'priority only';
            $log->out("Added from $is_priority_str resync queue: $stat_str");
        }
    }

    undef @sql_lock_guards;

    my $iteration_duration = time() - $iter_start_time;

    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
    monitor_values({
        flow              => { bsExportMaster => {
                                                   iteration_duration => { "shard_$SHARD" => $iteration_duration },
                                                   resynced_banners   => { "shard_$SHARD" => ($resync_stat->{banners} // 0) },
                                                   resynced_contexts  => { "shard_$SHARD" => ($resync_stat->{contexts} // 0) },
                                                   resynced_campaigns => { "shard_$SHARD" => ($resync_stat->{campaigns} // 0) },
                                                 },
                             },
    });

    juggler_ok(service_suffix => (EnvTools::is_sandbox() ? 'sandbox' : undef));

    $log->out("iteration end");
    last if $ONCE;

    my $sleep_interval = $ITERATION_INTERVAL - $iteration_duration;
    sleep($sleep_interval) if $sleep_interval > 0;
}

release_file_lock();

