#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */2 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 800
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 6h
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.update_counters_num.by_users_createtime
    raw_events:     scripts.update_counters_num.by_users_createtime.shard_$shard
    sharded:        1
    ttl:            14h
    tag: direct_group_internal_systems
</juggler>


=cut

=head1 DESCRIPTION

    $Id$
    Скрипт для обновления количества счётчика на пользователя из Метрики

=cut

use warnings;
use strict;
use List::Util qw/maxstr/;
use List::MoreUtils qw/all/;

use my_inc "../..";

use ScriptHelper 'Yandex::Log' => 'messages';

use JSON;
use Property;
use Settings;
use ShardingTools;
use Tools ();

use Yandex::DateTime;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::HTTP;
use Yandex::ListUtils;
use Yandex::TVM2;

use constant FETCH_TIMEOUT => 240;
my @METRICA_UPDATED_USER_COUNTERS_NUM_RESPONSE_FIELDS = qw/owner last_update_time counters_cnt/;
my @METRICA_USER_COUNTERS_NUM_RESPONSE_FIELDS = qw/owner counters_cnt/;

# настройки
my $COUNTER_CHANGE_PROP = 'conv_last_counter_change';
my $USER_CREATE_PROP = 'conv_last_user_create';
# класс счётчиков Директа
my $DIRECT_EXT_CLASS = 0;

$log->out("START");

Tools::stop_the_world_for_dst_transition_hour($log);

my $refresh_all_users = 0; # перезабираем счетчики по всем пользователям

# получаем последний lastchange
my $lastchange_prop = Property->new($COUNTER_CHANGE_PROP);
my $lastchange = $lastchange_prop->get();
if ($lastchange) {
    if (datetime($lastchange)->strftime("%Y-%m-%d") ne DateTime->today(time_zone => 'local')->strftime("%Y-%m-%d")) {
        # раз в сутки перезабираем данные за неделю назад
        $lastchange = DateTime->today(time_zone => 'local')->subtract(days => 7)->strftime("%Y-%m-%d");
        # а по пользователям - перезабираем всё
        $refresh_all_users = 1;
        $log->out('refresh_all_users');
    } else {
        # каждую итерацию перезабираем данные за 10 минут
        $lastchange = datetime($lastchange)->subtract(minutes => 10)->strftime("%Y-%m-%d %H:%M:%S");
    }
} else {
    # по-умолчанию - забираем данные за неделю
    $lastchange = DateTime->today(time_zone => 'local')->subtract(days => 7)->strftime("%Y-%m-%d");
}
$log->out("lastchange: $lastchange");

# делаем  запрос по пользователям, у которых менялись счётчики
my $form_param = [
        last_time       => $lastchange,
        external_class  => $DIRECT_EXT_CLASS,
    ];

$log->out('get tvm2 ticket');
my $ticket = eval{Yandex::TVM2::get_ticket($Settings::METRIKA_TVM2_ID)} or $log->die("Cannot get ticket for $Settings::METRIKA_TVM2_ID: $@");

$log->out('get updated counters num');
my $ret = get_counters_list($Settings::METRIKA_UPDATED_USER_COUNTERS_NUM_URL, $form_param, \@METRICA_UPDATED_USER_COUNTERS_NUM_RESPONSE_FIELDS, $ticket) || [];
my $new_lastchange = maxstr $lastchange, grep {defined $_} map {$_->{last_update_time}} @{$ret};
my %users_counters =  map { $_->{owner} => $_->{counters_cnt} || 0 } @{$ret};

my %users_counters_by_shard;
foreach_shard uid => [keys %users_counters], with_undef_shard => 1, sub {
    my ($shard, $uids_chunk) = @_;
    my $data_chunk = hash_cut \%users_counters, $uids_chunk;
    if ($shard) {
        $users_counters_by_shard{$shard} = $data_chunk;
    } else {
        $log->out({data_without_shard => $data_chunk});
    }
};

# делаем запросы по пользователям, которые недавно создавались в директе (параллельно по всем шардам)
$log->out('get counters num for created users');
my $shard_res = foreach_shard_parallel shard => [ppc_shards()], sub {
    my $shard = shift;
    $log->msg_prefix("shard_$shard: ");

    # получаем последний users.createtime
    my $createtime_prop = Property->new(join '_', $USER_CREATE_PROP, 'shard', $shard);
    my ($createtime, $createtime_uid) = (0, 0);
    unless ($refresh_all_users) {
        ($createtime, $createtime_uid) = split ':', $createtime_prop->get() || '0:0';
    }
    $log->out("get created_users with $createtime:$createtime_uid");
    my $current_max_createtime = get_one_field_sql(PPC(shard => $shard), 'SELECT MAX(createtime) FROM users');
    my $created_users = get_all_sql(PPC(shard => $shard), "
                    SELECT uid, createtime
                      FROM users
                     WHERE createtime < unix_timestamp(now() - interval 1 minute)
                       AND createtime >= ?
                       AND (createtime > ? OR uid > ?)
                     ORDER BY createtime, uid
                     LIMIT 10000
                     ", $createtime, $createtime, $createtime_uid);
    my @created_users_uids = map {$_->{uid}} @{$created_users};
    my $ret = [];
    # если есть UID-ы, то делаем запрос в метрику
    if (@created_users_uids) {
        for my $uids_chunk_for_metrika (chunks(\@created_users_uids, 500)) {
            my $uids_list = join (',', @$uids_chunk_for_metrika);
            my $form_param = [
                    uids       => $uids_list,
                    external_class  => $DIRECT_EXT_CLASS,
                ];
            my $ret_chunk = get_counters_list($Settings::METRIKA_USERS_COUNTERS_NUM_URL, $form_param, \@METRICA_USER_COUNTERS_NUM_RESPONSE_FIELDS, $ticket);
            push @$ret, @$ret_chunk;
        }
    }
    my %created_users_counters = map {$_->{owner} => $_->{counters_cnt}} @{$ret};
    my %created_users_counters_count_by_uid = map {$_->{uid} => $created_users_counters{$_->{uid}} || 0} @$created_users;

    if (defined $users_counters_by_shard{$shard}) {
        hash_merge $users_counters_by_shard{$shard}, %created_users_counters_count_by_uid;
    } else {
        $log->out("users_counters_by_shard has not got changes in counters for shard: $shard");
        $users_counters_by_shard{$shard} = \%created_users_counters_count_by_uid;
    }

    my ($new_createtime, $new_uid) = @$created_users ? ($created_users->[-1]->{createtime}, $created_users->[-1]->{uid}) : ();

    # получаем список клиентов, которые есть в директе
    my %direct_users = map {$_->{uid} => $_->{ya_counters}}
                @{get_all_sql(PPC(shard => $shard), [
                    "SELECT u.uid, ifnull(uo.ya_counters,0) as ya_counters
                       FROM users u
                            LEFT JOIN users_options uo ON uo.uid = u.uid",
                    WHERE => {'u.uid' => [grep {/^\d+$/} keys %{$users_counters_by_shard{$shard}}]}
                    ])
                 };
    my @data = map {{ owner => $_, counters_cnt => $users_counters_by_shard{$shard}->{$_} }} grep {defined $direct_users{$_} && $direct_users{$_} != $users_counters_by_shard{$shard}->{$_}} keys %{$users_counters_by_shard{$shard}};

    # если нужно - обновляем данные
    if (@data) {
        $log->out("let's update data for ".scalar(@data)." users");
        my $uid2shard = get_shard_multi(uid => [map {$_->{owner}} @data]);
        my @filtered;
        for my $d (@data) {
            if ($uid2shard->{$d->{owner}} // 0 == $shard) {
                $log->out("update user $d->{owner}: $d->{counters_cnt}");
                push @filtered, $d;
            } else {
                $log->out("skip user $d->{owner}: $d->{counters_cnt}, shard: ".($uid2shard->{$d->{owner}} // 'undef'));
            }
        }

        do_mass_insert_sql(PPC(shard => $shard), "
                    INSERT INTO users_options (uid, ya_counters) 
                    VALUES %s
                        ON DUPLICATE KEY UPDATE ya_counters = values(ya_counters)
                    ", [map {[$_->{owner}, $_->{counters_cnt}]} @filtered]
                    );
    } else  {
        $log->out("no data to update");
    }

    if ($new_createtime) {
        $log->out("set new createtime: $new_createtime:$new_uid");
        $createtime_prop->set("$new_createtime:$new_uid");
        if ($new_createtime == $current_max_createtime) {
            # догнали конец списка (разные uid'ы в расчет не берем)
            juggler_ok(service => "scripts.update_counters_num.by_users_createtime.shard_$shard", description => 'Processed latest createtime');
        }
    } else {
        juggler_ok(service => "scripts.update_counters_num.by_users_createtime.shard_$shard", description => 'no new users in this shard');
    }

};

my @shards_with_errors = grep { ! all { $_ } @{$shard_res->{$_}} } keys %$shard_res;
if (@shards_with_errors) {
    $log->die("Error in shards: " . join ', ', @shards_with_errors);
}

if ($new_lastchange && $lastchange ne $new_lastchange) {
    $log->out("set new lastchange: $new_lastchange");
    $lastchange_prop->set($new_lastchange);
}

juggler_ok();

$log->out("FINISH");

=head2 get_counters_list

    Получение статистики из Метрики, возвращает массив хешей.

=cut

sub get_counters_list {
    my ($api_url, $form_param, $metrica_response_fields, $ticket) = @_;

    $log->out("Fetching new notify data from $api_url");
    my $resp = Yandex::HTTP::submit_form('POST', $api_url, $form_param, timeout => FETCH_TIMEOUT, headers => {'X-Ya-Service-Ticket' => $ticket});
    if ($resp->is_error) {
        my $response_code = $resp->code();
        my $response_status_line = $resp->status_line();
        my $url = Yandex::HTTP::make_url($api_url, $form_param);
        my $error = "Can't fetch statistic from $url. Got $response_code: $response_status_line";
        $log->die($error);
    }
    my $content = $resp->decoded_content;
    $log->out('Got ' . length($content) . ' character from metrica');

    return unless $content;

    # разобрать ответ Метрики
    my $lines = from_json($content);
    $log->out('Got data from metrica:');

    foreach my $line (@{$lines}) {
        my @fields = keys %{$line};
        # если не получили всех нужных полей
        if (@{xminus($metrica_response_fields, \@fields)} > 0) {
            my $error = "Error in metrica response! Array must contain fields: ".join(", ",  @$metrica_response_fields).". Metrica response: ".join(", ",  @fields);
            $log->die($error);
        }
        $log->out($line);
    }

    return $lines;
}
