#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */59 4,5,6,7 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 5000
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 2d4h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Скрипт для сбора статистики для доменов по кликам и показам
    Выполняется раз в сутки

    Собранная статистика показывается во внутреннем отчёте api_domain_stat (все данные) и используется:
        * для изменения минимальной цены MFA доменов (нужны только sum_approx и clicks_approx)
        * для репутации в API и XLS (api_users_units_consumption)

    Статистика пишется в таблицу ppcdict.api_domain_stat

=head1 AUTHOR

#  sergeysl: sergeysl@yandex-team.ru
#  zhur: zhur@yandex-team.ru

=cut

use strict;
use warnings;
use utf8;
use open ':std' => ':utf8';

use List::MoreUtils qw(mesh);
use LWP::UserAgent;

use Yandex::DBTools;
use Yandex::DBShards qw/foreach_shard SHARD_IDS/;
use Yandex::HashUtils;
use Yandex::Interpolate;
use Yandex::TimeCommon;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::ListUtils qw/chunks/;

use my_inc "..";

use ScriptHelper 'Yandex::Log' => 'messages';

use LogTools;
use PrimitivesIds;
use Settings;
use TextTools;
use ShardingTools qw/ppc_shards/;

use Property;

use ServicedClient;

$log->out("start");

my $yesterday = yesterday();
my $today = today();

my $API_STAT_MAX_AGE_IN_TABLE = 365;
my $API_STAT_SELECT_CHUNK_SIZE = 100000;
my $API_STAT_DELETE_CHUNK_SIZE = 1000;

=head2 Разрешённые профили системы контроля баллов

API - собственно API
XLS - Заливка/редактирование баннеров через Excel

создавая новую схему, не забудьте добавить её в список разрешённых значений поля schema таблицы ppc.users_units
и в редактирование пользователя (DoCmdStaff::cmd_modifyUser + Staff::save_edited_user_API_settings)

=cut

my @ALLOWED_API_UNITS_SCHEMES = qw/API XLS/;

=head2 КОЭФФИЦИЕНТЫ

Используются для расчета рейтнигов пользователей и доменов

все коэффициенты равны 1 при нормальных значениях переменных,
меньше единицы, если значечения "плохие"
больше единицы, если значечения "хорошие"

Для расчета рэйтинга домена коэфициенты перемножаются
Для расчета рэйтинга пользователя складываются рейтинги рекламирумых им доменов, умноженные на долю кликов по ним

=cut

=head2 коэффициент BadCTRValues

коэффициент для фраз отклоненных за низкий CTR
"смотрит" на отношение фраз отклоненных за низкий CTR к общему числу фраз

=cut

my $BadCTRValues = [
                        {x => 0,        y => 2},
                        {x => 0.1,      y => 1},
                        {x => 1,        y => 0.1}
];

=head2 коэффициент moderateBannersValues

коэффициент для баннеров отклоненных по модерации
"смотрит" на отношение баннеров отклоненных по модерации к общему числу баннеров

=cut

my $moderateBannersValues = [
                        {x => 0,	y => 5},
                        {x => 0.2,	y => 1},
                        {x => 1,	y => 0.1}
];

=head2 коэффициент domainCTRValues

коэффициент для CTR доменов
"смотрит" на отношение кликов для домена к кол-ву показов

=cut

my $domainCTRValues = [
                        {x => 0, 	y => 0.5},
                        {x => 0.015,y => 1},
                        {x => 0.1,	y => 5}
];

=head2 коэффициент domainClickCostValues

коэффициент для стоимости клика для домена
"смотрит" на отношение средств потраченных по домену к кол-ву кликов для домена

=cut

my $domainClickCostValues = [
                        {x => 0.01,	y => 0.5},
                        {x => 0.1,	y => 1},
                        {x => 2,	y => 3}
];

=head2 коэффициент userRatingValues

коэффициент для расчета кол-ва unit на основании рейтинга пользователя

=cut

my $userRatingValues = [
#                        {x => 0.50,y => 60},
#                        {x => 1,	y => 120},
#                        {x => 15,	y => 600}

# new values for XLS, API uses the same values
                        {x => 0.60, y => 3_200},
                        {x => 1,	y => 32_000},
                        {x => 15,	y => 64_000}
];

=head2 коэффициент $badReasonsValue

    коэффициент для наличия грубых причин отклонения

=cut

my $badReasonsValue = 0.8;

# Проверяем, был ли сегодня уже успешный запуск скрипта
my $last_successed_start = Property->new("get_api_domain_start_date")->get() || '';
if ($last_successed_start ne $today) {
    getBsDomainStat();
    getApiUserDomainStat();
    getApiDomainStat();

    calcUsersRatingAndUnits();

    # выставляем свойство, что скрипт уже запускался сегодня
    Property->new("get_api_domain_start_date")->set($today);

    juggler_ok();
} else {
    juggler_ok(description => 'Already worked today');
    exit(0);
}


#############################################################################################################################################

=head2 функция calcRatingX

 функция для расчета поправочного коэффициента на основе различных данных
 получает на вход значение и описание функции для линейной апроксимизации по трем точкам
 например:
 функция
   при x = 0.015: y = 1        (minx=0.015, miny = 1)
   при x = 0:   y = 0.5        (minx=0,     miny = 0.5)
   при x = 0.1: y = 5          (minx=0.1,   miny = 5)

   отсюда при x=0.05 calcRatingX  возвращает 2.64705882352941

=cut

sub calcRatingX{
	return interpolate_linear(@_);
}


#############################################################################################################################################

=head2 функция calcRating

 функция для расчета рейтингов

=cut

sub calcRating{
    my ($value, $delitel, $values) = @_;

    return 1 if($delitel < 10);

    my $x = $value / $delitel;
    my $r = calcRatingX($x, @$values);
    return $r if($delitel > 50);

    return 1 + ( ($delitel - 10) / (50 - 10) ) * ($r - 1);
}

#############################################################################################################################################

=head2 функция calcUserUnits

 расчет кол-ва баллов на основе пользовательского рейтинга

=cut

sub calcUserUnits{
    my $x = shift;
    return calcRatingX($x, @$userRatingValues);
}

#############################################################################################################################################

=head2 функция calcBadCTRDomainRating

 расчет поправочного коэффициента для домена по фразам, отключенным за плохой CTR

=cut

sub calcBadCTRDomainRating{
    my $domain = shift;
    $domain->{phrases_count} = $domain->{good_ctr_phrases} + $domain->{bad_ctr_phrases};
    return calcRating($domain->{bad_ctr_phrases}, $domain->{phrases_count}, $BadCTRValues);
}

#############################################################################################################################################

=head2 функция calcModerateBannersDomainRating

 расчет поправочного коэффициента для домена по баннерам, отклоненным на модерации

=cut


sub calcModerateBannersDomainRating{
    my $domain = shift;
    $domain->{banners_count} = $domain->{accepted_items} + $domain->{declined_items};
    return calcRating($domain->{declined_items}, $domain->{banners_count}, $moderateBannersValues);
}

#############################################################################################################################################

=head2 функция calcDomainCTRRating

 расчет поправочного коэффициента для домена по CTR

=cut

sub calcDomainCTRRating {
    my $domain = shift;
    return calcRating($domain->{clicks_approx}, $domain->{shows_approx}, $domainCTRValues);
}

#############################################################################################################################################

=head2 функция calcClickCostRating

расчет поправочного коэффициента для средней цены клика

=cut

sub calcClickCostRating {
    my $domain = shift;
    return calcRating(($domain->{sum_approx} / 1_000_000), $domain->{clicks_approx}, $domainClickCostValues);
}

############################################################################################################################################


=head2 функция setDomainRating

 расчет рейтинга домена

=cut

sub setDomainRating{
    my $domain = shift;

    $domain->{ctr_rating} = calcDomainCTRRating($domain);
    $domain->{bad_ctr_rating} = calcBadCTRDomainRating($domain);
    $domain->{bad_banner_rating} = calcModerateBannersDomainRating($domain);
    $domain->{click_cost_rating} = 1;# calcClickCostRating($domain);

    $domain->{rating} = $domain->{ctr_rating} * $domain->{bad_ctr_rating} * $domain->{bad_banner_rating} * $domain->{click_cost_rating};

}

=head2 функция calcUsersRatingAndUnits

 расчет рейтинга и кол-ва баллов для зачисления за день у пользователей

=cut

sub calcUsersRatingAndUnits
{
    my $profile = Yandex::Trace::new_profile('getApiDomainStat:calcUsersRatingAndUnits');
    my %domains;

    # очищаем таблицу от всех данных старше $API_STAT_MAX_AGE_IN_TABLE дней
    _clear_api_domain_stat();

    # читаем данные по доменам за $Settings::API_UNITS_CALC_PERIOD ДНЕЙ по доменам
    my $sql = "select
                        filter_domain,
                        AVG(shows_approx) as shows_approx,
                        AVG(clicks_approx) as clicks_approx,
                        AVG(sum_approx) as sum_approx,
                        AVG(good_ctr_phrases) as good_ctr_phrases,
                        AVG(bad_ctr_phrases) as bad_ctr_phrases,
                        AVG(accepted_items) as accepted_items,
                        AVG(declined_items) as declined_items,
                        SUM(bad_reasons) as bad_reasons
                from
                        api_domain_stat
                where
                        stat_date > DATE_SUB(NOW(), INTERVAL $Settings::API_UNITS_CALC_PERIOD DAY)
                GROUP BY filter_domain";

    my $sth = exec_sql(PPCDICT, $sql);
    while(my $hash = $sth->fetchrow_hashref()){
        for(qw(shows_approx clicks_approx sum_approx good_ctr_phrases bad_ctr_phrases accepted_items declined_items bad_reasons)){
            $hash->{$_} = $hash->{$_} ? $hash->{$_} : 0;
        }
        $domains{$hash->{filter_domain}} = $hash;

        setDomainRating($domains{$hash->{filter_domain}});
    }
    $sth->finish();

    my %users;
    ################################################################################################################################################
    my $xls_out_flag = 0; # выводить ли данные в csv файл, удобно для отладки расчета
    ################################################################################################################################################

    foreach my $shard (ppc_shards()) {
        my $manual_units_users = get_all_sql(PPC(shard => $shard), "select uid, api_units_daily from users_api_options where api_units_daily > 0") || [];
        foreach my $u (@$manual_units_users) {
            $users{$u->{uid}} ||= {};
            $users{$u->{uid}}->{api_units_daily} = $u->{api_units_daily};
            $users{$u->{uid}}->{shard} = $shard if $xls_out_flag;
        }
    }

    # читаем данные по доменам у пользователя за N дней и коэффициенты для добавления баллов
    $sql = "
        select a.uid,
               uao.api_units_daily,
               a.filter_domain,
               AVG(a.shows_approx),
               AVG(a.clicks_approx),
               ca.is_autobanned
          from api_user_domain_stat a
     left join users_api_options uao on a.uid=uao.uid
     left join users u on u.uid=a.uid
     left join clients_autoban ca on u.ClientID = ca.ClientID
         WHERE a.stat_date > DATE_SUB(?, INTERVAL ? DAY)
         AND u.uid > 0
                 -- and u.uid = 117676200
      GROUP BY uid, filter_domain
        ";

    foreach my $shard (ppc_shards()) {
        $sth = exec_sql(PPC(shard => $shard), $sql, $yesterday, $Settings::API_UNITS_CALC_PERIOD);

        while(my ($uid, $api_units_daily, $filter_domain, $shows_approx, $clicks_approx, $bids_autobanner) = $sth->fetchrow_array()){
            next if ! $filter_domain;
            $clicks_approx ||= 1;
            $users{$uid}->{domains}->{$filter_domain} = {shows => $shows_approx, clicks => $clicks_approx};
            $users{$uid}->{clicks_sum} += $clicks_approx;

            # если пользователь создал слишком много фраз и сработал автобан-фильтр (скрипт ppcAutobanUsers.pl) -- снижаем ему рейтинг в 10 раз
            $users{$uid}->{units_factor} = $bids_autobanner && $bids_autobanner eq 'Yes' ? 0.1 : 1;

            ###
            $users{$uid}->{api_units_daily} = $api_units_daily;
            $users{$uid}->{bids_autobanner} = $bids_autobanner;
            $users{$uid}->{shard} = $shard if $xls_out_flag;
        }
        $sth->finish();
    }

    my $old_uids_with_units = {map {$_ => 1} @{ get_one_column_sql(PPC(shard=>'all'), 'select distinct(uid) from api_users_units_consumption')}};

    $log->out("total units_from users: ".  scalar keys %$old_uids_with_units );

    # считаем рейтинг пользователя
    my @user_ratings;
    my @user_logs;
    my @user_logs_keys = qw/uid scheme rating units/;
    my $insert_sql = 'insert into api_users_units_consumption (uid, scheme, daily_units)
                        VALUES %s
                        ON DUPLICATE KEY UPDATE
                        daily_units=VALUES(daily_units)';

    my $xls_user_rating = {};

    ################################################################################################################################################

    # собираем информацию о наличии у пользователя агентских или менеджерских кампаний для учета в карме
    my %HAS_USER_SERV_CAMPS = ();
    my @uids = keys %users;
    while (my @chunk = splice @uids, 0, 5000) {
        my $res = ServicedClient::users_has_agency_manager_camps(\@chunk) || [];
        hash_merge \%HAS_USER_SERV_CAMPS, $res;
    }

    while( my ($uid, $user) = each %users ) {

        $xls_user_rating->{$uid} = {login => '', rating => 0, domains => [], api_units_daily => $user->{api_units_daily}};
        $user->{rating} = 0;
        $xls_user_rating->{$uid}->{shard} = $user->{shard} if $xls_out_flag;

        while(my ($domain, $user_domain) = each %{$user->{domains}}) {
            if (defined $domains{$domain}){

                if($user->{clicks_sum}){

                    $user->{rating} += $user->{clicks_sum} ? $domains{$domain}->{rating} * $user_domain->{clicks} / $user->{clicks_sum} : 0;

                    $user->{bad_reasons_sum} += $domains{$domain}->{bad_reasons};

                    if ($xls_out_flag){
                        push (@{$xls_user_rating->{$uid}->{domains}},
                            {
                                shows_approx => int($domains{$domain}->{shows_approx}),
                                clicks_approx => int($domains{$domain}->{clicks_approx}),
                                sum_approx => $domains{$domain}->{sum_approx} / 1_000_000,
                                good_ctr_phrases => int($domains{$domain}->{good_ctr_phrases}),
                                bad_ctr_phrases => int($domains{$domain}->{bad_ctr_phrases}),
                                accepted_items => int($domains{$domain}->{accepted_items}),
                                declined_items => int($domains{$domain}->{declined_items}),
                                bad_reasons => $domains{$domain}->{bad_reasons},
                                domain_rating => $domains{$domain}->{rating},
                                clicks => int($user_domain->{clicks}),
                                clicks_sum => int($user->{clicks_sum}),
                                domain => $domain,
                                ctr_rating => $domains{$domain}->{ctr_rating},
                                bad_ctr_rating => $domains{$domain}->{bad_ctr_rating},
                                bad_banner_rating => $domains{$domain}->{bad_banner_rating},
                                click_cost_rating => $domains{$domain}->{click_cost_rating},
                            });
                    } else {
                        # для логов
                        push (@{$xls_user_rating->{$uid}->{domains}},
                            {
                                domain_rating => $domains{$domain}->{rating},
                                domain => $domain,
                                domain_part => ($user->{clicks_sum} ? $domains{$domain}->{rating} * $user_domain->{clicks} / $user->{clicks_sum} : 0),
                                ctr_rating => $domains{$domain}->{ctr_rating},
                                bad_ctr_rating => $domains{$domain}->{bad_ctr_rating},
                                bad_banner_rating => $domains{$domain}->{bad_banner_rating},
                                click_cost_rating => $domains{$domain}->{click_cost_rating},
                            });
                    }
                }
            }
        }

        # учитываем bad_reasons в рейтинге
        if ($user->{bad_reasons_sum}) {
            $user->{rating} *= $badReasonsValue;
        }

        #$user->{old_rating} = $user->{rating};
        #$user->{old_units} = calcUserUnits($user->{old_rating});

        $user->{units_factor} = 1 unless defined $user->{units_factor};
        $user->{units} = ($user->{api_units_daily} && $user->{api_units_daily} > 0 ? $user->{api_units_daily} : calcUserUnits($user->{rating})) * $user->{units_factor};

        if ($user->{api_units_daily} || $user->{rating} > 0) {
            if ($xls_out_flag){
                $xls_user_rating->{$uid}->{rating} = $user->{rating};
                $xls_user_rating->{$uid}->{login} = get_login(uid => $uid); # если только нужно
                $xls_user_rating->{$uid}->{units_factor} = $user->{units_factor};
                $xls_user_rating->{$uid}->{units} = $user->{units};
                $xls_user_rating->{$uid}->{bad_reasons_sum} = $user->{bad_reasons_sum};
                $xls_user_rating->{$uid}->{autobanned} = $user->{bids_autobanner};
            } else { # для логирования рейтинга
                $xls_user_rating->{$uid}->{rating} = $user->{rating};
            }

            for my $scheme (@ALLOWED_API_UNITS_SCHEMES) {
                my $units = $user->{units};
                push(@user_ratings, [$uid, $scheme, $units]);
                push(@user_logs, [$uid, $scheme, $user->{rating}, $units ]);

                delete $old_uids_with_units->{$uid};
            }
        }

        if(@user_ratings > 1000){
            foreach_shard uid => \@user_ratings, by => sub {$_->[0]}, sub {
                my ($shard, $chunk) = @_;
                do_mass_insert_sql(PPC(shard => $shard), $insert_sql, $chunk);
            };
            @user_ratings = ();
        }

        if(@user_logs > 1000){
            LogTools::log_user_units([ map { +{mesh @user_logs_keys, @$_} } @user_logs ]);
            @user_logs = ();
        }

        #warn Dumper {user => $user};
    }

    foreach_shard uid => \@user_ratings, by => sub {$_->[0]}, sub {
        my ($shard, $chunk) = @_;
        do_mass_insert_sql(PPC(shard => $shard), $insert_sql, $chunk);
    };
    LogTools::log_user_units([ map { +{mesh @user_logs_keys, @$_} } @user_logs ]) if(@user_logs);

    $log->out("removing units_from users: ".  scalar keys %$old_uids_with_units );
    # $log->out("removing units_from users: list: ". join(', ', keys %$old_uids_with_units) );
    # Удаляем баллы у пользователей, которым не добавили баллы
    do_sql(PPC(uid => [keys %$old_uids_with_units]), [
       'delete from api_users_units_consumption',
       where => { uid => SHARD_IDS }
    ]) if keys %$old_uids_with_units;

    my @user_ratings_logs;
    my @user_ratings_logs_keys = qw/uid
                                    rating
                                    domain
                                    domain_part
                                    domain_rating
                                    ctr_rating
                                    bad_ctr_rating
                                    bad_banner_rating
                                    click_cost_rating/;
    while( my ($uid, $user_rating) = each %$xls_user_rating ) {
        foreach my $d (@{$user_rating->{domains}}){
            push (@user_ratings_logs, [$uid, $user_rating->{rating}, (map {$d->{$_}} qw(domain domain_part domain_rating ctr_rating bad_ctr_rating bad_banner_rating click_cost_rating))]);
        }

        if(@user_ratings_logs > 1000){
            LogTools::log_user_ratings([ map { +{mesh @user_ratings_logs_keys, @$_} } @user_ratings_logs ]);
            @user_ratings_logs = ();
        }
    }

    LogTools::log_user_ratings([ map { +{mesh @user_ratings_logs_keys, @$_} } @user_ratings_logs ]) if (@user_ratings_logs);

    if ($xls_out_flag){

        my @spec_fields = qw/shard rating api_units_daily units bad_reasons_sum autobanned/;
        my @xls_fields = qw(domain domain_rating ctr_rating bad_ctr_rating bad_banner_rating click_cost_rating shows_approx clicks_approx sum_approx good_ctr_phrases bad_ctr_phrases accepted_items declined_items clicks clicks_sum bad_reasons);

        open my $fh, ">", "user_rating.csv";
        print $fh join(";", ('login', @spec_fields, @xls_fields))."\n";

        while( my ($uid, $user_rating) = each %$xls_user_rating ) {
            if (scalar @{$user_rating->{domains}}) {
                foreach my $d (@{$user_rating->{domains}}) {
                    $user_rating->{rating} =~ s/\./\,/g if $user_rating->{rating};

                    print $fh ($user_rating->{login} || $uid || "") . ";";
                    print $fh join (";", map { ($user_rating->{$_} // '') } @spec_fields).";";
                    print $fh join (";", map {$d->{$_} =~ s/\./\,/g if ($d->{$_} && $_ ne 'domain'); $d->{$_} || ""} @xls_fields)."\n";
                }
            } else {
                $user_rating->{rating} =~ s/\./\,/g if $user_rating->{rating};

                print $fh ($user_rating->{login} || $uid || "") . ";";
                print $fh join (";", map { ($user_rating->{$_} // '') } @spec_fields).";"."\n";
            }
        }
        close($fh);
    }

}

#############################################################################################################################################

=head2 функция getBsDomainStat

 получает данные по домена из Крутилки (показы клики открутки) и записываем эти данные в таблицу ppcdict.api_domain_stat

=cut

sub getBsDomainStat
{
    my $profile = Yandex::Trace::new_profile('getApiDomainStat:getBsDomainStat');

    $log->out("start getBsDomainStat");
    my $ua = new LWP::UserAgent(timeout => 5*60);

    my $url = $Settings::BS_EXPORT_PROXY_READONLY.'export/domainstat.cgi?from='.$yesterday;
    my $resp = $ua->get($url);
    my $content;
    if ($resp->is_success) {
        $content = $resp->content;
    } else {
        die "Can't get data from BS ( $url ): " . $resp->status_line;
    }

    my $base_sql = "INSERT INTO %s (filter_domain, shows_approx, clicks_approx, sum_approx, stat_date)
                             VALUES %%s
                             ON DUPLICATE KEY UPDATE
                             shows_approx=VALUES(shows_approx),
                             clicks_approx=VALUES(clicks_approx),
                             sum_approx=VALUES(sum_approx)";

    my $insert_sql = sprintf($base_sql, "api_domain_stat");

    my ($cnt_records, $fmt, @names) = (0);
    my @values;
    foreach my $line ( split("\n", $content) ) {
        if ($line =~ /^\#(.*)$/) {
            ($fmt, @names) = split(/\t/, $1);
        } elsif ($fmt && $line =~ /^\+$fmt\t/) {
            $cnt_records++;

             my ($t, @vals) = split(/\t/, $line);
             my %hash = mesh @names, @vals;
             push(@values, [(map {$hash{$_}} qw/Domain Shows Clicks Cost/), $yesterday]);
             if(@values > 1000){
                do_mass_insert_sql(PPCDICT, $insert_sql, \@values);

                $log->out("write chunk of data to api_domain_stat table");
                @values = ();
             }
        } elsif(! $fmt ) {
            warn "unknown line format: $line";
        }
    }

    do_mass_insert_sql(PPCDICT, $insert_sql, \@values) if @values;

    $log->out("getBsDomainStat: processed $cnt_records rows");
    $log->out("finish getBsDomainStat");

}

#############################################################################################################################################

=head2 функция getApiUserDomainStat

 получает статистику для пользователей по доменам: показы и клики

=cut

sub getApiUserDomainStat
{
    my $profile = Yandex::Trace::new_profile('getApiDomainStat:getApiUserDomainStat');
    $log->out("start getApiUserDomainStat");

    # двойное условие для fdomain в запросе связано с тем, что  у пользователя может быть как
    # несколько доменов, так и несколько визиток у баннеров с отсутствующими доменами, таким образом
    # группировку данных нам нужно производить по одному из этих полей. Поэтому в случае отсутствия домена,
    # заменяем его на телефон из визитки, группируем данные, а потом преобразуем телефон в псевдодомен.
    # кажется, что это самое короткое решение, хоть и несколько дурацкое.
    my $sql = "
            SELECT c.uid as uid,
                   ifnull( ifnull(fd.filter_domain, b.domain), v.phone ) as fdomain,
                   sum(t.shows) as shows,
                   sum(t.clicks) as clicks,
                   v.phone
              FROM ( SELECT pid,
                            sum(shows+pshows) shows,
                            sum(clicks+pclicks) clicks
                       FROM bs_auction_stat auct
                      WHERE auct.stattime > ?
                      GROUP BY pid
                   ) t
                   JOIN phrases p using(pid)
                   JOIN campaigns c using(cid)
                   JOIN banners b ON b.pid = p.pid
                   LEFT JOIN filter_domain fd using(domain)
                   LEFT JOIN vcards v using(vcard_id)
             GROUP BY c.uid, fdomain
        ";


    my $base_sql = "INSERT INTO %s (uid, filter_domain, shows_approx, clicks_approx, stat_date)
                             VALUES %%s
                             ON DUPLICATE KEY UPDATE
                             shows_approx=VALUES(shows_approx), clicks_approx=VALUES(clicks_approx)";

    my $insert_sql = sprintf($base_sql, "api_user_domain_stat");

    my $cnt_records = 0;

    foreach my $shard (ppc_shards()) {
        my $sth = exec_sql(PPC(shard => $shard), $sql, $yesterday);
        my @values;

        while(my $row = $sth->fetchrow_hashref()){
            # если нет домена, но есть телефон, преобразуем последний в псевдодомен
            if( $row->{phone} && $row->{fdomain} eq $row->{phone} ) {
                $row->{fdomain} = phone_domain($row->{fdomain});
            }

            if ($row->{fdomain}) {
                push(@values, [ @{$row}{ qw/ uid fdomain shows clicks / }, $yesterday ]);
                $cnt_records++;
            }

            if (@values > 1000) {
                foreach_shard uid => \@values, by => sub {$_[0]->[0]}, sub {
                    my ($shard, $chunk) = @_;
                    do_mass_insert_sql(PPC(shard => $shard), $insert_sql, $chunk);
                };

                $log->out("write chunk of data to api_user_domain_stat table");
                @values = ();
            }
        }

        if (@values) {
            foreach_shard uid => \@values, by => sub {$_[0]->[0]}, sub {
                my ($shard, $chunk) = @_;
                do_mass_insert_sql(PPC(shard => $shard), $insert_sql, $chunk);
            };
        }
    }
    $log->out("getApiUserDomainStat processed $cnt_records rows ");

    # чистим таблицу от старых данных
    my $res1 = do_sql(PPC(shard => 'all'), "DELETE FROM api_user_domain_stat WHERE stat_date <= DATE_SUB(now(), INTERVAL $Settings::API_UNITS_CALC_PERIOD DAY)");

    $log->out("getApiUserDomainStat deleted from api_user_domain_stat $res1 rows");
    $log->out("finish getApiUserDomainStat");

}

#############################################################################################################################################

=head2 функция getApiDomainStat

 получает статистику по доменам: кол-во фраз отключенных за плохой CTR и фразы с хорошим CTR

=cut

sub getApiDomainStat
{
    my $profile = Yandex::Trace::new_profile('getApiDomainStat:getApiDomainStat');
    $log->out("start getApiDomainStat");

    # см. комментарий к похожему запросу из getApiUserDomainStat
    my $sql = "SELECT ifnull( ifnull(fd.filter_domain, b.domain), v.phone ) as fdomain,
                      v.phone,
                      sum(good_ctr) as good_ctr_count,
                      sum(bad_ctr) as bad_ctr_count
                FROM (
                     SELECT pid,
                            sum(if(auct.rank>0, 1, 0)) as good_ctr,
                            sum(if(auct.rank=0, 1, 0)) as bad_ctr
                       FROM bs_auction_stat auct
                      WHERE auct.stattime > ?
                      GROUP BY 1
                     ) t
                     JOIN phrases p USING(pid)
                     JOIN banners b ON b.pid = p.pid
                     LEFT JOIN filter_domain fd using(domain)
                     LEFT JOIN vcards v on b.vcard_id = v.vcard_id
               GROUP by fdomain
        ";

    my $base_sql = "INSERT INTO %s (filter_domain, good_ctr_phrases, bad_ctr_phrases, stat_date)
                            VALUES %%s
                            ON DUPLICATE KEY UPDATE bad_ctr_phrases=VALUES(bad_ctr_phrases),
                            good_ctr_phrases=VALUES(good_ctr_phrases)";

    my $insert_sql = sprintf($base_sql, "api_domain_stat");

    my $cnt_records = 0;

    foreach my $shard (ppc_shards()) {
        my $sth = exec_sql(PPC(shard => $shard), $sql, $yesterday);
        my @values;
        while(my $row = $sth->fetchrow_hashref()){
            # если нет домена, но есть телефон, преобразуем последний в псевдодомен
            if( $row->{phone} && $row->{fdomain} eq $row->{phone} ) {
                $row->{fdomain} = phone_domain($row->{fdomain});
            }

            if ($row->{fdomain}) {
                push(@values, [ @{$row}{ qw/ fdomain good_ctr_count bad_ctr_count / }, $yesterday] );
                $cnt_records++;
            }

            if(@values > 1000){

                do_mass_insert_sql(PPCDICT, $insert_sql, \@values);

                $log->out("write chunk of data to api_domain_stat table");
                @values = ();
            }
        }

        do_mass_insert_sql(PPCDICT, $insert_sql, \@values) if @values;
    }

    $log->out("getApiDomainStat processed $cnt_records rows");
    $log->out("finish getApiDomainStat");
}

#############################################################################################################################################

=head2 _clear_api_domain_stat

    Удаляет часть данных из PPCDICT.api_domain_stat.
    В таблице остаются только данные за последние $API_STAT_MAX_AGE_IN_TABLE дней.

=cut

sub _clear_api_domain_stat
{
    my $profile = Yandex::Trace::new_profile('getApiDomainStat:_clear_api_domain_stat');
    get_dbh(PPCDICT)->{use_mysql_result} = 1;

    my $cnt = 0;
    $log->out("Deleting everything older than $API_STAT_MAX_AGE_IN_TABLE days from api_domain_stat");
    while (1) {
        $log->out("Starting iteration with chunk size of $API_STAT_SELECT_CHUNK_SIZE items");
        my $iter_guard = relaxed_guard();
        my $chunk = get_all_sql(PPCDICT, "SELECT filter_domain, stat_date
                                            FROM api_domain_stat
                                           WHERE stat_date <= DATE_SUB(NOW(), INTERVAL $API_STAT_MAX_AGE_IN_TABLE DAY)
                                           LIMIT $API_STAT_SELECT_CHUNK_SIZE");
        my $real_chunk_length = scalar @$chunk;
        last if !$real_chunk_length;

        my %day2domain;
        for my $item (@$chunk) {
            push @{ $day2domain{$item->{stat_date}} }, $item->{filter_domain};
        }
        for my $day (keys %day2domain) {
            my $domains_num = scalar @{ $day2domain{$day} };
            $log->out("Deleting $domains_num domains from $day");

            for my $domains_chunk (chunks($day2domain{$day}, $API_STAT_DELETE_CHUNK_SIZE)) {
                my $delete_guard = relaxed_guard();
                $cnt += do_sql(PPCDICT, ["DELETE FROM api_domain_stat", WHERE => {stat_date => $day, filter_domain => $domains_chunk}]);
            }
        }

        last if $real_chunk_length < $API_STAT_SELECT_CHUNK_SIZE;
    };
    $log->out("Deleting finished: deleted $cnt rows");

    return;
}
