#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */59 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 100
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host: checks_auto.direct.yandex.ru
    ttl:  3h
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

=cut
=head1 NAME

    getBannerStorageDicts.pl
    getBannerStorageDicts.pl --translate

=head1 DESCRIPTION

    Обновление тем, макетов и шаблонов из BannerStorage в БД ppcdict

=cut

use my_inc '..';

use ScriptHelper 'Yandex::Log' => 'messages';
use Direct::Modern;
use BannerStorage;
use Settings;
use JSON;
use Yandex::DBTools;

use List::MoreUtils qw(uniq);
use Path::Tiny;

# workaround for: Can't call method "_put_session" on an undefined value at /usr/lib/perl5/AnyEvent/Handle.pm
# it's a race condition in the object destroying order
use Carp::Always;


my $TRANSLATE = 0;
extract_script_params(
    'translate' => \$TRANSLATE,
);

my $dicts = {};

$log->out('START');

$log->out('quering <dictionaries/themes>');
my $bs_themes = BannerStorage::banner_storage_call(GET => 'dictionaries/themes', {})->{content}->{items};
$dicts->{theme} = [ map {_theme_item($_)} @$bs_themes ];
$log->out(sprintf 'got %d themes', scalar @$bs_themes);

$log->out('quering <dictionaries/layouts>');
my $bs_layouts = BannerStorage::banner_storage_call(GET => 'dictionaries/layouts', {})->{content}->{items};
$dicts->{layout} = [ map {_layout_item($_)} @$bs_layouts ];
$log->out(sprintf 'got %d layouts', scalar @$bs_layouts);

if (!$TRANSLATE) {
    $log->out('quering <template>');
    my $bs_templates = BannerStorage::banner_storage_call(GET => 'templates', { include => 'businessType,layoutCodes' })->{content}->{items};
    $dicts->{template} = [ map {_template_item($_)} @$bs_templates ];
    $log->out(sprintf 'got %d templates', scalar @$bs_templates);
}


# Выгрузка фраз для перевода в файл
if ($TRANSLATE) {
    my $path = "$Settings::ROOT/protected/data/translation/bannerstorage_layouts_and_themes.trans";
    $log->out("saving names for translation to $path");

    my @phrases = uniq sort map {$_->{name}} (@{$dicts->{theme}}, @{$dicts->{layout}});
    my $phrases_json = JSON->new->pretty->encode(\@phrases);
    path($path)->spew_utf8($phrases_json);
}

# Выгрузка в базу данных (ppcdict)
else {
    foreach my $dict_name (sort keys %$dicts) {
        $log->out("updating <$dict_name> in banner_storage_dict");

        my $old_dict = get_hash_sql(PPCDICT, [
                "SELECT id, json_content FROM banner_storage_dict",
                WHERE => { type => $dict_name },
            ]);

        my $dict_items = $dicts->{$dict_name};
        my @data_to_update;
        for my $item (sort {$a->{value} <=> $b->{value}} @{$dicts->{$dict_name}}) {
            my $id = $item->{value};
            my $old_value = $old_dict->{$id} || '';
            my $new_value = to_json $item, {canonical => 1};
            next if $new_value eq $old_value;

            $log->out("set $dict_name:$id to $new_value");
            push @data_to_update, [$id, $dict_name, $new_value];
        }

        if (!@data_to_update) {
            $log->out('nothing to update');
            next;
        }

        $log->out('saving to db');
        do_mass_insert_sql(PPCDICT,
            'insert into banner_storage_dict (id, type, json_content) values %s
            ON DUPLICATE KEY UPDATE json_content=VALUES(json_content)',
            \@data_to_update,
        );
    }
}

$log->out('FINISH');

juggler_ok();


sub _theme_item {
    my $bs_item = shift;
    return {
        name => $bs_item->{name},
        value => $bs_item->{id},
    };
}

sub _layout_item {
    my $bs_item = shift;
    return {
        name => $bs_item->{name},
        value => $bs_item->{id},
        img_src => $bs_item->{iconUrl},
    };
}

sub _template_item {
    my $bs_item = shift;

    my $template_id = $bs_item->{id};
    my %layouts = map { $_->{id} => $_ } @{$bs_item->{layoutCodes} // []};

    # проверяем шаблоны (только новые и только смарт) на наличие layouts
    if (
        $bs_item->{typeId} == 9
        && !$BannerStorage::TEMPLATE_WITHOUT_LAYOUTS{$template_id}
        && !%layouts
    ) {
        $log->warn("ERROR: missed layouts for template $template_id; skipping");
        return ();
    }

    return {
        name => $bs_item->{name},
        templateType => $bs_item->{typeId},
        templateTypeName => $bs_item->{typeName},
        value => $template_id,
        type => $bs_item->{businessType}->{directId},
        layouts => \%layouts,
    };
}

