#!/usr/bin/perl

use my_inc "..";

# workaround for: Can't call method "_put_session" on an undefined value at /usr/lib/perl5/AnyEvent/Handle.pm
# it's a race condition in the object destroying order
use Carp::Always;

# $Id$

=head1 METADATA

# ВАЖНО, при измении списка target нужно поправить juggler-проверку в файле getNews.yml
<crontab>
    time: */10 * * * *
    package: frontend
    params: --target json --target rss
</crontab>
<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 150
        </leases>
    </switchman>
    package: scripts-switchman
    params: --target db
</crontab>
<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    params: --target db
</crontab>
# проверяем только работоспособность скрипта, но не то, что все новости правильно обработаны.
# для скриптовых машин
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.getNews.working.to_db
    raw_events:     scripts.getNews.working.db
    ttl:            40m
    tag: direct_group_internal_systems
</juggler>
# для фронтэндов
<juggler>
    from_file:  getNews.yml
</juggler>

=cut

=head1 DESCRIPTION

    Скрипт для получения новостей из API блогов.
    Полученные данные сохраняются в таблицу adv_news_items  также из них генерится json и RSS-фид.
    Новости забираем из API блогов для каждого доступного языка в $Settings::NEWS_FEED.
    Каждый полученный язык кладется в регоны в которых он доступен из $Settings::NEWS_LANG_REGIONS.
    По сути для нас регион - это домен второго уровня (.ru, .com, .com.tr).

    В ответе API блогов нас интересуют следующие поля:
        - publishDate - дата публикации новости в iso8601, переводим в YYYY-MM-dd и сохраняем в поле adv_news_items.news_date. Также это поле пишется в json в поле data в локализованном формате (для RU - "d MMMM yyyy г", EN - "MMMM d, yyyy", TR - "d MMMM yyyy")
        - approvedTitle - заголовок новости, пишется в json в поле data
        - slug - уникальный id новости. Представляет собой заголовок новости записанный транслитом через дефисы (Н-р: novaia-ficha-direkta). Используется для формирования поля ext_news_id и ссылки на новость.
    В adv_news_items.ext_news_id пишем в формате d-<YYYY-MM-dd от publishDate>-<slug> - делаем так для совместимости с Инфоблоком - он получает дату новости из id.
    В adv_news_items.data пишем json формата:
        {
            "date": "<publishDate (формат для RU - 'd MMMM yyyy г', EN - 'MMMM d, yyyy', TR - 'd MMMM yyyy')>",
            "content": [
                {
                    "text": "<approvedTitle>",
                    "link_url": "https://yandex.<tld>/adv/news/<slug>"
                }
            ]
        }

    Параметры:
      --target   куда сохраняем (json, rss, db)

    ./getNews.pl --target rss --target db

=cut

use Direct::Modern;

use Path::Tiny;
use Encode qw/encode decode is_utf8/;
use JSON;
use XML::RSS;

use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;

use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::HTTP qw/http_fetch/;
use Yandex::I18n;
use Yandex::DateTime;
use Yandex::TVM2;


our %OUTPUT_SUB = (
    json => \&_output_json_file,
    rss => \&_output_rss_file,
    db => \&_output_db,
);

# количество новостей, которые забираем из api блогов
my $NEWS_COUNT = 99;

extract_script_params(
    'action|target=s@' => \my @actions,
);

usage() if !@actions;

$log->out("START");
my ($juggler_status, $juggler_description) = ('OK', 'OK');

my $ticket = eval{Yandex::TVM2::get_ticket($Settings::ADV_NEWS_TVM2_ID)} or $log->die("Cannot get ticket for $Settings::ADV_NEWS_TVM2_ID: $@");

while ( my ($lang, $info) = each %Settings::NEWS_FEED ) {
    my $url = "$Settings::ADV_NEWS_API_URL"."$info->{api_slug}";
    my $locale = $info->{locale};

    my $response = http_fetch('GET', $url, { lang => $locale, tag => $info->{tag}, size => $NEWS_COUNT }, log => $log, headers => {'X-Ya-Service-Ticket' => $ticket});
    my $content = eval { decode_json($response) };
    if (!defined $content || $@) {
        my $msg = "Can't get content from $url";
        $log->out($msg, $@);
        $juggler_status = 'WARN';
        $juggler_description = $msg;
        next;
    }

    my @news_items = _parse_news($content, $info, $locale);
    $log->out(sprintf "Got %d items", scalar @news_items);

    for my $action ( @actions ) {
        $log->out("saving to $action");
        for my $region ( @{$Settings::NEWS_LANG_REGIONS{$lang}->{region}} ) {
            # формируем ссылки на новости
            for my $news_item (@news_items) {
                my $url = $news_item->{url} = _get_news_url($lang, $region, $news_item->{slug});
                my $text = $news_item->{title};
                $news_item->{content} = [
                    {link_url => $url, text => $text},
                ];
                $news_item->{text} = qq{<a href="$url">$text</a>};
            }
            $OUTPUT_SUB{$action}->( $region, $lang, \@news_items );
        }
    }
}

for my $action (@actions) {
    juggler_event(service_suffix => $action, status => $juggler_status, description => $juggler_description);
}
$log->out("FINISH");


# для новой морды [и инфоблока]: записываем в базу
sub _output_db {
    my ($region, $lang, $items) = @_;

    $log->out("writing region $region");
    for my $item ( @$items ) {
        my $data = to_json hash_cut $item, qw/ date content /;
        my $row = {
            region => $region,
            lang => $lang,
            news_date => $item->{news_date},
            ext_news_id => $item->{id},
            data => $data,
        };
        if (get_one_field_sql(PPCDICT, ["SELECT 1 FROM adv_news_items", WHERE => $row])) {
            # точно такая же запись уже есть
            next;
        } else {
            do_insert_into_table(PPCDICT, 'adv_news_items', $row,
                                 on_duplicate_key_update => 1,
                                 key => 'news_id',
                );
        }
    }

    return;
}


# для старой морды: пишем в json-файлы
sub _output_json_file {
    my ($region, $lang, $items) = @_;
    my $dir = $Settings::ROOT;

    my $news_path = "$dir/data/t/news_${region}_$lang.json";
    $log->out("writing $news_path");
    write_json_file($news_path, [grep {$_} map {$items->[$_]} ( 0 .. $Settings::NUM_OF_NEWS_ON_FRONT-1 )] );

    return;
}


# rss-фиды
sub _output_rss_file {
    my ($region, $lang, $items) = @_;
    my $dir = $Settings::ROOT;

    my $guard = Yandex::I18n::init_i18n_guard($lang, check_file => 1);
    my $rss_path = "$dir/data/news_${region}_$lang.rss";
    $log->out("writing $rss_path");
    my $rss = XML::RSS->new( version => '2.0');
    my $tld = _get_lang_region_tld($lang, $region);
    $rss->channel(
        title => iget("Новости Яндекс.Директа"),
        link => "http://direct.yandex.$tld/",
    );

    for my $item ( @$items ) {
        my $title = join q{}, map {$_->{text}} @{$item->{content}};
        $rss->add_item(
            title => "$item->{date}. $title",
            permaLink => $item->{url},
            description => "$item->{date}. $item->{text}",
        );
    }

    $rss->save($rss_path);
    return;
}



sub _parse_news {
    my ($content, $info, $locale) = @_;

    return
        grep {$_}
        map {_parse_news_item($_, $info, $locale)} @$content;
}


sub _parse_news_item {
    my ($content, $info, $locale) = @_;

    my $dt = iso8601_2_datetime($content->{publishDate});
    $dt->set_locale($locale);
    my $date = $dt->format_cldr($info->{date_format});
    my $news_date = $dt->format_cldr('YYYY-MM-dd');

    my $text = $content->{approvedTitle};
    my $slug = $content->{slug};

    my $item = {
        date => $date,
        id => "d-$news_date-$slug",
        news_date => $news_date,
        title => $text,
        slug => $slug,
    };

    return $item;
}


sub _get_news_url
{
    my ($lang, $region, $slug) = @_;

    my $tld = _get_lang_region_tld($lang, $region);
    my $url = "https://yandex.$tld/adv/news/$slug";
    return $url;
}

sub _get_lang_region_tld {
    my ($lang, $region) = @_;

    my $tld = $Settings::NEWS_FEED{$lang}->{tld};
    # для Казахстана и Белорусии показываем новости на русском, но подменяем домены в ссылках на национальные
    if ($lang eq 'ru' && $region eq 'kz') {
        $tld = 'kz';
    }
    if ($lang eq 'ru' && $region eq 'by') {
        $tld = 'by';
    }

    return $tld;
}


sub write_json_file
{
    my ($path, $items) = @_;

    my @news = map { hash_cut $_, qw/ date content / } @$items;
    path($path)->spew_utf8(to_json \@news);
}


1;

