#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

# getOptimizeStat.pl: avg: 124 MB, max: 213 MB
<crontab>
    time: */59 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 180
        </leases>
    </switchman>
    package: scripts-switchman
    sharded: 1
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:          4h
    sharded:      1
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Скрипт для сбора статистики по кампаниям до оптимизации и после
    Заполняет таблицу optimizing_campaign_stat
    multicurrency: работает в фишках, при конвертировании копированием данные о прошлой кампании не учитываются

    Параметры командной строки:
        --help
            выводит справку по использованию скрипта
        --cid номер_кампании
            работать только с указанной кампанией
        --mediaplanner <логин>
            работать только с заявками на оптимизацию указанного медиапланера

=head1 RUNNING

    LOG_TEE=1 ./protected/getOptimizeStat.pl --shard-id 2
    LOG_TEE=1 ./protected/getOptimizeStat.pl --cid 263 --shard-id 2
    LOG_TEE=1 ./protected/getOptimizeStat.pl --mediaplanner medialogin --shard-id 2

=cut

use Direct::Modern;

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Settings;
use Yandex::DBTools;

use Stat::CustomizedArray;
use Currency::Rate;
use Yandex::SendMail qw/send_alert/;
use PrimitivesIds;
use List::Util qw/sum minstr maxstr/;


$log->out("START");

my (@cids, $mediaplanner_login);
extract_script_params (
    'cid=i@' => \@cids,
    'mediaplanner=s' => \$mediaplanner_login,
    'skip-mediaplans' => \my $skip_mp,
);

my @additional_conds;
if (@cids) {
    $log->out('Working only on cids:', \@cids);
    @additional_conds = ('AND', {'c.cid' => \@cids});
}

my $mediaplanner_uid;
if ($mediaplanner_login) {
    $mediaplanner_uid = get_uid(login => $mediaplanner_login);
    $log->out("Working only on requests processed by mediaplanner $mediaplanner_login (uid=$mediaplanner_uid):");
    @additional_conds = ('AND', {'o.MediaUID' => $mediaplanner_uid});
}

$log->out('Start processing accepted mediaplans');

my %interval = (
    before_1_week => '1 WEEK',
    before_2_week => '2 WEEK',
    before_4_week => '4 WEEK',
    after_1_week => '1 WEEK',
    after_2_week => '2 WEEK',
    after_4_week => '4 WEEK',
);

my $mediaplans = !$skip_mp && get_all_sql(PPC(shard => $SHARD), [
        'SELECT ms.mpid, c.cid, c.OrderID, u.ClientID,',
        (map {(
                "mas_$_.mpid IS NULL and accept_time < now() - interval 1 day as need_$_, ",
                "date(accept_time - interval $interval{$_}) as start_$_, ",
                "date(accept_time - interval 1 day) as finish_$_, ",
            )} grep {/^before/} keys %interval),
        (map {(
                "mas_$_.mpid IS NULL and accept_time < now() - interval 1 day - interval $interval{$_} as need_$_, ",
                "date(accept_time + interval 1 day) as start_$_, ",
                "date(accept_time + interval $interval{$_}) as finish_$_, ",
            )} grep {/^after/} keys %interval),
        '  IFNULL(c.currency, "YND_FIXED") AS currency
        FROM mediaplan_stats ms
        JOIN campaigns c USING(cid)
        JOIN users u USING(uid)',
        (map {"LEFT JOIN mediaplan_accept_stats mas_$_ ON mas_$_.mpid=ms.mpid AND mas_$_.stat_type='$_' "} keys %interval),
        WHERE => {
            (@cids ? (cid => \@cids) : ()),
            ($mediaplanner_uid ? (MediaUID => $mediaplanner_uid) : ()),
            OrderID__ne => 0,
            accepted => 'Yes',
            accept_time__gt__dont_quote => 'now() - interval 2 month',
        },
        HAVING => {
            _OR => +{ map {("need_$_" => 1)} keys %interval },
        },
    ]);

$log->out(sprintf 'Going to process %d mediaplans', scalar @{$mediaplans || []});
for my $mp (@{$mediaplans || []}) {
    my $mpid = $mp->{mpid};

    my $stat_start = minstr map {$mp->{"start_$_"}} grep {$mp->{"need_$_"}} keys %interval;
    next if !$stat_start;
    my $stat_finish = maxstr map {$mp->{"finish_$_"}} grep {$mp->{"need_$_"}} keys %interval;

    $log->out("Processing mpid=$mpid (cid=$mp->{cid})");

    my $dbstat = new Stat::CustomizedArray(OrderID => $mp->{OrderID}, translocal_params => {tree => 'api'}); # регион не используется, пофиг какое дерево

    my $success = eval {
        for my $stat_type (sort keys %interval) {
            next if !$mp->{"need_$stat_type"};

            my @params = ($dbstat, $mp->{OrderID}, $mp->{currency});
            my $stat = _calc_stat_values(@params, $mp->{"start_$stat_type"}, $mp->{"finish_$stat_type"});

            do_replace_into_table(PPC(shard => $SHARD), mediaplan_accept_stats => {
                    mpid => $mpid,
                    stat_type => $stat_type,
                    map {($_ => $stat->{$_} || 0)} qw/days shows clicks sum/,
                },
            );
        }
        'success';
    };

    if (!$success) {
        my $err = $@;
        $log->warn("Error: $err");
    }
}



$log->out('Start processing optimization requests');

my $opt_requests = get_all_sql(PPC(shard => $SHARD), ["select 
                                            o.request_id, 
                                            c.OrderID, 
                                            IFNULL(c.currency, 'YND_FIXED') AS currency,
                                            date(ifnull(o.accept_time, o.create_time)) as create_time, 
                                            date(DATE_ADD(ifnull(o.accept_time, o.create_time), INTERVAL 1 DAY)) as next_time, 
                                            date(DATE_SUB(ifnull(o.accept_time, o.create_time), INTERVAL 1 WEEK)) as before_time, 
                                            date(DATE_ADD(ifnull(o.accept_time, o.create_time), INTERVAL 8 DAY)) as after_time 
                                    from 
                                        optimizing_campaign_requests o 
                                        join campaigns c 
                                        on o.cid=c.cid 
                                        left join optimizing_campaign_stat s on o.request_id=s.request_id 
                                        where 
                                            status='Accepted' and  
                                            s.request_id IS NULL and  
                                            c.OrderID > 0 and
                                            ifnull(o.accept_time, o.create_time) < DATE_SUB(now(), INTERVAL 8 DAY)
                                        ", @additional_conds]);

foreach my $request (@$opt_requests){
    my $request_id = $request->{request_id};
    my $old_msg_prefix = $log->msg_prefix;
    $log->msg_prefix("[Request $request_id]");

    $log->out('Start processing request:', $request);
    my $dbstat = new Stat::CustomizedArray(OrderID => $request->{OrderID}, translocal_params => {tree => 'api'}); # регион не используется, пофиг какое дерево

    my $success = eval {
        my @params = ($dbstat, $request->{OrderID}, $request->{currency});
        my $stat_before = _calc_stat_values(@params, $request->{before_time}, $request->{create_time});
        my $stat_after  = _calc_stat_values(@params, $request->{next_time}, $request->{after_time});

        $log->out('Inserting collected values into DB');
        my $profile = Yandex::Trace::new_profile('getOptimizeStat:insert');
        my $ctr_phrase_before = sprintf("%0.2f", $stat_before->{ctr} || 0);
        my $ctr_phrase_after  = sprintf("%0.2f", $stat_after->{ctr} || 0);        

        my $sum_before = sprintf("%0.2f", $stat_before->{sum} || 0);
        my $sum_after  = sprintf("%0.2f", $stat_after->{sum} || 0);

        my $ctr_date_before = sprintf("%0.2f", $stat_before->{ctr_by_date} || 0); 
        my $ctr_date_after = sprintf("%0.2f", $stat_after->{ctr_by_date} || 0);        

        do_insert_into_table(PPC(shard => $SHARD), 'optimizing_campaign_stat', {
            request_id => $request_id,
            ctr_phrase_before => $ctr_phrase_before,
            ctr_phrase_after => $ctr_phrase_after,
            sum_before => $sum_before,
            sum_after => $sum_after,
            ctr_date_before => $ctr_date_before,
            ctr_date_after => $ctr_date_after,
        }, on_duplicate_key_update => 1, key => 'request_id');

        undef $profile;
        return 1;
    };

    if (!$success) {
        my $err = $@;
        $log->warn("Error: $err");
        send_alert("Error processing request $request_id: $err", 'getOptimizeStat error');
    }

    $log->msg_prefix($old_msg_prefix);
}

unless (@additional_conds) {
    juggler_ok();
}

$log->out("FINISH");




sub _calc_stat_values {
    my ($dbstat, $OrderID, $currency, $start_time, $finish_time) = @_;

    $log->out("Fetching stat $start_time .. $finish_time");
    my $profile = Yandex::Trace::new_profile('getOptimizeStat:stat_fetch');
    state $opt = {
        with_discount => 1,
        with_nds => 1,
    };

    my $by_date = $dbstat->get_stat_customized($OrderID, $start_time, $finish_time, ['date'], 'year', undef, %$opt);
    my $by_phrase = $dbstat->get_stat_customized($OrderID, $start_time, $finish_time, ['phrase'], 'year', undef, %$opt);
    undef $profile;

    $profile = Yandex::Trace::new_profile('getOptimizeStat:avg_calc');
    my %stat = (
        days => $by_date->{days_num},
        ctr_by_date => $by_date->{tctr} || 0,
        period => [@{$by_phrase->{period}}],
    );

    # считаем суммарные значения за все даты
    my $count = @{$by_phrase->{data} || []};
    for my $key (qw(shows clicks sum shows_0 clicks_0 sum_0 shows_1 clicks_1 sum_1)) {
        $stat{$key} = sum(map {$_->{$key} || 0} @{$by_phrase->{data}}) || 0;
    }
    for my $key (qw(ctr ctr_0 ctr_1)) {
        $stat{$key} = $count && sum(map {$_->{$key} || 0} @{$by_phrase->{data}}) / $count;
    }
    $stat{sum} = convert_currency($stat{sum}, $currency, 'YND_FIXED', with_nds => 1);
    undef $profile;

    return \%stat;
}


