#!/usr/bin/perl

=head1 NAME

    api_generate_persistent_token.pl - генерация постоянного токена для доступа в API и хеша для записи в API::Settings

    # создать токен и подпись
    api_generate_persistent_token.pl 

    # вычислить подпись существующего токена
    api_generate_persistent_token.pl token

=cut
    

use strict;
use warnings;
use utf8;

use Carp qw/croak/;
use List::MoreUtils qw/uniq/;
use Path::Tiny qw/path/;

use my_inc '../..';

use Settings;
use API::Settings;
use HashingTools qw/sha256_hex hmac_sha256_hex/;

my $token;
if (@ARGV) {
    $token = $ARGV[0];
} else {
    read path("/dev/random")->openr_raw, my $buf, 32;
    $token = join '', map {sprintf "%02x", ord($_)} split //, $buf;
}

if (length($token) < 32) {
    croak "token is too short: '$token'";
} elsif ($token =~ /[^a-z0-9]/i) {
    croak "token contains incorrect symbols: '$token'";
} elsif (scalar(uniq $token =~ /(..)/g) < 10) {
    croak "token is too simple: '$token'";    
}

my $hmac = hmac_sha256_hex($token, $API::Settings::API_PERSISTENT_TOKENS_KEY);

print "TOKEN:
$token

DATA FOR API::Settings:
\$API_PERSISTENT_TOKENS->{'$hmac'} = {
    login => '',
    application_id => '',
    allow_to => [
        'networks:internal.txt',
    ],
};
";
