#!/usr/bin/perl

use my_inc "../..";


# $Id$

=head1 NAME

    beta_httpd_conf.pl - инициализация разработческого сервера

=head1 SYNOPSIS

    # перегенерировать конфиг и перезапустить апач
    # старный конфиг сохраняется в директорию apache/conf/old
    perl protected/maintenance/beta_httpd_conf.pl

    perl protected/maintenance/beta_httpd_conf.pl 8087 /var/www/beta.zhur.8087 --user=zhur --conf=devtest --init

    # инициализация сервера "с нуля"
    svn co https://svn.yandex.ru/direct/trunk/ ./beta.USER.PORT
    cd ./beta.USER.PORT
    perl protected/maintenance/beta_httpd_conf.pl --init

    Опции:
        --no-soap - отключить виртуальный сервер для soap-апи
        --no-intapi - отключить виртуальный сервер для intapi
        --api5-coverage - проксирует запросы JSON/SOAP API5 на адерса 127.0.0.(10,11) нужно для сбора покрытия, сами psgi-приложения запускаются через starman
        --sandbox - добавить в конфиг виртуальный сервер для фэйковых сервисов песочницы
        --yandex-lib-local, -y
            использовать модули Yandex:: из каталога yandex-lib в рабочей копии ($PERL5LIB в init.sh)
        --conf=prod|test|test2|devtest|sbdevtest|sbdev7|sb|sbtest|tl - настроить на указанную конфигурацию

=head1 APACHE/NGINX

    Для каждой беты запускается по одному инстансу apache и nginx, которые будут слушать порт, соответствующий бете, на следующих адресах:

    127.0.0.1 - apache/API
    127.0.0.2 - apache/web
    127.0.0.15 - apache/intapi

    127.0.0.3 - nginx/API/http
    127.0.0.4 - nginx/API/https
    127.0.0.5 - nginx/web/http
    127.0.0.6 - nginx/web/https
    127.0.0.7 - nginx/intapi/http
    127.0.0.8 - nginx/intapi/https

    127.0.0.30 - nginx/canvas/https
    Tакже IP 127.0.0.3{1,2,3,4,5} заняты сервисами canvas'а поднимаемыми в докере (см direct-utils/beta-ctl/bin/beta-ctl функция enable_canvas)
    127.0.0.35 - rtbhost stub для работы canvas'а на бете (если вдруг нужен)
    127.0.0.37 - freact, фронт к html5-конструтору
    127.0.0.38 - canvas frontend, фронт к ГО и видео конструкторам
    127.0.0.40 - java canvas backend

    А также есть nginx из пакетов, который слушает на
    127.0.0.{7,8}:{7080,7443} - общий nginx для беты

=cut

use strict;
use warnings;

use Cwd;
use File::Path;
use File::Find;
use File::Slurp;
use File::Basename qw/dirname basename/;
use Fatal qw/rename unlink/;
use Template;
use Time::HiRes;
use Getopt::Long;
use FindBin qw/$Bin/;
use JSON;
use Yandex::Hostname;
use Socket;

use utf8;

my $ROBOT_USER = "ppc";

my $dev_soap = 1;
my $dev_intapi = 1;
my $api5_coverage = 0;
my $dev_sandbox = 0;
my $init;
my $user = '';
my $conf = 'devtest';
my $max_clients = Yandex::Hostname::hostfqdn() =~ /ppcdev[56]/ ? 10 : 0;
my $yandex_lib_local = 0;
GetOptions(
    "soap!" => \$dev_soap,
    "intapi!" => \$dev_intapi,
    "sandbox" => \$dev_sandbox,
    "api5-coverage" => \$api5_coverage,
    "init" => \$init,
    "user=s" => \$user,
    "conf=s" => \$conf,
    "y|yandex-lib-local" => \$yandex_lib_local,
    "max-clients=i" => \$max_clients,
    "h|help" => sub {
        system("pod2text-utf8 $0");
        exit(0);
    },
    );

my ($port, $root) = @ARGV;

$user ||= [getpwuid($<)]->[0];

# Пытаемся достать данные из текущего пути
if (!defined $root) {
    $root = getcwd();
    $root =~ s!(/www/[^/]+(?:(?:\/arcadia)?/direct/perl)?).*!$1!;
}
if ($root =~ /\/[a-z0-9_-]+\.([a-z0-9_\-]+)\.(\d+)(?:(?:\/arcadia)?\/direct\/perl)?\/*$/i) {
    $user ||= $1;
    $port ||= $2;
}

# удаляем из рута конечные слеши
$root =~ s/\/+$//g;

# $root -- корень для всего perl-ового. $arc_root -- корень всей беты. Для svn совпадают, для arc -- нет
(my $arc_root = $root) =~ s!/arcadia/direct/perl!!;

if (!defined $port || $port !~ /^\d+$/ || !defined $root || $root !~ /^\// || !-d $root) {
    warn "root=$root";
    warn "port=$port";
    system("pod2text-utf8 $0");
    exit(1);
}

my $beta_ctl = '/usr/local/bin/beta-ctl';
my $beta_settings = {};
if (-x $beta_ctl) {
    my $beta_settings_json = `'$beta_ctl' --dir $arc_root get-settings`;
    $beta_settings = eval { decode_json($beta_settings_json) };
    if (!defined $beta_settings && $@) {
        warn $@;
        $beta_settings = {};
    }
}

my $APACHE_DIR = "$root/apache";
my $NGINX_DIR  = "$root/nginx";
my $JS_TEMPLATER_DIR  = "$arc_root/js-templater";

my %DB_CONFIGURATIONS = (
    prod => '',
    test => 'Test',
    test2 => 'Test2',
    devtest => 'DevTest',
    sbdevtest => 'SandboxDevTest',
    sbdev7 => 'SandboxDev7',
    sb => 'Sandbox',
    sbtest => 'SandboxTest',
    tl => 'TestLoad',
);


my %JAVA_DEV_PORTS = (
    logviewer => 80,
    api5 => 14080,
    intapi => 80,
    web_api => 83,
    canvas_api => 84,
    test_steps => 85,
);

if (!exists $DB_CONFIGURATIONS{$conf}) {
    die "Неверно указана конфигурация: $conf";
}

# если --init - создаём директории apache и nginx
if ($init) {
    # создаём директории apache/..., init.sh
    mkpath("$APACHE_DIR/logs", 0, 0777);
    mkpath("$APACHE_DIR/conf", 0, 0777);
    mkpath("$APACHE_DIR/run", 0, 0777);

    mkpath("$NGINX_DIR/log", 0, 0777);
    mkpath("$NGINX_DIR/conf/sites-available", 0, 0777);
    mkpath("$NGINX_DIR/run/body", 0, 0777);
    mkpath("$NGINX_DIR/run/proxy", 0, 0777);

    mkpath("$JS_TEMPLATER_DIR", 0, 0777);

    mkpath("$root/protected/run", 0, 0777);

    # set up SettingsLocal
    if (-e "$root/perl/settings/SettingsLocal.pm") {
        print "Skip SettingsLocal\n";
    } else {
        if ($conf eq 'prod') {
        } elsif (exists $DB_CONFIGURATIONS{$conf}) {
            my $conf_file_name_part = $DB_CONFIGURATIONS{$conf};
            my $conf_file_name = "$root/perl/settings/Settings${conf_file_name_part}.pm";
            die "Файл $conf_file_name не существует" unless -f $conf_file_name;
            symlink($conf_file_name, "$root/perl/settings/SettingsLocal.pm")
        }
    }
}

my $ad_designer_host_without_tld;
my $is_roprod = 0;
{
    no warnings 'once';
    require Settings;
    if ($Settings::CONFIGURATION =~ /sandbox/) {
        $dev_sandbox = 1;
    }

    if ($Settings::CONFIGURATION =~ /roprod/) {
        $is_roprod = 1;
    }

    # перекладывание значений в эти переменные нужно для того, чтобы избежать ворнинга 'Name "Settings::AD_DESIGNER_STAND_ID" used only once'
    $ad_designer_host_without_tld = $Settings::AD_DESIGNER_HOST_WITHOUT_TLD;
}


my $server_name = 'ppc.yandex.ru';

(my $db_conf = ($beta_settings->{db_config_file} || `perl -Iperl/settings -MSettings -e 'print \$Yandex::DBTools::CONFIG_FILE'`)) =~ s/.*\/db-config\.(.*)\.json$/$1/;
my $nonexistent_backend = '127.0.0.1:9';    # конфигурации, для которых нет стабильных стендов, отправляем на "несуществующий" бекэнд -- порт 9 зарезервирован для discard protocol
my %java_endpoints;
if ($db_conf eq 'test' || $db_conf eq 'devtest' || $db_conf eq 'dev7' || $db_conf eq 'test2' || $db_conf =~ /\/db-config\.json$/) {
    my $db_conf_part = $db_conf;
    if ($db_conf =~ /\/db-config\.json$/) { #roprod, эта конфигурация не работает в java, заворачиваем на девтест
        $db_conf_part = 'devtest';
    }
    # хосты java-ТС теперь в zookeeper, и java-instance-np.json для них не используется. Ориентируемся по cname, куда отправлять запросы
    %java_endpoints = (
        logviewer => 'direct-logviewer-devtest.in.yandex.net',
        api5 => 'java-api5.'.$db_conf_part.'.direct.yandex.ru:'.$JAVA_DEV_PORTS{api5},
        intapi => 'java-intapi.'.$db_conf_part.'.direct.yandex.ru:'.$JAVA_DEV_PORTS{intapi},
        web_api => 'java-web.'.$db_conf_part.'.direct.yandex.ru:'.$JAVA_DEV_PORTS{web_api},
        canvas_api => 'canvas.'.$db_conf_part.'.direct.yandex.ru:'.$JAVA_DEV_PORTS{canvas_api},
        test_steps => 'steps.'.$db_conf_part.'.direct.yandex.ru:'.$JAVA_DEV_PORTS{test_steps}
    );
} elsif ($db_conf eq 'sandboxtest' || $db_conf eq 'sandboxdevtest' || $db_conf eq 'sandboxdev7' || $db_conf eq 'sandboxtest2') {
    %java_endpoints = map {$_ => $nonexistent_backend} keys %JAVA_DEV_PORTS;
} else {
    die "не найдена конфигурация java балансеров для $db_conf"
}

my $t = Template->new(
    {
        ABSOLUTE => 1,
        PRE_DEFINE => {
            dev => 1,
            dev_soap => $dev_soap,
            dev_intapi => $dev_intapi,
            port => $port,
            root => $root,
            max_clients => $max_clients,
            dev_sandbox => $dev_sandbox,
            api5_coverage => $api5_coverage,
            AD_DESIGNER_HOST_WITHOUT_TLD => $ad_designer_host_without_tld,
            java_endpoints => \%java_endpoints,
            java_apps => {map {$_ => 1} @{$beta_settings->{java_apps} || []}},
            beta_settings => $beta_settings,
        },
    }
);

if (-f "$APACHE_DIR/run/ppc.yandex.ru.pid") {
    my ($apache_pid) = read_file("$APACHE_DIR/run/ppc.yandex.ru.pid", chomp => 1);
    print "$APACHE_DIR/init.sh stop\n";
    system "$APACHE_DIR/init.sh stop";
    # Ждём не более 10 секунд выхода апача
    for(1..100) {
        Time::HiRes::sleep 0.3;
        if (!-d "/proc/$apache_pid") {
            last;
        } else {
            my $cmd = read_file("/proc/$apache_pid/cmdline", chomp => 1);
            if (index($cmd, $APACHE_DIR) == -1) { # скорее всего, другой процесс, не наш апач
                last;
            }
        }
    }
}

my $res_apache_conf = "$APACHE_DIR/conf/httpd.conf";
# определяем текущий ip -- чтобы беты можно было запускать на любой машине без исправления ip-адресов
# (и все равно есть еще несколько мест, где захардкожен ip ppcdev'а)
my $current_ip = get_current_ip();
$t->process("$Bin/../../etc/frontend/apache2/$server_name.conf", { ppcdev => $current_ip }, "$res_apache_conf.new") || die $t->error();
rotate_file($res_apache_conf, "$res_apache_conf.new");

# иногда нужно проверить работу soap-апача на бете, в нормально режиме использоваться не будет
# подробнее https://wiki.yandex-team.ru/direct/api/soap_apache_on_betas
my $res_soap_apache_conf = "$APACHE_DIR/conf/soap.direct.yandex.ru.conf";
$t->process("$Bin/../../etc/soap/apache2/soap.direct.yandex.ru.conf", { ppcdev => $current_ip }, "$res_soap_apache_conf.new") || die $t->error();
rotate_file($res_soap_apache_conf, "$res_soap_apache_conf.new");

# Если надо использовать локальные модули Yandex:: -- собираем $PERL5LIB
my $perl5lib = '';
if ($yandex_lib_local){
    die "can't find yandex-lib directory" unless -d "$root/yandex-lib";
    my @yandex_lib_dirs = grep {-d} glob "$root/yandex-lib/*/lib";
    $perl5lib = join ":", @yandex_lib_dirs;
}

my $res_apache_init = "$APACHE_DIR/init.sh";
$t->process("$Bin/../../etc/frontend/init.d/$server_name", { perl5lib => $perl5lib }, "$res_apache_init.new") || die $t->error();
rotate_file($res_apache_init, "$res_apache_init.new");
chmod(0755, $res_apache_init);

# иногда нужно проверить работу soap-апача на бете, в нормально режиме использоваться не будет
# подробнее https://wiki.yandex-team.ru/direct/api/soap_apache_on_betas
my $res_soap_apache_init = "$APACHE_DIR/soap.direct.yandex.ru";
$t->process("$Bin/../../etc/soap/init.d/soap.direct.yandex.ru", { perl5lib => $perl5lib }, "$res_soap_apache_init.new") || die $t->error();
rotate_file($res_soap_apache_init, "$res_soap_apache_init.new");
chmod(0755, $res_soap_apache_init);

my $nginx_pid_file = "$NGINX_DIR/run/direct-accel.pid";
if (-f $nginx_pid_file) {
    my ($pid) = read_file($nginx_pid_file, chomp => 1);
    print "nginx stop\n";
    system "$NGINX_DIR/init.sh stop";
    for (1 .. 100) {
        Time::HiRes::sleep 0.3;
        if (!-d "/proc/$pid") {
            last;
        } else {
            my ($cmd) = read_file("/proc/$pid/cmdline", chomp => 1);
            if (index($cmd, $NGINX_DIR) == -1) {
                last;
            }
        }
    }
}

my $res_nginx_conf = "$NGINX_DIR/conf/direct-accel.conf";
$t->process("$Bin/../../etc/frontend/nginx/direct-accel.conf", {}, "$res_nginx_conf.new") or die $t->error();
rotate_file($res_nginx_conf, "$res_nginx_conf.new");

my $res_nginx_vhost_conf = "$NGINX_DIR/conf/sites-available/direct-vhost.conf";
$t->process("$Bin/../../etc/frontend/nginx/direct-vhost.conf", {}, "$res_nginx_vhost_conf.new") or die $t->error();
rotate_file($res_nginx_vhost_conf, "$res_nginx_vhost_conf.new");

my $res_nginx_init = "$NGINX_DIR/init.sh";
$t->process("$Bin/../../etc/frontend/init.d/direct-accel", {}, "$res_nginx_init.new") or die $t->error();
rotate_file($res_nginx_init, "$res_nginx_init.new");
chmod(0755, $res_nginx_init);

my $res_nginx_soap_vhosts = "$NGINX_DIR/conf/soap-direct-vhost.conf";
$t->process("$Bin/../../etc/soap/nginx/soap-direct-vhost.conf", {}, "$res_nginx_soap_vhosts.new") or die $t->error();
rotate_file($res_nginx_soap_vhosts, "$res_nginx_soap_vhosts.new");

my $res_nginx_intapi_vhosts = "$NGINX_DIR/conf/intapi-direct-vhost.conf";
$t->process("$Bin/../../etc/intapi/nginx/intapi-direct-vhost.conf", {}, "$res_nginx_intapi_vhosts.new") or die $t->error();
rotate_file($res_nginx_intapi_vhosts, "$res_nginx_intapi_vhosts.new");

my $res_js_templater_conf = "$JS_TEMPLATER_DIR/conf.sh";
$t->process("$Bin/../../etc/frontend/js-templater/conf.sh", {}, "$res_js_templater_conf.new") or die $t->error();
rotate_file($res_js_templater_conf, "$res_js_templater_conf.new");

my $res_js_templater_init = "$JS_TEMPLATER_DIR/init.sh";
$t->process("$Bin/../../etc/frontend/init.d/js-templater", {}, "$res_js_templater_init.new") or die $t->error();
rotate_file($res_js_templater_init, "$res_js_templater_init.new");
chmod(0755, $res_js_templater_init);

if ($init) {
    chdir $root;

    system("$root/protected/getNews.pl --target rss --target json --target db");
    system("$root/protected/prebuild/convert_rtf_docs.pl");
    system("$root/protected/prebuild/update_networks_file.pl", "$root/data/t/prebuild/networks", "$root/protected/data/networks");
    system("/usr/local/bin/compile_mo.pl -locale_path $root/locale/ -project_id direct");
    system("/usr/local/bin/update_emails.pl --base-dir $root");
    system("$root/protected/maintenance/make_english_pdf_appendix_for_reports.pl");
    system("find $root -maxdepth 1 -mindepth 1 -not -name dna -not -name dna_arc -not -name dna_arc_store | xargs -n 1 -I {} chown -R $user:ppc {}");

    if (-e "$root/dna") {
        system("chown $user:ppc $root/dna");
    }

    if ($user ne $ROBOT_USER) {
        system("find $root -maxdepth 1 -mindepth 1 -not -name dna -not -name dna_arc -not -name dna_arc_store | xargs -n 1 -I {} chmod -R g+rw {}");
        system("find $root -type d -not -name dna -not -name dna_arc -not -name dna_arc_store -print0 | xargs -0 chmod g+s");

        if (-e "$root/dna") {
            system("chmod g+rw $root/dna");
            system("chmod g+s $root/dna");
        }
    } else {
        chmod 0755, $root;  # direct-create-beta выставляет 0775
    }
}

my $sudo = $> ? '' : "sudo -u $user";
print "$sudo $APACHE_DIR/init.sh start\n";
system "$sudo $APACHE_DIR/init.sh start";

print "$sudo $NGINX_DIR/init.sh start\n";
system "$sudo $NGINX_DIR/init.sh start";


# если новый файл отличается от старого - ротируем старый в директорию old и переименовываем новый файл
sub rotate_file {
    my ($file, $new_file) = @_;
    if (-f $file && scalar(read_file($file)) eq scalar(read_file($new_file))) {
        unlink $new_file;
        return;
    }
    my $old_dir = dirname($file)."/old";
    my $fname = basename($file);
    mkdir $old_dir, 0775;
    for my $i (8..0) {
        if (-f "$old_dir/$fname.$i") {
            rename("$old_dir/$fname.$i", "$old_dir/$fname.".($i+1));
        }
    }
    rename($file, "$old_dir/$fname.0") if -f $file;
    rename($new_file, $file);
}

sub get_current_ip
{
     my ($err, @res) = Socket::getaddrinfo(Yandex::Hostname::hostfqdn(), 0, {'protocol' => Socket::IPPROTO_TCP, 'family' => Socket::AF_INET6});
     die "Cannot getaddrinfo - $err" if $err;
     while( my $ai = shift @res ) {
         my ($err, $ipaddr) = Socket::getnameinfo($ai->{addr}, Socket::NI_NUMERICHOST, Socket::NIx_NOSERV);
         die "Cannot getnameinfo - $err" if $err;
         return $ipaddr;
     }
}

