#!/usr/bin/perl

=head1 NAME

    check-mysql-int-types.pl

=head1 DESCRIPTION

    Проверка на то,  что все числовые поля в базе достаточно далеки от 
    своего максимального значения.
    Близкие к максимуму столбцы нужно поальтерить, иначе переполнение и беда-беда-беда.
    (столбцы с хэшами обычно альтерить не надо)

    В качестве агрументов скрипты передаются имена баз, скрипт
    - получает список таблиц,
    - для каждой таблицы получает список целочисленных столбцов
    - делает запросы вида
        SELECT max(int_field1), max(int_field2), ... FROM table
    - сравниваются полученные значения и максимально допустимое значение типа, результат выводится в таблицу

    Запросы в худшем случае делают full-scan, поэтому лучше запускать на наименьшем репрезентативном шарде

    Для ускорения можно передать параметр --sample-size N - просматривать только N последний строк в каждой таблицу
    (ориентируясь по первичному ключу)

=cut

use Direct::Modern;

use Math::BigInt;

use Yandex::DBTools;

use my_inc '../..';

use ScriptHelper;

$|++;

my $SAMPLE_SIZE = 0;
extract_script_params(
    'sample-size=i' => \$SAMPLE_SIZE,
    );

my @DBS = @ARGV;
for my $db (@DBS) {
    check_db($db);
}


=head1 check_db($db)

    проверить базу

=cut
sub check_db {
    my ($db) = @_;
    my @tables = @{get_one_column_sql($db, "show tables")};
    for my $table (@tables) {
        check_table($db, $table);
    }
}


=head2 check_table($db, $tbl)
    
    проверить таблицу

=cut
sub check_table {
    my ($db, $table) = @_;
    my @cols = 
        @{get_all_sql($db, "
                    SELECT column_name, column_type
                      FROM information_schema.columns
                     WHERE table_name = ?
                       AND table_schema = database()
                    ", $table)};
    $_->{type_info} = parse_column_type($_->{column_type}) for @cols;
    my @int_cols = grep {$_->{type_info}{is_int}} @cols;

    if (!@int_cols) {
        return {int_cols => []};
    }

    my $from = $table;
    if ($SAMPLE_SIZE) {
        my $pk_prefix = get_one_field_sql($db, "
                                SELECT column_name
                                  FROM information_schema.key_column_usage
                                 WHERE table_name = ?
                                   AND table_schema = database()
                                   AND constraint_name = 'PRIMARY'
                                   AND ordinal_position = 1
                                ", $table);
        if ($pk_prefix) {
            $from = "(SELECT * FROM $table ORDER BY `$pk_prefix` DESC LIMIT $SAMPLE_SIZE) as t";
        }
    }
    my $SQL = "SELECT ".join(", ", map {"max(`$_->{column_name}`) as `$_->{column_name}`"} @int_cols).
        " FROM $from";
    my $max_vals = get_one_line_sql($db, $SQL);

    for my $int_col (@int_cols) {
        my $val = $max_vals->{$int_col->{column_name}};
        my $max_val = $int_col->{type_info}{int_max_value};
        printf "%s\t%0.2f%%\t(%s of %s)\n", "$db.$table.$int_col->{column_name}", 100*($val//0)/$max_val, $val//'null', $max_val;
    }
}


=head2 parse_column_type($type)

    попрарсить mysql-ный тип столбца, вернуть хэш со знаниями о типе

=cut
sub parse_column_type {
    my ($type) = @_;
    my $ret = {
        type => $type,
    };
    if ($type =~ /^(tiny|small||big)int/i) {
        $ret->{is_int} = 1;
        $ret->{int_modifier} = $1 // '';
        $ret->{is_int_unsigned} = $type =~ /unsigned/i ? 1 : 0;
        $ret->{size} = {tiny => 1, small => 2, '' => 4, big => 8}->{$ret->{int_modifier}};
        $ret->{int_max_value} = Math::BigInt->new(2)->bpow($ret->{size} * 8 - ($ret->{is_int_unsigned} ? 0 : 1))->bsub(1)->numify;
    }
    return $ret;
}
