#!/usr/bin/perl

use my_inc '../..';

=head1 NAME

    currency_convert_clients_report.pl

=head1 DESCRIPTION

    Генерирует отчет по клиентам  (с разделением на типы – агентский, менеджеский, самостоятельный),
    которых можем принудительно сконвертировать в валюту для https://st.yandex-team.ru/DIRECT-59341
    Работает с текущей конфигурацией, т.е. чтобы считать по проду надо переключать бету на продакшен-конфигурацию

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use Yandex::DBTools;
use Yandex::ListUtils qw(chunks);
use Yandex::HashUtils;

use Settings;
use ScriptHelper;
use ShardingTools qw(ppc_shards);
use User;
use RBACElementary;
use RBACDirect;
use Client;

my $OPERATOR_UID = 1;
# для скольки клиентов получаем главных представителей за один запрос
my $CHUNK_SIZE = 5_000;

my $clid2chief_uid = {};
my $clid2manager_uid = {};
my $chief_uid2login = {};
my $client_clientid2agency_clientid = {};

my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);

$log->out("START");

foreach my $shard (ppc_shards()) {
    my $log_prefix_guard = $log->msg_prefix_guard("[shard $shard]");
    my $clients_can_convert = get_one_column_sql(PPC(shard => $shard), "
            SELECT fcc.ClientID
            FROM force_currency_convert fcc
            INNER JOIN users u ON u.ClientID = fcc.ClientID
            LEFT JOIN clients cl ON fcc.ClientID = cl.ClientID
            INNER JOIN clients_to_force_multicurrency_teaser t ON fcc.ClientID = t.ClientID
            INNER JOIN client_firm_country_currency cfcc ON cfcc.ClientID = fcc.ClientID
            LEFT JOIN currency_convert_queue q ON fcc.ClientID = q.ClientID
            INNER JOIN campaigns c ON u.uid = c.uid
            WHERE
                q.ClientID IS NULL
                AND IFNULL(cl.work_currency, 'YND_FIXED') = 'YND_FIXED'
            GROUP BY fcc.ClientID
            HAVING
                    COUNT(DISTINCT cfcc.country_region_id) = 1
                AND COUNT(DISTINCT cfcc.currency) = 1
        ");
    $log->out('Got '. scalar @$clients_can_convert .' clients');

    for my $clids_chunk (chunks($clients_can_convert, $CHUNK_SIZE)) {
        # разбиваем на чанки, чтобы не падать по памяти при большом количестве клиентов
        hash_merge($clid2chief_uid, RBACElementary::rbac_get_chief_reps_of_clients($clids_chunk));
    }
    hash_merge($clid2manager_uid, rbac_get_managers_of_clients_by_clientids($rbac, $clients_can_convert));
    hash_merge($client_clientid2agency_clientid,
        get_hash_sql(PPC(shard => $shard), ['
                     SELECT client_client_id, MAX(agency_client_id)
                     FROM agency_client_relations
                 ',  WHERE => {
                         client_client_id => $clients_can_convert,
                         bind => 'Yes',
                 }, 'GROUP BY client_client_id']));
    my $users_data = User::get_users_data([values %$clid2chief_uid], [qw/login/]);
    hash_merge($chief_uid2login, { map { $_ => $users_data->{$_}->{login} } keys %$users_data});
}
$log->out('Generate report');

print("ClientID\tlogin\tclient_type\n");
foreach my $client_id (keys %$clid2chief_uid) {
    my $client_type;
    if ($client_clientid2agency_clientid->{$client_id}) {
        $client_type = 'агентский';
    } elsif ($clid2manager_uid->{$client_id}) {
        $client_type = 'менеджеский';
    } else {
        $client_type = 'самостоятельный';
    }
    print("$client_id\t$chief_uid2login->{$clid2chief_uid->{$client_id}}\t$client_type\n");
}

$log->out("FINISH")
