#!/usr/bin/env perl

# $Id$

use my_inc "../..";

=pod

=encoding utf-8

=head1 NAME

debsums_monitor - отслеживание целостности установленных в системе deb-пакетов

=head1 DESCRIPTION

    Запускает debsums-check.pl, dpkg и dpkg-query

    Опции:
    --help
        вывести справку
    --conf=s
        обязательный параметр; файл с конфигурацией, может содержать ключи exclude-packages, exclude-files
        должен содержать include-packages и include-environments

    debsums_monitor.pl --conf=/var/www/ppc.yandex.ru/etc/debsums-monitor/debsums-monitor.ts.conf

    запишет в трекер список изменённых или отсутствующих файлов и диффы изменений

=head1 METADATA

<crontab>
    time: */15 * * * *
    package:
    params: --conf=/var/www/ppc.yandex.ru/etc/debsums-monitor/debsums-monitor.pr.conf
    env: PROJECT_SPECIFIC=Direct
</crontab>
<crontab>
    time: */10 * * * *
    package: conf-sandbox
    params: --conf=/var/www/ppc.yandex.ru/etc/debsums-monitor/debsums-monitor.pr_sand.conf
    env: PROJECT_SPECIFIC=Direct
</crontab>
<juggler>
    from_file: debsums_monitor.yml
</juggler>

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use Digest::SHA qw(sha384_hex);
use File::Spec;
use Getopt::Long;
use Path::Tiny;
use ScriptHelper get_file_lock => ['dont die'];
use Startrek::Client::Easy;
use YAML;
use Yandex::Retry qw/retry/;
use Yandex::Hostname;
use ProjectSpecific qw/get_sign_comment/;
use File::Basename;

my $SIGN_COMMENT = get_sign_comment(basename(__FILE__));

$Startrek::Client::Easy::USE_DEFAULT_ASSIGNEE = 0;

my $CONFIG;
GetOptions(
    "help" => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0;},
    "config=s" => \$CONFIG,
) or $log->die("can't parse options");

$log->die("no --config option given") unless $CONFIG;
$log->die("configuration file not found") unless -r $CONFIG;
my $CONF = YAML::LoadFile($CONFIG);
$log->die("no include-packages entry in the configuration") unless $CONF->{'include-packages'};
$log->die("bad include-packages entry") unless ref $CONF->{'include-packages'} && ref $CONF->{'include-packages'} eq 'ARRAY' && @{$CONF->{'include-packages'}};
$log->die("no include-environments entry in the configuration") unless $CONF->{'include-environments'};
$log->die("bad include-environments entry") unless ref $CONF->{'include-environments'} && ref $CONF->{'include-environments'} eq 'ARRAY' && @{$CONF->{'include-environments'}};
$log->die("no startrek-queue in config") unless exists $CONF->{'startrek-queue'} && $CONF->{'startrek-queue'};
my $ST_queue = $CONF->{'startrek-queue'};
my %include_environments = map {$_ => undef} @{$CONF->{'include-environments'}};

my $fqdn = Yandex::Hostname::hostfqdn();
$log->die("Empty FQDN") unless $fqdn;

my $environment = path('/etc/yandex/environment.type')->slurp;
chomp $environment;
unless (exists $include_environments{$environment}) {
    exit(0);
}

if ($CONF->{'exclude-fqdn-expressions'}) {
    unless (ref $CONF->{'exclude-fqdn-expressions'}
        && ref $CONF->{'exclude-fqdn-expressions'} eq 'ARRAY'
        && @{$CONF->{'exclude-fqdn-expressions'}}) {
        $log->die("bad exclude-fqdn-expressions entry");
    }
    foreach my $expression (@{$CONF->{'exclude-fqdn-expressions'}}) {
        exit(0) if ($fqdn =~ $expression);
    }
}

my $startrek = Startrek::Client::Easy->new(token => Startrek::Client::Easy::get_startrek_client_token(file => '/etc/direct-tokens/startrek'));
#my $startrek = Startrek::Client::Easy->new();

my (%previous_diffs, %current_diffs, $old_key);
my $issue = $startrek->get(
    query => qq/Queue: $ST_queue AND Summary: "$fqdn package inconsistency" "Sort By": Key DESC/,
    perPage => 1,
    array => 1
); # здесь заодно проверяется связь с трекером
if (@$issue && $issue->[0]->{description}) {
    %previous_diffs = map {$_ => undef} grep {s/^Diff hash: //} split /[\r\n]+/, $issue->[0]->{description};
    $old_key = $issue->[0]->{key};
}

my @packages_installed;
retry(
    tries => 3,
    pauses => [20],
    sub {
        for my $regex (@{$CONF->{'include-packages'}}) {
            next unless $regex;
            push @packages_installed, split /[\r\n]+/, `dpkg -l | grep '^ii' | awk '{print \$2}' | grep -P $regex`;
        }
        $log->die("couldn't get installed packages list") unless @packages_installed && (grep {$_} @packages_installed == @packages_installed);
    }
);

my %exclude_packages;
if ($CONF->{'exclude-packages'} && ref $CONF->{'exclude-packages'} && ref $CONF->{'exclude-packages'} eq 'ARRAY' && @{$CONF->{'exclude-packages'}}) {
    %exclude_packages = map {$_ => undef} @{$CONF->{'exclude-packages'}};
}

my %exclude_files;
if ($CONF->{'exclude-files'} && ref $CONF->{'exclude-files'} && ref $CONF->{'exclude-files'} eq 'ARRAY' && @{$CONF->{'exclude-files'}}) {
    %exclude_files = map {$_ => undef} @{$CONF->{'exclude-files'}};
}

my $tmpdir_path = File::Spec->tmpdir();

for my $package (@packages_installed) {
    next if exists $exclude_packages{$package};
    if ($package !~ /^[a-zA-Z0-9\.\-]+$/) {
        $log->die("unexpected package name: $package\n");
    }

    my $changed = `/usr/local/bin/debsums-check.pl --package=$package --conf=$CONFIG 2>&1`;
    if ($? == -1) {
        $log->die("failed to execute debsums-check.pl: $!\n");
    }

    if ($changed) {
        my $package_version = `dpkg-query -f='\${Version}' -W $package`;
        $package_version ||= 'unknown';
        my @lines = split /[\r\n]+/, $changed;
        my $diff; # здесь могут быть не только диффы, но и строчки с прочими обнаруженными проблемами
        foreach my $line (@lines) {
            if ($line =~ /^---|^\+\+\+/) {
                $line =~ s/\s+\d{4}-\d{2}-\d{2}.+$//;
                if ($line =~ /^---/) {
                    $line =~ s/(\s+)\Q$tmpdir_path\E\/[^\/]+\//$1TEMPDIR\//;
                }
            } elsif ($line =~ /Binary files .* differ/) {
                $line =~ s/(\s+)\Q$tmpdir_path\E\/[^\/]+\//$1TEMPDIR\//;
            }
            $diff .= $line . "\r\n";
        }
        utf8::encode($diff);
        my $hash = sha384_hex($diff);
        utf8::decode($diff);
        $diff = "Installed version for package $package: $package_version\r\n$diff"; # версию пакетов не хешируем
        $current_diffs{$hash} = $diff;
    }
}

if (equal_keys(\%current_diffs, \%previous_diffs)) {
    # do nothing, possibly reopen an issue
    if (%current_diffs && ($issue->[0]->{status} eq 'closed')) {
        $startrek->do(
            key => $old_key,
            actions => ["reopen"],
            comment => "Тикет переоткрыт\n$SIGN_COMMENT",
        );
        $log->out("Reopened $old_key");
    }
} elsif (%previous_diffs && !%current_diffs) {
    my $comment = 'Состояние поменялось, сейчас проблем не наблюдается.';
    foreach my $hash (keys %previous_diffs) {
        $issue->[0]->{description} =~ s/(^Diff hash: $hash)/--$1--/m;
    }

    $startrek->do(
        key => $old_key,
        description => $issue->[0]->{description},
        comment => "$comment\n$SIGN_COMMENT",
    );
} else {
    my $description = '';
    foreach my $hash (sort keys %current_diffs) {
        $description .= "Diff hash: $hash\r\n";
        $description .= '%%' . "\r\n";
        $description .= "$current_diffs{$hash}\r\n";
        $description .= '%%' . "\r\n";
    }

    my $new_issue_key = $startrek->do(
        create => 1,
        queue => $ST_queue,
        type => "bug",
        summary => "$fqdn package inconsistency",
        description => "$description\n$SIGN_COMMENT",
    );
    $log->out("Created an issue $new_issue_key");

    if (%previous_diffs) {
        my $comment = "Состояние поменялось, актуальный отчёт в новом тикете: $new_issue_key";
        $startrek->do(
            key => $old_key,
            comment => "$comment\n$SIGN_COMMENT",
        );
    }
}

juggler_ok();

sub equal_keys {
    my ($h1, $h2) = @_;
    my $k1string = join ';', sort keys %$h1;
    my $k2string = join ';', sort keys %$h2;
    return $k1string eq $k2string;
}
