#!/usr/bin/perl
use strict;
use warnings; 
=head1 DESCRIPTION

    Скрипт для расшифровки тайматргетингов из БД Директа (campaigns.timeTarget)

    Не зависит ни от чего в Директе, нужен для того, чтобы отдавать аналитикам.

    Читает из stdin таймтаргетинги, по одному в строке, 
    на stdout пишет json с раскладкой по дням и часам: для каждого дня и каждого часа -- коэффициент, который должен применяться к ставке. 

    100 -- полная ставка
    0 или отсутствие -- не показывать

    Функции взяты из svn+ssh://svn.yandex.ru/direct/trunk/protected/TimeTarget.pm@85387 

    Пример:

    > protected/maintenance/decode_direct_db_timetarget.pl                              
    -------ABCDEF------------------
    {"6":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"4":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"1":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"3":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"7":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"2":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100},"5":{"11":100,"21":100,"7":100,"17":100,"22":100,"18":100,"13":100,"23":100,"16":100,"6":100,"9":100,"12":100,"14":100,"15":100,"20":100,"8":100,"19":100,"10":100}}

=cut

use JSON;


run() unless caller();

sub run
{
    while (my $t = <>){
        $t =~ s/\s*$//;
        print encode_json(hours_hash($t))."\n";
    }
    exit 0;
}


=head2 hours_hash

    Из строки timetarget сделать хэш с ключами деньНедели / час

=cut
sub hours_hash {
    my ($timetarget) = @_;
    my %hours;
    if (length($timetarget) == 31 && $timetarget =~ /^[1-7\-]{7}[A-X\-]{24}$/) {
        # старый формат
        # TODO - убрать после конвертации базы, expire_date - 01.04.2010
        my %day_hours = map {$_ => 100} grep {index($timetarget, hour2letter($_)) < 0} (0..23);
        %hours = map {$_ => {%day_hours}} grep {$timetarget !~ /$_/} (1..7);
    } else {
        # новый, нормальный формат
        $timetarget =~ s/^([^;]*);(.*)$/$1/;
        my $dow = '';
        my $hour;
        for my $char (grep {/[0-9A-Xb-jl-u]/} split //, $timetarget) {
            if ($char =~ /\d/) {
                $dow = $char;
                $hours{$dow} = {};
            } elsif ($char =~ /[A-X]/) {
                $hour = letter2hour($char);
                $hours{$dow}->{$hour} = 100;
            } elsif ($char =~ /[b-jl-u]/ && defined $hour) {
                $hours{$dow}->{$hour} = letter2coef($char);
            } else {
                warn "$timetarget is not valid";
            }
        }
    }
    return \%hours;
}


=head2 letter2hour

    По букве A-X вернуть номер часа 0-23

=cut

sub letter2hour {
    my $letter = shift;
    return ord(uc($letter)) - ord('A');
}


=head2 letter2coef

    По букве b-j, l-u вернуть коэффициент 10 - 90, 110 - 200
    для "k" возвращаем 100, но сохранять такой не нужно, также как и "a"

=cut

sub letter2coef {
    my $letter = shift;
    return (ord(lc($letter)) - ord('b') + 1) * 10;
}


=head2 hour2letter

    По номеру часа 0-23 вернуть букву A-X

=cut

sub hour2letter {
    my $hour = shift;
    return chr(ord('A') + $hour);
}

