#!/usr/bin/perl

=head1 NAME

    enrich_tsv.pl

=head1 DESCRIPTION

    Дополняет tab-separated файл указанными данными из базы
    Удобно применять поверх выгрузок из баз, когда нужно заполнить то, что нельзя заполнить джойнами:
        - менеджеров
        - агентства
    Данные берёт из текущей для беты базы. Если хочется из продакшена -- надо переключать конфигурацию беты.

=head1 RUNNING

    Данные читает из STDIN
    В качестве аргументов командной строки принимает пары имён полей из какого надо сделать какое. Например:
        ManagerUID=uid->login=manager_login,email=manager_email
    это означает, что из поля STDIN'а ManagerUID надо взять значение и, рассматривая его как uid, сделать из него логин, который сохранить в поле manager_login.
    Заполняемое поле не должно существовать.

    Умеет делать такие преобразования:
        uid => login, email, FIO, phone, ClientID
        ClientID => name, currency, country_region_id, manager_uid, logins, chief_uid, chief_login, primary_manager_uid, primary_bayan_manager_uid, primary_geo_manager_uid

    m -B pr:ppc:all 'SELECT cid, ManagerUID, AgencyID FROM campaigns WHERE uid = 6138950' > raw.tsv
    ./protected/maintenance/enrich_tsv.pl 'ManagerUID=uid->login=manager_login,email=manager_email' 'AgencyID=ClientID->name=agency_name,logins=agency_logins < raw.tsv > enriched.tsv

=head1 CAVEAT

    Несколько преобразований за раз теоретически поддерживает, но не проверялось.

=cut

use my_inc '../..';

use Settings;
use Direct::Modern;
use open ':std' => ':utf8';
use ScriptHelper get_file_lock => undef, script_timer => undef;
use PrimitivesIds qw(get_uid2login);
use RBAC2::Extended ();
use RBACElementary qw(rbac_get_chief_reps_of_clients rbac_get_chief_reps_of_agencies);

use Yandex::DBTools;
use Yandex::Overshard;
use Yandex::DBShards;

use List::MoreUtils qw(zip any uniq);

use constant ENRICH_CHUNK_SIZE => 1_000;

extract_script_params();

do_sql(PPC(shard => 'all'), 'SET GROUP_CONCAT_MAX_LEN=9999');

my (%field_types, %field2field_map);
for my $raw_mapping (@ARGV) {
    my ($from_raw, $to_raw) = split /->/, $raw_mapping, 2;

    my ($from_field, $from_type) = split /=/, $from_raw, 2;
    $field_types{$from_field} = $from_type;

    my @to_mappings = split /,/, $to_raw;
    for my $to_mapping (@to_mappings) {
        my ($to_field, $to_type) = split /=/, $to_mapping, 2;
        $field2field_map{$from_field}{$to_field} = $to_type;
    }
}


my (@in_header_fields, @header_fields, @values_buffer, $header_field_count);
while (my $line_raw = <STDIN>) {
    $line_raw =~ s/[\r\n]//g;

    # этим гадит mysql-клиент и, как следствие, direct-sql
    next if $line_raw eq 'PAGER set to stdout';

    if (@in_header_fields) {
        my @line_values = split /\t/, $line_raw, $header_field_count;
        my %line = zip @in_header_fields, @line_values;
        push @values_buffer, \%line;

        if (@values_buffer > ENRICH_CHUNK_SIZE) {
            _enrich_print_rows(\@values_buffer);
            @values_buffer = ();
        }
    } else {
        @in_header_fields = split /\t/, $line_raw;
        my %already_in_header = map {$_ => undef} @in_header_fields;
        $header_field_count = scalar @in_header_fields;

        @header_fields = @in_header_fields;
        push @header_fields, grep { !$already_in_header{$_} } sort map { values %$_ } values %field2field_map;
        say join("\t", @header_fields);
    }
}

if (@values_buffer) {
    _enrich_print_rows(\@values_buffer);
    @values_buffer = ();
}

sub _enrich_print_rows {
    my ($rows) = @_;

    _enrich_rows($rows);
    _print_rows($rows);
}

sub _enrich_rows {
    my ($rows) = @_;

    for my $field (sort keys %field_types) {
        my $type = $field_types{$field};
        my $field_map = $field2field_map{$field};
        my $enrich_sub;
        if ($type eq 'uid') {
            $enrich_sub = sub {
                my ($uids) = @_;
                return _enrich_uids($uids, $field);
            };
        } elsif ($type eq 'ClientID') {
            $enrich_sub = sub {
                my ($clientids) = @_;
                return _enrich_clientids($clientids, $field);
            };
        } elsif ($type eq 'cid') {
            $enrich_sub = sub {
                my ($cids) = @_;
                return _enrich_cids($cids, $field);
            }
        } else {
            die "unknown type '$type'";
        }
        enrich_data($rows, using => $field, map_fields => $field_map, $enrich_sub);
    }
}

sub _print_rows {
    my ($rows) = @_;

    for my $row (@$rows) {
        say join("\t", map { $row->{$_} // '' } @header_fields);
    }
}

sub _enrich_uids {
    my ($uids, $field) = @_;

    return get_hashes_hash_sql(PPC(uid => [grep { $_ && $_ ne 'NULL' } @$uids]), ["SELECT uid AS $field, login, email, FIO, phone, ClientID FROM users", WHERE => { uid => SHARD_IDS }]);
}

sub _enrich_cids {
    my ($cids, $field) = @_;

    return get_hashes_hash_sql(PPC(cid => [grep { $_ && $_ ne 'NULL' } @$cids]), ["SELECT cid AS $field, uid, ManagerUID, AgencyUID, name, LastChange, OrderID, AgencyID, IFNULL(currency, 'YND_FIXED') AS currency, sum, sum_spent, wallet_cid, statusModerate, statusShow, shows, clicks, archived, type, currencyConverted, copiedFrom FROM campaigns", WHERE => { cid => SHARD_IDS }]);
}


sub _enrich_clientids {
    my ($clientids, $field) = @_;

    my $clid2data = get_hashes_hash_sql(PPC(ClientID => [grep { $_ && $_ ne 'NULL' } @$clientids]), ["
        SELECT u.ClientID AS $field
             , cl.name
             , IFNULL(cl.work_currency, 'YND_FIXED') AS currency
             , cl.country_region_id
             , cl.primary_manager_uid
             , cl.primary_manager_set_by_idm
             , cl.primary_bayan_manager_uid
             , cl.primary_geo_manager_uid
             , IFNULL(cl.primary_manager_uid, IFNULL(cl.primary_bayan_manager_uid, cl.primary_geo_manager_uid)) AS manager_uid
             , GROUP_CONCAT(u.login) AS logins
        FROM users u
        LEFT JOIN clients cl ON cl.ClientID = u.ClientID
     ", WHERE => { 'u.ClientID' => SHARD_IDS }, "
        GROUP BY u.ClientID
     "]);
     if (%$clid2data) {
         my $rbac = RBAC2::Extended->get_singleton(1);
         my $clid2client_chief_uid = rbac_get_chief_reps_of_clients($clientids);
         my $clid2agency_chief_uid = rbac_get_chief_reps_of_agencies($clientids);
         my @uids = uniq(values %$clid2client_chief_uid, values %$clid2agency_chief_uid);
         my $chief_uid2login = get_uid2login(uid => \@uids);
         for my $clid (keys %$clid2data) {
             my $chief_uid = $clid2client_chief_uid->{$clid} || $clid2agency_chief_uid->{$clid};
             my $chief_login = (defined $chief_uid) ? $chief_uid2login->{$chief_uid} : '';
             $clid2data->{$clid}->{chief_uid} = $chief_uid;
             $clid2data->{$clid}->{chief_login} = $chief_login;
         }
     }
     return $clid2data;
}
