#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    export_wordstat_history_csv.pl

=head1 DESCRIPTION

    Выгружает в CSV количество запросов за предыдущие периоды для указанных запросов.
    Работает с ADVQ6. 

    Принимает параметры:
        --min-date YYYYMM -- позволяющий ограничить минимальную дату, за которую будет выгружено количество запросов
        --regions -- разделённый запятыми список регионов, по которым будет производиться выгрузка (например, 225 -- Россия)

    Было бы здорово его распараллелить. Сейчас скорость порядка 250 фраз в минуту.

    LOG_TEE=1 ./protected/maintenance/export_wordstat_history_csv.pl --min-date 201001 --regions 225 < words.txt > wordstat.csv

=cut

use strict;
use warnings;

use ScriptHelper get_file_lock => undef;
use ADVQ6;
use List::MoreUtils qw/uniq/;
use IO::Wrap;
use Text::CSV;

use utf8;
use open ':std' => ':utf8';

my $BUNDLE_SIZE = 100;
my $BUNDLE_TIMEOUT = 600;

$log->out('START');

my ($startdate, $regions);
extract_script_params(
    'min-date=i' => \$startdate,
    'regions=s' => \$regions,
);

my (%words_data, %avaible_dates);

my @all_words = <>;
chomp @all_words;
@all_words = uniq @all_words;
utf8::encode($_) for @all_words;

while(my @words = splice @all_words, 0, $BUNDLE_SIZE) {
    $log->out('Fetching next bundle with ' . scalar(@words) . ' phrases (' . scalar(@all_words) . ' left)');
    my %request_params = (function => 'monthly_hist', timeout => $BUNDLE_TIMEOUT);
    $request_params{regions} = $regions if $regions;
    my $data = ADVQ6::advq_request(\@words, %request_params);
    for my $worddata(@$data) {
        if ($worddata->{error}) {
            $log->warn("ADVQ ERROR: " . $worddata->{error});
            $log->out($worddata);
        }
        my $word = $worddata->{req};
        for my $histdata (@{$worddata->{hist}}) {
            my $date = sprintf '%.4d%.2d', $histdata->{year}, $histdata->{month};
            $words_data{$word}{$date} = $histdata->{total_count};
            $avaible_dates{$date} = 1;
        }
    }
}

my @dates = sort {$a <=> $b} keys %avaible_dates;
@dates = grep {$_ >= $startdate} @dates if $startdate;

my $csv = Text::CSV->new({binary => 1, eol => "\r\n"}) or die "Cannot use CSV: ".Text::CSV->error_diag ();
my $stdout_fh = wraphandle(\*STDOUT);

$csv->print($stdout_fh, ['keyword', @dates]);
while (my($word, $word_data) = each %words_data) {
    $csv->print($stdout_fh, [$word, map {$word_data->{$_} || 0} @dates]);
}

$log->out('FINISH');
