#!/usr/bin/perl

use my_inc "../..";


# $Id$

=head1 NAME

    fix_addresses_auto_points.pl

=head1 DESCRIPTION

    Скрипт для перепроверки кеша точек геокодера и обновления авто-точек адресов из визиток

    Поддерживает следующие ключи запуска (полезны для отладки):
    --shard-id=NN   номер шарда для обработки (если не указан - будут обработаны все шарды)
    --from-date     Добавляет к sql-запросу, получающему данные для обработки условие в WHERE
                        AND logtime > $opt{'from-date'}
    --aid           Выполняет все действия только для одного, указанного address_id

    --dry-run       Отменяет обновление данных в базе
    --store-cache   Сохраняет данные, полученные из геокодера в файл cache-path
    --load-cache    Считывает данные по адресам из файла cache-path. Существенно ускоряет выполнение
    --cache-path    Задает путь к файлу для хранения/загрузки кеша геокодера.
                        По-умолчанию /tmp/fix_addr_geo.txt
    --stats         Подсчитывает и записывает в лог затрагиваемое работой скрипта количество:
                        визиток, баннеров, кампаний, пользователей
    --verbose       Печатает все что происходит на STDERR. (!) больше 100 Мб данных

=cut

use strict;
use warnings;
use utf8;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry;
use Yandex::TimeCommon qw/check_mysql_date/;
use Yandex::HashUtils qw/hash_copy hash_cut/;
use ScriptHelper sharded => 0;

use Settings;
use CommonMaps;
use Lang::Guess qw/analyze_text_lang/;
use EnvTools;
use ShardingTools;

use JavaIntapi::GeosearchSearchObject;

use open ':std' => ':utf8';
$|++;

# На бетах qeryrec не доступен.
local $Lang::Guess::EXTERNAL_QUERYREC = sub { return {eng => 1}; } if !is_production();

my @all_ppc_shards = ppc_shards();

# Задает "во сколько раз больше времени выполнения" апдейта к базе делать sleep после каждого.
my $SLEEP_COEF = 2;


extract_script_params(\my %opt,
    "dry-run+",
    "store-cache+",
    "load-cache+",
    "cache-path=s",
    "from_date=s",
    "verbose+",
    "aid=i",
    'shard-id=i',
    "stats+"
);

if ($opt{'shard-id'}) {
    $log->die('Invalid shard number: ' . ($opt{'shard-id'}||'undef')) unless any { $_ == $opt{'shard-id'} } @all_ppc_shards;
} else {
    $log->out('All shards will be processed');
}

# Путь для отладочной информации
$opt{'cache-path'} ||= '/tmp/fix_addr_geo.txt';

if ($opt{'store-cache'}) {
    # Открываем файл сейчас, чтобы умереть до запроса данных, а не после.
    open (GEO_W, ($opt{'load-cache'} ? '>' : '>>') . ':utf8', $opt{'cache-path'}) or die "Can't open file ".$opt{'cache-path'}.": $!";
}

# хеш, хранящий ответы геокодера, ключ - адрес
my $maps_from_geocoder = {};

# хеш под хранение того, какими id нужно будет обновлять поле map_id_auto в таблице addresses
# ключ - mid таблицы maps, значение - массив aid таблицы addresses, для которых нужно будет присвоить map_id_auto = mid
my $map_ids_for_update = {};

# хеш старых значений таблицы addresses. ключ - aid, значение - map_id_auto до обновления.
my $backup_log = {};


$log->out('START');

my $and_aid = '';
if ($opt{aid}) {
    $and_aid = "AND aid = " . sql_quote($opt{aid});
}

my $totally_failed_gc_requests = 0;
# цикл по шардам
foreach my $SHARD ($opt{'shard-id'} || @all_ppc_shards) {
    $log->out("Processing shard $SHARD");

    # Проставляем id автоматической точки в качестве ручной для тех адресов, где ручной точки нет (совсем старые, примерно 500 штук)
    my $addresses_without_map_id = get_one_column_sql(PPC(shard => $SHARD),
        "SELECT aid FROM addresses WHERE map_id_auto IS NOT NULL AND map_id IS NULL $and_aid;") || [];
    if (scalar @$addresses_without_map_id) {
        $log->out('Updating addresses without manual point id - SET map_id = map_id_auto');
        $log->out('_updated_aids: ' . (join ', ', @$addresses_without_map_id));
        unless ($opt{'dry-run'}) {
            do_update_table(PPC(shard => $SHARD), 'addresses',
                {
                    map_id  => 'map_id_auto',
                    logtime => 'NOW()',
                },
                where => { aid => $addresses_without_map_id },
                dont_quote => ['map_id', 'logtime']
            );
        }
    } else {
        $log->out('Addresses without manual point id were not found');        
    }
    $log->out('Assigned manual point id equal to auto id for ' . (scalar @$addresses_without_map_id) . ' addresses');


    # Загрузка ответов геокодера из tab-separated файла
    if ($opt{'load-cache'}) {
        # Подгружаем данные из кеша
        $log->out('Loading cached geo data');
        open (GEO, '<:utf8', $opt{'cache-path'}) or die "can't load cached geo from file ".$opt{'cache-path'}.": $!";
        while (my $line = <GEO>) {
            chomp ($line);
            my @data = split "\t", $line;
            my $addr = shift @data;
            unless (exists $maps_from_geocoder->{$addr}) {
                $maps_from_geocoder->{$addr}->{$_} = shift @data for qw/auto_point auto_bounds x y x1 y1 x2 y2/;
            }
        }
        close GEO;
        $log->out('Loading cached geo data finished');
    }

    # Получим id точки "вся земля" (такую точку CommonMaps ставит при плохом ответе гео-кодера)
    my $dp = {};
    ($dp->{x}, $dp->{y}) = split ',', $CommonMaps::DEFAULT_AUTO_POINT;
    ($dp->{x1}, $dp->{y1}, $dp->{x2}, $dp->{y2}) = split ',', $CommonMaps::DEFAULT_AUTO_BOUNDS;
    my $default_id = get_one_field_sql(PPC(shard => $SHARD), ["
        SELECT  mid
        FROM    maps",
        where => $dp]
    );
    unless ($default_id) {
        $log->out("Default point wasn't found in DB");

        if ($opt{'dry-run'}) {
            # настоящее значение id не требуется при dry-run
            $default_id = 1;
        } else {
            # Добавим в базу точку "по-умолчанию", чтобы знать её id
            $dp->{mid} = get_new_id('maps_id');
            $default_id = $dp->{mid};
            do_insert_into_table(PPC(shard => $SHARD), 'maps' => $dp);
            $log->out("Added default point to DB, id: $default_id");
        }
    }
    die "Unknown default map point ID." unless $default_id; # impossible

    # Выборка всех адресов из базы
    $log->out('SELECT FROM addresses - start');
    my $and_logtime = '';
    if ($opt{'from-date'}) {
        if (!check_mysql_date ($opt{'from-date'})) {
            die 'incorrect date specified!';
        } else {
            $and_logtime = "AND logtime > " . $opt{'from-date'};
        }
    }
    # У некоторых адресов попадаются одинаковые map_id_auto на разные адреса
        # Среди старых точек такое для совсем разных улиц и домов бывает
        # Среди новых точек - различия в записи адреса (проезд - пр, пробелы)
    # Поэтому группируем выборку не по map_id_auto, а по address_id
    # Еще тонкость - в базе попадаются адреса, у которых есть ручная точка, но нет авто
        # после предыдущего UPDATE базы можно считать, что у всех адресов "с точками" - есть ручная
    # Поэтому в условии WHERE ставим не map_id_auto, а map_id
    my $sth = exec_sql(PPC(shard => $SHARD), "
        SELECT      aid, map_id_auto, address, city,
                    CONCAT_WS(',', maps_auto.x, maps_auto.y) as auto_point
        FROM        addresses adr
                    LEFT JOIN maps maps_auto ON adr.map_id_auto = maps_auto.mid
                    LEFT JOIN vcards v ON adr.aid = v.address_id
        WHERE       map_id IS NOT NULL
                    AND address != ''
                    $and_logtime
                    $and_aid
        GROUP BY    aid
        -- LIMIT       100
    ");
    $log->out('SELECT FROM addresses - end');

    $log->out('Processing selected data');
    my $rows = 0;
    my $gc_requests = 0;

    while(my ($aid, $map_id_auto, $address, $city, $auto_point) = $sth->fetchrow_array) {
        $rows++;

        $address =~ s/,null$//; # В старых адресах попадается такой хвост
        # Убираем лишние пробелы, запятые, мусор
        $address =~ s/^\W+//;
        $address = CommonMaps::_filter_address($address);

        print STDERR "processing $address,\taid $aid,\tmap_id_auto $map_id_auto\n" if $opt{verbose};

        if (!exists $maps_from_geocoder->{$address} || !$maps_from_geocoder->{$address}->{x} || !$maps_from_geocoder->{$address}->{y}) {
            # У нас еще нет результатов из геокодера по заданному адресу
            # Запрашиваем и сохраняем себе в хеш
            print STDERR "request coordinates from geocoder\n" if $opt{verbose};
            my $map;

            # Иногда у гео-кодера случается "нелётная погода" на каких-то отрезках времени,
            # Поэтому так много повторов и все равно eval и сохраняем в лог то, что не получилось
            eval {
                retry tries => 10, pauses => [1, 3], sub {
                    $gc_requests++;
                    my ($maps, $error) = JavaIntapi::GeosearchSearchObject->_search_address($address, {lang => get_lang($address)});
                    $map = CommonMaps::_select_first_point($maps, $city);
                };
            };
            if ($@) {
                $totally_failed_gc_requests++;
                $log->out("!request failed for address_id $aid");
                next;
            }

            # Сохраним себе в хеш только интересующие нас поля ответа геокодера, в плоском виде
            $maps_from_geocoder->{$address} ||= {};
            hash_copy($maps_from_geocoder->{$address}, $map->{point}, 'x', 'y');
            hash_copy($maps_from_geocoder->{$address}, $map->{bound}, 'x1', 'y1', 'x2', 'y2');
            hash_copy($maps_from_geocoder->{$address}, $map, 'auto_point', 'auto_bounds');

            # Координаты "всей земли".
            if ($map->{auto_point} eq $CommonMaps::DEFAULT_AUTO_POINT
                && $map->{auto_bounds} eq $CommonMaps::DEFAULT_AUTO_BOUNDS
            ) {
                # Результат геокодера не понравился модулю CommonMaps
                # и тот вернул в качестве точки москву с зумом на всю землю
                # ID такой точки у нас уже есть
                $maps_from_geocoder->{$address}->{auto_id} = $default_id;
            }
        }

        if ( $auto_point ne $maps_from_geocoder->{$address}->{auto_point} ) {
            print STDERR "points not equal\n" if $opt{verbose};

            # Координаты из геокодера и нашего кеша не совпадают - нужно сохранить новые координаты в maps.
            # В предпололжении, что геокодер не может вернуть на один и тот же адрес разные координаты.
            unless ($maps_from_geocoder->{$address}->{auto_id}) {
                # Еще не знаем какой mid таблицы maps соответствует нашим координатам
                unless ($opt{'dry-run'}) {
                    $maps_from_geocoder->{$address}->{auto_id} = do_insert_into_table(PPC(shard => $SHARD),
                        'maps' => hash_cut($maps_from_geocoder->{$address}, 'x', 'y', 'x1', 'y1', 'x2', 'y2'),
                        on_duplicate_key_update => 1,
                        key => 'mid'
                    );
                } else {
                    # "заглушка", чтобы не перезапрашивать координаты в гео-кодере
                    $maps_from_geocoder->{$address}->{auto_id} = $default_id;
                }
            }

            # Сохраняем aid чтобы потом обновить в addresses map_id_auto
            push @{ $map_ids_for_update->{ $maps_from_geocoder->{$address}->{auto_id} } }, $aid;

            # Сохраним старое значение map_id_auto для логов
            $backup_log->{$aid} = $map_id_auto;

            print STDERR "pushed auto_id $maps_from_geocoder->{$address}->{auto_id} aid $aid\n" if $opt{verbose};
        } else {
            print STDERR "point ok\n" if $opt{verbose};
        }
    }
    $sth->finish();
    $log->out('Processing selected data finished');


    # Сохраняем ответы геокодера в файл
    if ($opt{'store-cache'}) {
        $log->out('Flushing geo data');
        while (my ($addr, $map) = each %$maps_from_geocoder) {
            print GEO_W join "\t", $addr, map {$map->{$_} || ''} qw/auto_point auto_bounds x y x1 y1 x2 y2/;
            print GEO_W "\n";
        }
        close GEO_W;
        $log->out('Flushing geo data finished');
    }


    # Обновляем таблицу адресов - проставляем новые id авто-точек
    $log->out('Updating addresses with new map_id_auto');
    # Плоский список aid для запроса статистики
    my $affected_aids = [];
    my $updated_addresses = 0;
    while (my ($map_auto_id, $address_ids) = each %$map_ids_for_update) {
        print STDERR "updating mid $map_auto_id\tcount aid ", (scalar @$address_ids), ': ', (join ',', @$address_ids), "\n" if $opt{verbose};

        # Пишем в лог что меняем и что было
        $log->out('_update_aid ' . $_ . "\tset map_id_auto " . $map_auto_id . "\told " . ($backup_log->{$_} || 'NULL') ) for @$address_ids;

        unless ($opt{'dry-run'}) {
            my $rows = relaxed times => $SLEEP_COEF, sub {
                do_update_table(PPC(shard => $SHARD), 'addresses',
                    {
                        map_id_auto         => $map_auto_id,
                        logtime__dont_quote => 'NOW()',
                    },
                    where => {
                        aid     => $address_ids,
                    }
                ) || 0;
            };
            $updated_addresses += $rows;
            print STDERR "updated addresses $rows\n" if $opt{verbose};
        }
        if ($opt{stats}) {
            push @$affected_aids, @$address_ids;
        }
    }
    $log->out('Updating addresses finished.');


    # Статистика по затрагиваемым данным. Тяжелые запросы
    if ($opt{stats}) {
        $log->out('Selectings stats:');

        my $count = get_one_field_sql(PPC(shard => $SHARD), ["
                SELECT  COUNT(DISTINCT vcard_id)
                FROM    addresses a
                            LEFT JOIN vcards v ON v.address_id = a.aid
                            LEFT JOIN banners b USING(vcard_id)
                            LEFT JOIN campaigns c on c.cid = b.cid
                            LEFT JOIN phrases p on p.pid=b.pid
            ",
            where   => { aid => $affected_aids },
        ]);
        $log->out("Affected $count vcards");

        $count = get_one_field_sql(PPC(shard => $SHARD), ["
                SELECT  COUNT(DISTINCT b.bid)
                FROM    addresses a
                            LEFT JOIN vcards v ON v.address_id = a.aid
                            LEFT JOIN banners b USING(vcard_id)
                            LEFT JOIN campaigns c on c.cid = b.cid
                            LEFT JOIN phrases p on p.pid=b.pid
            ",
            where   => { aid => $affected_aids },
        ]);
        $log->out("Affected $count banners");

        $count = get_one_field_sql(PPC(shard => $SHARD), ["
                SELECT  COUNT(DISTINCT b.cid)
                FROM    addresses a
                            LEFT JOIN vcards v ON v.address_id = a.aid
                            LEFT JOIN banners b USING(vcard_id)
                            LEFT JOIN campaigns c on c.cid = b.cid
                            LEFT JOIN phrases p on p.pid=b.pid
            ",
            where   => { aid => $affected_aids },
        ]);
        $log->out("Affected $count campaigns");

        $count = get_one_field_sql(PPC(shard => $SHARD), ["
                SELECT  COUNT(DISTINCT c.uid)
                FROM    addresses a
                            LEFT JOIN vcards v ON v.address_id = a.aid
                            LEFT JOIN banners b USING(vcard_id)
                            LEFT JOIN campaigns c on c.cid = b.cid
                            LEFT JOIN phrases p on p.pid=b.pid
            ",
            where   => { aid => $affected_aids },
        ]);
        $log->out("Affected $count users");
    }

    my $cnt = scalar keys %$map_ids_for_update;
    $cnt-- if exists $map_ids_for_update->{$default_id};

    $log->out('Selected & processed rows: ' . $rows);
    $log->out('Processed addresses: ' . (scalar keys %$maps_from_geocoder) );
    $log->out("Requests from geocoder $gc_requests");
    $log->out("new/updated map points: $cnt") unless $opt{'dry-run'};
    $log->out("Affected $updated_addresses addresses rows");

} # конец цикла по шардам

# Если какие-то адреса совсем не удалось отработать - предупредим об этом
if ($totally_failed_gc_requests) {
    $log->out('!' x 10);
    $log->out('Totally failed requests from geocoder: ' . $totally_failed_gc_requests);
    $log->out("Check logfile for '!request failed for address_id XXXXX' entries");
}

$log->out('FINISH');
exit;

sub get_lang {
    my $text_lang = analyze_text_lang(shift) || '';
    if ($text_lang eq 'uk') {
        return 'uk-UA';
    } elsif ($text_lang eq 'en') {
        return 'en-US';
    } elsif ($text_lang eq 'tr') {
        return 'tr-TR';
    } else {
        return 'ru-RU';
    }
}
