#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    fix_vcard_empty_geo_id.pl

=head1 DESCRIPTION

    Скрипт для заполнения поля geo_id в визитках, у которых geo_id = 0 установлено ошибочно
        (т.е. правильное значение может быть определено)

    Принимает опциональный параметр:
        --shard-id=NN   - номер шарда для обработки. Если не указан - будут обработаны все шарды

=cut

use strict;
use warnings;

use Yandex::Retry;
use Yandex::DBTools;

use lib::abs '..';

use Settings;
use GeoTools qw/get_geoid_by_cityname/;
use ShardingTools;
use BS::ResyncQueue;
use List::MoreUtils qw/any/;

use ScriptHelper sharded => 0;  # потому что модуль не позволяет пока работать со всеми шардами

use utf8;

$log->out('START');

my @all_ppc_shards = ppc_shards();
my $one_shard;
extract_script_params(
    'shard-id=i'    => \$one_shard,
);

if ($one_shard) {
    $log->die('Invalid shard number: ' . ($one_shard||'undef')) unless any { $_ == $one_shard } @all_ppc_shards;
} else {
    $log->out('All shards will be processed');
}

foreach my $SHARD ($one_shard || @all_ppc_shards) {
    $log->msg_prefix("[shard $SHARD]");
    $log->out("Start processing shard");

    my $sth = exec_sql(PPC(shard => $SHARD), q/SELECT vcard_id, city FROM vcards WHERE geo_id = 0/);

    while (my ($vcard_id, $city) = $sth->fetchrow_array) {
        # Пытаемся вычислить geo_id
        my $geo_id = get_geoid_by_cityname($city);
        # Определить не получилось
        next unless $geo_id;

        my $banners_to_resync = get_all_sql( PPC(shard => $SHARD), [
                'SELECT cid, bid FROM banners b',
                WHERE => {
                    vcard_id => $vcard_id,
                    'b.statusBsSynced' => 'Yes',
                },
            ]
        );

        relaxed times => 3, sub {
            do_in_transaction {
                my $updated = do_update_table( PPC(shard => $SHARD), 'vcards',
                    { geo_id => $geo_id },
                    where => {
                        vcard_id => $vcard_id,
                        geo_id => 0,
                        city => $city,
                    },
                );

                if ( ($updated + 0) && @$banners_to_resync ) {
                    $_->{priority} = BS::ResyncQueue::PRIORITY_MAINTENANCE_FIX_VCARD_GEO_ID for @$banners_to_resync;
                    bs_resync($banners_to_resync);
                }
            };
            $log->out("updated vcard #$vcard_id: set geo_id = $geo_id");
            $log->out("banners ro resync: ".join(", ", map {$_->{bid}} @$banners_to_resync)) if @$banners_to_resync;
        };
    }
    $sth->finish();
}
$log->msg_prefix(undef);
$log->out('FINISH');
