#!/usr/bin/perl

use my_inc "../..";

use strict;
use warnings;
use utf8;

# $Id$

=head1 NAME

protected/maintenance/generate-glusterfs-config.pl - создать конфигурацию для сервера glusterfs

=head1 SYNOPSIS

 protected/maintenance/generate-glusterfs-config.pl \
     --package yandex-direct --outfile debian/yandex-direct/etc/gluster-automount.d/yandex-direct.conf

=head1 DESCRIPTION

http://wiki.yandex-team.ru/direkt/gluster-logs

Краткая версия: реальные каталоги с логами нужно примонтировать с опцией bind в /var/remote-log;
к этой папке с помощью glusterfs-server предоставляется доступ по сети, и клиент на ppcdev может
получить к ним доступ и примонтировать в /mnt/remote-log.

Этот скрипт отвечает за файлы конфигурации, определяющие первую часть: какие реальные каталоги
с логами и куда в /var/remote-log монтировать. Таблица соответствия между названием пакета и списком
каталогов хранится в исходном коде константой.

Поддерживаемые аргументы командной строки:

    --help
    --usage

    --package   Пакет, которому принадлежит папка с логами.

    --outfile   Куда сохранить получившуюся конфигурацию.

Пример файла конфигурации:

$ cat /etc/gluster-automount.d/yandex-direct.conf
/var/www/ppc.yandex.ru/protected/logs protected

=cut

use autodie;

use File::Basename qw( dirname );
use File::Path qw( make_path );
use File::Slurp qw( write_file );
use Getopt::Long;
use Readonly;

use Yandex::Shell;

# здесь хранится соответствие между названием пакета и содержимом файла в нём;
# содержимое файла - тоже соответствие между целевым каталогом в /var/remote-log и исходным каталогом:
# ( 'package' => { 'destdir' => '/var/log/sourcedir' }, ... )
Readonly my %DIRECTORY_MAPS => {
    'yandex-direct' => {
        'protected' => '/var/www/ppc.yandex.ru/protected/logs',
        'dscribe' => '/var/log/dscribe',
    },
    'yandex-direct-fake-services' => {
        'fake-services.direct.yandex.ru' => '/var/log/fake-services.direct.yandex.ru',
    },
    'yandex-direct-frontend' => {
        'ppc.yandex.ru' => '/var/log/ppc.yandex.ru',
        'direct-accel'  => '/var/log/direct-accel',
    },
    'yandex-direct-intapi' => {
        'intapi.direct.yandex.ru' => '/var/log/intapi.direct.yandex.ru',
        'intapi-direct-accel' => '/var/log/intapi-direct-accel',
    },
    'yandex-direct-soap' => {
        'soap.direct.yandex.ru' => '/var/log/soap.direct.yandex.ru',
        'soap-direct-accel'     => '/var/log/soap-direct-accel',
    },
};

=head1 SUBROUTINES/METHODS

=head2 usage

=cut

sub usage {
    yash_system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    return;
}

=head2 run

=cut

sub run {
    my ( $opt_usage, $opt_package, $opt_outfile );
    my $getopt_ok = GetOptions(
        'help|usage' => \$opt_usage,
        'package=s'  => \$opt_package,
        'outfile=s'  => \$opt_outfile,
    );

    if ( !$getopt_ok || $opt_usage ) {
        usage();
        exit 1;
    }

    die 'Package is unspecified' unless $opt_package;

    my $package_config = $DIRECTORY_MAPS{$opt_package};
    die "Invalid package $opt_package" unless $package_config;

    die 'Output file is unspecified' unless $opt_outfile;

    my $config_string = '';
    foreach my $directory ( sort keys %$package_config ) {
        my $source_directory = $package_config->{$directory};
        $config_string .= "$source_directory $directory\n";
    }

    File::Path::make_path( dirname($opt_outfile), { mode => 0755 } );
    write_file( $opt_outfile, { perms => 0644 }, $config_string );
}

run();
