#!/usr/bin/perl

=encoding utf8

=head1 SYNOPSIS

    import_client_data.pl --dir ./client-data

=head1 DESCRIPTION

    Скрипт-антипод export_client_data.pl

    Привозит данные клиентов из sql-файлов в директории, указанной опцией dir, в базу.
    Файлы образуются в результате работы скрипта export_client_data.pl --dir <dir> --login ...
    Если клиент уже есть в базе и указана опция --move-existing-to-next-shard, то пишет в следующий шард ppc (или в первый, если клиент из последнего шарда) и прописывает новый шард в метабазе.
    Если указана опция --replace-existing, то данные по уже существующим ключам перезапишутся (полезно при повторной попытке после неудачи).

=cut

use strict;
use warnings;

use utf8;

use open ':encoding(UTF-8)';

use Getopt::Long;

use my_inc '../..';
use Settings;
use Rbac qw/:const get_perminfo/;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Log;

our $OPERATOR_UID = 1;

run() unless caller();

sub run {
    my $opt = parse_options();
    my $dir = $opt->{dir};
    my @client_ids = map { s!^\Q$dir\E/ppc\.([0-9]+)\.sql$!$1!r } glob "$dir/ppc.*.sql";

    my $log = Yandex::Log->new(
        log_file_name => 'import_client_data.log',
        date_suf => '%Y%m%d',
        auto_rotate => 1,
    );
    $log->{tee} = $ENV{LOG_TEE};

    $log->out("creating clients in ppcdata");
    my %shard_for = %{ get_shard_multi(ClientID => \@client_ids) };
    my %clients_for_shard;
    for my $client_id (@client_ids) {
        my $new_shard;
        if (defined $shard_for{$client_id}) {
            if ($opt->{new_shard}) {
                $new_shard = $opt->{new_shard};
            } elsif ($opt->{move_existing_to_next_shard}) {
                $new_shard = $shard_for{$client_id} % $Settings::SHARDS_NUM + 1;
            } elsif ($opt->{replace_existing}) {
                $new_shard = $shard_for{$client_id};
            } else {
                $log->warn("ClientID $client_id already exists, use --move-existing-to-next-shard if you want to recreate client in another shard or --replace-existing to replace data in current shard\n");
                next;
            }
        } else {
            $new_shard = 1;
        }
        push @{ $clients_for_shard{$new_shard} }, $client_id;
    }

    for my $client_id (@client_ids) {
        open my $fh, '<:encoding(UTF-8)', "$dir/ppcdict.$client_id.sql" or do { $log->warn("can't open $dir/ppcdict.$client_id.sql: $!"); next };
        while (my $line = <$fh>) {
            if ($line !~ /^INSERT\s+INTO\s+/) {
                $log->warn("suspicious query, not executing: $line");
                next;
            }
            chomp $line;
            if ($opt->{replace_existing}) {
                $line =~ s/^INSERT/REPLACE/;
            } else {
                $line =~ s/^INSERT/INSERT IGNORE/;
            }
            do_sql(PPCDICT, $line);
        }
        close $fh;
    }

    for my $shard (keys %clients_for_shard) {
        my @shard_client_ids = @{ $clients_for_shard{ $shard } };
        do_sql(PPC(shard => $shard), 'SET SESSION FOREIGN_KEY_CHECKS = 0');
        for my $client_id (@shard_client_ids) {
            # открываем с raw, потому что есть blob'ы, не представимые в UTF-8
            open my $fh, '<:raw', "$dir/ppc.$client_id.sql" or do { $log->warn("can't open $dir/ppc.$client_id.sql, skipping to next client"); next };
            while (my $line = <$fh>) {
                if ($line !~ /^(INSERT|REPLACE\s+INTO)\s+/) {
                    $log->warn("suspicious query, not executing: $line");
                    next;
                }
                chomp $line;
                $line =~ s/^INSERT\b/REPLACE/ if $opt->{replace_existing} && $line !~ /^INSERT\s+IGNORE\b/;
                if ($line =~ /bs_resync_queue/) {
                    warn "skipping bs_resync_queue\n";
                    next;
                } 
                do_sql(PPC(shard => $shard), $line);
            }
            close $fh;
            eval { do_sql(PPC(shard => $shard), 'UPDATE users SET fio = "Танзиля Потехина" WHERE ClientID = ?', $client_id); } or $log->warn("couldn't change name to fake: $@");
        }
        do_update_table(PPCDICT, 'shard_client_id', { shard => $shard }, where => { ClientID => \@shard_client_ids });
    }

    my @camps = @{ get_all_sql(PPC(ClientID => \@client_ids), ['SELECT c.cid, u.uid, c.ManagerUID, c.AgencyUID, c.AgencyID FROM campaigns c JOIN users u on c.uid = u.uid', where => {'u.ClientID' => \@client_ids} ]) };
    for my $camp (@camps) {
        if ($camp->{ManagerUID}) {
            my $manager_perminfo = get_perminfo( uid => $camp->{ManagerUID} );
            if (! ( $manager_perminfo && $manager_perminfo->{role} eq $ROLE_MANAGER ) ) {
                $log->warn("you must import manager (ManagerUID = $camp->{ManagerUID})");
            }
        } elsif ($camp->{AgencyUID}) {
            my $agency_perminfo = get_perminfo( uid => $camp->{AgencyUID} );
            if (! ( $agency_perminfo && $agency_perminfo->{role} eq $ROLE_AGENCY ) ) {
                $log->warn("you must import agency (AgencyUID = $camp->{AgencyUID})");
            }
        }
    }

    $log->out("done");
}

sub parse_options {
    my %O;
    GetOptions(
        'dir=s' => \$O{dir},
        'move-existing-to-next-shard' => \$O{move_existing_to_next_shard},
        'm|manager-for-agency=s%' => \$O{manager_for_agency},
        'replace-existing' => \$O{replace_existing},
        'new-shard=i' => \$O{new_shard},
        'h|help' => sub { system 'podselect -section SYNOPSIS -section DESCRIPTION | pod2text >&2' ; exit 0 },
    ) || die "can't parse options, stop\n";
    die "no dir given\n" unless $O{dir};
    return \%O;
}

1;
__END__
