#!/usr/bin/perl

=head1 DESCRIPTION

    Проксирует запросы, попутно сохраняя заголовки и тела всех запросов в виде отдельных файлов (conn_<PID>_\d+\.txt) в указанную директорию.
    Пример запуска:
        ./tcp-proxy.pl --listen-on beta.direct.yandex.ru:8701 --proxy-to ppctest-load-front.yandex.ru:14080 --conn-dir ./conn
        ./tcp-proxy.pl --listen-on localhost:8701 --proxy-to ppctest-load-front.yandex.ru:14080 --conn-dir ./conn
    Умирает по ^C.

=cut

use warnings;
use strict;
use utf8;
use open ':std' => ':utf8';

use IO::Socket;
use IO::Select;
use Getopt::Long;

my ($LISTEN_ON, $PROXY_TO, $CONN_DIR) = ('beta.direct.yandex.ru:8701', 'localhost:8700', './conn');
GetOptions(
    'listen-on=s' => \$LISTEN_ON,
    'proxy-to=s' => \$PROXY_TO,
    'conn-dir=s' => \$CONN_DIR,
);

my $ioset = IO::Select->new;
my (%socket_map, %conn_ids);

my ($proxy_to_host, $proxy_to_port) = split /:/, $PROXY_TO;
my ($listen_on_host, $listen_on_port) = split /:/, $LISTEN_ON;

print "Starting server on $listen_on_host:$listen_on_port and proxying requests to $proxy_to_host:$proxy_to_port\n";
my $server = new_server($listen_on_host, $listen_on_port);
$ioset->add($server);

while (1) {
    for my $socket ($ioset->can_read) {
        if (fileno $socket == fileno $server) {
            new_connection($server, $proxy_to_host, $proxy_to_port);
        }
        else {
            next unless exists $socket_map{$socket};
            my $remote = $socket_map{$socket};
            my $buffer;
            my $read = $socket->sysread($buffer, 4_096_000);
            if ($read) {
                $remote->syswrite($buffer);
                my $connid = $conn_ids{fileno $socket};
                if($connid) {
                    open F, '>>:encoding(UTF-8)', "$CONN_DIR/conn_${$}_${connid}.txt";
                    print F $buffer;
                    close F;
                }
            }
            else {
                close_connection($socket);
            }
        }
    }
}


sub new_conn {
    my ($host, $port) = @_;
    return IO::Socket::INET->new(
        PeerAddr => $host,
        PeerPort => $port
    ) || die "Unable to connect to $host:$port: $!";
}

sub new_server {
    my ($host, $port) = @_;
    my $server = IO::Socket::INET->new(
        LocalAddr => $host,
        LocalPort => $port,
        ReuseAddr => 1,
        Reuse     => 1,
        Listen    => 100
    ) || die "Unable to listen on $host:$port: $!";
}

{
my $last_conn_id = 0;
sub new_connection {
    my ($server, $host, $port) = @_;
    my $client = $server->accept;

    print "Connection accepted.\n";

    my $remote = new_conn($host, $port);
    $ioset->add($client);
    $ioset->add($remote);

    $socket_map{$client} = $remote;
    $socket_map{$remote} = $client;
    $conn_ids{fileno $client} = ++$last_conn_id;
}
}

sub close_connection {
    my ($client) = @_;
    my $remote = $socket_map{$client};

    $ioset->remove($client);
    $ioset->remove($remote);

    delete $socket_map{$client};
    delete $socket_map{$remote};
    delete $conn_ids{fileno $client};

    $client->close;
    $remote->close;

    print "Connection closed.\n";
}
