#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    mk_moderate_reasons.pl

=head1 DESCRIPTION

    Генерация файла со списком причин отклонения для танкера
    $Id$

=head1 RUNNING 

    При появлении новых причин отклонения нужно запускать данный скриптик на бете
    и закоммитить файл ./protected/data/translation/moderate_reasons.trans, который затем подхватится танкером.

    До выкладки файла в продакшен будет продолжать приходить предупреждение на расслыку direct-maintenance

    В норме надо запускать скрипт с продакшен-конфигурацией, чтобы сложить в файл продакшен-причины отклонения. 
    Если почему-то нужны непродакшен-причины, можно запускать скрипт с ключом --force

=cut


use warnings;
use strict;
use JSON;

use Settings;
use Yandex::DBTools;
use ScriptHelper;
use EnvTools;
use List::MoreUtils qw(uniq);
use Lang::Guess;

use File::Slurp;

use utf8;
use open ':std' => ':utf8';

my $OUT_FILE = "$Settings::ROOT/protected/data/translation/moderate_reasons.trans";

my ($no_file, $force);
extract_script_params(
    'no-file' => \$no_file,
    'force'   => \$force,
);

die "Чтобы подготовить причины отклонения к коммиту, нужна продакшен-конфигурация (mk prod). 
Если ты знаешь, что делаешь, и тебе нужна разработческая конфигурация: запусти скрипт с ключом --force" if !is_production() && !$force;

my $diags = get_all_sql(PPCDICT,
    "SELECT text, textFull, type
       FROM moderate_diags
     ORDER BY text, textFull"
);

my @reasons;

sub _is_suitable_for_translation {
    my ($text, $type) = @_;

    return 0 if !$text;
    return 1 if $type eq 'common';

    # Выгружаем на перевод только русские причины отклонения, так как пока не понятно, что делать с
    # мультиязычными причинами, приходящими из БК
    local $Lang::Guess::EXTERNAL_QUERYREC = sub { return {eng => 1}; };
    return Lang::Guess::analyze_text_lang($text) eq 'ru' ? 1 : 0;
}

for my $diag ( @{$diags} ) {
    push @reasons, $diag->{text} if _is_suitable_for_translation($diag->{text}, $diag->{type});
    push @reasons, $diag->{textFull} if _is_suitable_for_translation($diag->{textFull}, $diag->{type});
}

my $reasons_json = JSON->new->pretty->encode( [uniq @reasons] );

if ($no_file) {
    print $reasons_json;
} else {
    write_file($OUT_FILE, {binmode => ':utf8'}, $reasons_json);
}
