#!/usr/bin/perl

use my_inc "../..";

use Direct::Modern;
use Module::Load;
use Carp;
use Getopt::Long;
use File::Slurp qw/write_file/;
use List::MoreUtils qw/any/;
use JSON;

my ($model, $out_file);
my $getopt = GetOptions(
    "model=s" => \$model,
    "out-file=s" => \$out_file  
) or die "Getopt error: $!\n";

unless ($model && $out_file) {
    warn qq[usage: model2schema-json.pl --model=Client --out-file=client.schema.json
            --model - model name in namespace Direct::Model::%model%\n];
    exit;
}; 

my %moose_type2js = (
    "Int" => "number",
    "Id" => "number",
    "Str" => "string",
    "Num" => "number",
    "Timestamp" => "string",
    "Bool" => ["number", {enum => [0,1]}],
    "Enum" => "string",
    "null" => "null",
    "HashRef" => "object",
    "ArrayRef" => "array",
    "OnlyFalse" => ["number", {enum => [0]}],
    "OnlyTrue" => ["number", {enum => [1]}],
);

my %properties;
my %schema = (
    type => 'object',
    id => sprintf("item-%s", lc $model),
    description => '',
    properties => \%properties 
);

load "Direct::Model::$model";

my @attributes = "Direct::Model::$model"->meta->get_all_attributes();
for my $attr (@attributes) {
    
    next unless $attr->does('Yandex::ORM::Meta::Attribute::Trait::Column');
    
    my $isa;
    $isa = $attr->params->{type} if $attr->params;
    $isa = $attr->type_constraint->name if !$isa && $attr->has_type_constraint;
    
    my (@types, %type_constraint);
    if ($isa) {
        push @types, "null" if $isa =~ /^Maybe\[/;
        push @types, $isa =~ /\[([^\]]+)\]/ ? $1 : $isa;
        if ($isa =~ 'Enum' && $attr->params) {
            $type_constraint{enum} = $attr->params->{values}; 
        }
    } else {
        warn "unrecognized moose type for field @{[$attr->name]}\n";
    }

    my @js_types;        
    for (@types) {
        if ($moose_type2js{$_}) {
            if (ref $moose_type2js{$_} eq 'ARRAY') {
                push @js_types, $moose_type2js{$_}->[0];
                %type_constraint = (%type_constraint, %{$moose_type2js{$_}->[1]}); 
            } else {
                push @js_types, $moose_type2js{$_};
            }
        } else {
            warn "don't known how to convert moose type $_ to js type (field @{[$attr->name]})\n";
        }
    }
    
    if (any {$_ =~ /^(object|array)$/} @js_types) {
        warn "field @{[$attr->name]} has complex type(object or array) but we didn't explain the value\n";
    }
    
    $properties{$attr->name} = {
        type => 1 == @js_types ? $js_types[0] : \@js_types,
        %type_constraint,
        description => ""
    }
}

my $scheme_json = JSON->new->utf8(1)->pretty(1)->canonical(1)->space_before(0)->encode(\%schema);
write_file($out_file, {append => 0, binmode => ":utf8"}, $scheme_json); 
warn "schema wrote in $out_file\n";
