#!/usr/bin/perl

use my_inc '../..';

=head1 METADATA

=cut

=head2 NAME

    ppcBSReexportEverything.pl - постановка всех кампаний директа на переотправку в БК.

=head2 DESCRIPTION

    Скрипт последовательно ставит на переотправку в БК все кампании в Директе,
    начиная с той, номер которой был максимальным при первом запуске этого скрипта,
    и двигаясь в сторону минимального cid. За один запуск добавляется 10000 кампаний.
    Скрипт не ставит ничего в очередь, если в ней уже содержится больше 1000000 объектов.

    Принимает опциональные параметры:

    --max-tasks - максимальное количество объектов с используемым приоритетом в очереди
        шарда на переотправку, при достижении которого добавление кампаний временно
        не производится

    LOG_TEE=1 ./protected/maintenance/ppcBSReexportEverything.pl --shard-id 8 --max-tasks 10000

=cut

use Direct::Modern;

use Property;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use Yandex::ListUtils qw/chunks nsort/;
use Yandex::DBTools;

use BS::ResyncQueue qw/bs_resync_whole_camps/;

my $PROP_NAME = 'bs_full_reexport_max_cid';
my $MAX_TASKS_IN_QUEUE = 1_000_000;
my $CIDS_REEXPORT_CHUNK_SIZE = 10000;
my $CHUNK_SIZE = 500;

sub _log_and_exit {
    juggler_ok();
    $log->out({shard => $SHARD, message => "FINISH"});
    exit(0);
}

extract_script_params(
    'max-tasks=i' => \$MAX_TASKS_IN_QUEUE,
);

$log->out({shard => $SHARD, message => "START"});
my $max_cid_prop = Property->new("${PROP_NAME}_shard_$SHARD");
my $max_cid = $max_cid_prop->get();
if (!defined $max_cid) {
    $log->out({shard => $SHARD, message => "Creating new max_cid property"});
    $max_cid = get_one_field_sql(PPC(shard => $SHARD), "SELECT MAX(cid) FROM campaigns") // 0;
    $max_cid_prop->set($max_cid);
}
$log->out({shard => $SHARD, max_cid => $max_cid, message => "Value at script start"});

if ($max_cid <= 0) {
    _log_and_exit();
}

my $tasks_count = get_one_field_sql(PPC(shard => $SHARD),
    "select count(*) as cnt from bs_resync_queue where priority = ?",
    BS::ResyncQueue::PRIORITY_INTAPI_RESYNC_CAMPAIGNS
);
$log->out({shard => $SHARD, tasks_count => $tasks_count, message => "Number of tasks in queue"});

if ($tasks_count > $MAX_TASKS_IN_QUEUE) {
    $log->out({shard => $SHARD, message => "There is more than $MAX_TASKS_IN_QUEUE tasks in queue. Stopping script execution."});
    _log_and_exit();
}

$log->out({shard => $SHARD, message => "Getting a list of cids to put in queue"});
my $existing_cids = get_one_column_sql(PPC(shard => $SHARD),
    "SELECT cid
       FROM campaigns
      WHERE cid <= ?
     ORDER BY cid DESC
     LIMIT $CIDS_REEXPORT_CHUNK_SIZE", $max_cid
);
my $camps_num = scalar(@$existing_cids);
$log->out({shard => $SHARD, camps_num => $camps_num, message => "Number of campaigns to reexport"});
$log->out({shard => $SHARD, message => "Putting campaigns into queue"});

my $campaigns_added = 0;
my $objects_added = 0;
for my $chunk (chunks($existing_cids, $CHUNK_SIZE)) {
    $log->out({shard => $SHARD, resync_chunk => $chunk});
    my $result = bs_resync_whole_camps($chunk, priority =>
        BS::ResyncQueue::PRIORITY_INTAPI_RESYNC_CAMPAIGNS);
    $campaigns_added += $result->{cid_count} // 0;
    $objects_added += $result->{total_count} // 0;
    $log->out({shard => $SHARD, result => $result});
}

$max_cid = scalar @$existing_cids > 0 ? $existing_cids->[-1] - 1 : 0;
$log->out({shard => $SHARD, max_cid => $max_cid, message => "max_cid value to save as property at the end of script work"});
$max_cid_prop->set($max_cid);

$log->out({shard => $SHARD, message => "Finishing to add new tasks to queue"});
$log->out({shard => $SHARD, total => {cid_count => $campaigns_added, total_count => $objects_added}});

_log_and_exit();
