#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    prepare_bs_resync_file.pl

=head1 DESCRIPTION

    Формирует файл с данными для очереди переотправки в БК (bs_resync_queue), подходящий для импорта через веб-интерфейс
    https://direct.yandex.ru/internal_tools/#add_objects_to_resync_queue
    Исходные данные получает из STDIN (должен быть указан тип данных --treat-as-(cids|orderids|bannerids)), результаты выводит в STDOUT

    Совсем огромные файлы веб-интерфейс не примет, так что придётся делить на части. Верхний предел проходит где-то в районе 1,5 млн строк.
    (1 700 000 строк уже не пролезают по памяти)

    ВНИМАНИЕ!!! Скрипт ходит на чтение в продакшен (prod:ppc:*)!

    Опции командной строки:
        --help
            выводит справку по использованию скрипта

        --treat-as-cids
        --treat-as-orderids
        --treat-as-bannerids
        --treat-as-bids
            рассматривать данные из STDIN как номера кампаний (для --treat-as-cids) 
            или БКшные номера заказов OrderID (для --treat-as-orderids)
            или БКшные номера текстовых баннеров BannerID (для --treat-as-bannerids) или картинок (для --treat-as-bannerids --images)
                использовать с осторожностью, так как ходит во все шарды! (нет маппинга по BannerID)
            или директовские номера текстовы баннеров (для --treat-as-bids) или картинок (для --treat-as-bids --images)
            по одному номеру на строку
            одновременно может быть указан только один из этих параметров; пустые строки игнорируются

        --images
            искать баннеры по bid/BannerID картинки, а не текстового баннера

        --priority=<num>
            в генерируемом файле будут приоритеты <num> (по-умолчанию -3)
            0 -- нормальный, <0 -- приоритет ниже обычного [отправится позже], >0 -- приоритет выше обычного [отправится раньше]
            > 100 -- переотправится в первую очередь вне зависимости от размера очереди

        --no-sync-campaigns
        --no-sync-banners
        --no-sync-conditions
            в генерированный файл не попадут кампании, баннеры или условия показа, т.е. переотправлены они не будут
            по умолчанию переотправляются кампании/заказы целиком

    ./protected/maintenance/prepare_bs_resync_file.pl --treat-as-cids < cids.txt | split -l 1000000 - data-to-import-
    ./protected/maintenance/prepare_bs_resync_file.pl --treat-as-orderids --priority=1 < orderids.txt | split -l 1000000 - data-to-import-
    ./protected/maintenance/prepare_bs_resync_file.pl --treat-as-cids --no-sync-campaigns --no-sync-banners --no-sync-conditions < cids.txt | split -l 1000000 - data-to-import-

=cut

use strict;
use warnings;

use lib::abs '..';

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils;

use SettingsALL;
use ScriptHelper get_file_lock => undef, 'Yandex::Log' => undef, script_timer => undef;
use BS::ResyncQueue ();

use utf8;
use open ':std' => ':utf8';

=head2 $ORDER_PACKET_LENGTH

    Количество кампаний/заказов, по которым выбирать данные из БД за один раз

=cut

my $ORDER_PACKET_LENGTH = 1_000;

my $treat_as_cids = 0;
my $treat_as_orderids = 0;
my $treat_as_bannerids = 0;
my $treat_as_bids = 0;

my $sync_campaigns = 1;
my $sync_banners = 1;
my $sync_conditions = 1;

my $images = 0;

my $priority = BS::ResyncQueue::PRIORITY_MAINTENANCE_PREPARE_RESYNC_FILE;

extract_script_params(
    'treat-as-cids' => \$treat_as_cids,
    'treat-as-orderids' => \$treat_as_orderids,
    'treat-as-bannerids' => \$treat_as_bannerids,
    'treat-as-bids' => \$treat_as_bids,
    'sync-campaigns!' => \$sync_campaigns,
    'sync-banners!' => \$sync_banners,
    'sync-conditions!' => \$sync_conditions,
    'priority=i' => \$priority,
    'images!' => \$images,
);

my @treats = ($treat_as_cids, $treat_as_orderids, $treat_as_bannerids, $treat_as_bids);
die 'Должен быть обязательно указан только один из параметров: --treat-as-(cids|orderids|bannerids)' if scalar(grep {$_} @treats) != 1;
die 'Синхронизировать нечего' if !$sync_campaigns && !$sync_banners && !$sync_conditions;

# ходим в продакшеновскую метабазу
local $Yandex::DBShards::SHARD_DB = PRODUCTION_PPCDICT;
$Yandex::DBTools::DB_USER = sub { $_[0] =~ /^((sharded_)?unit_tests|ppchouse)/ ? undef : 'direct-ro' };

# cid нужен всегда, даже если кампанию не отправляем
my @fields_to_fetch = ('c.cid');
my @tables_sql = ('FROM campaigns c');

if ($sync_banners) {
    push @fields_to_fetch, 'b.bid';
}
if ($sync_conditions) {
    push @fields_to_fetch, 'p.pid';
}

if ($sync_conditions || $sync_banners || $treat_as_bannerids || $treat_as_bids) {
    push @tables_sql, 'LEFT JOIN phrases p ON c.cid = p.cid';
}
if ($sync_banners || $treat_as_bannerids || $treat_as_bids) {
    if ($images) {
        push @tables_sql, '
            LEFT JOIN banners b ON p.pid = b.pid
            LEFT JOIN banner_images bi ON b.bid = bi.bid
        ';
    } else {
        push @tables_sql, 'LEFT JOIN banners b ON p.pid = b.pid';
    }
}

my @ids = map {s/^\s+//; s/\s+$//; $_} map {split /\s*,\s*/} <STDIN>;
chomp(@ids);
for my $ids_chunk (chunks \@ids, $ORDER_PACKET_LENGTH) {
    my (%cond, %shard);
    if ($treat_as_cids) {
        $shard{cid} = $ids_chunk;
        $cond{'c.cid'} = SHARD_IDS;
    } elsif ($treat_as_orderids) {
        $shard{OrderID} = $ids_chunk;
        $cond{'c.OrderID'} = SHARD_IDS;
    } elsif ($treat_as_bannerids) {
        $shard{shard} = 'all';
        if ($images) {
            $cond{'bi.BannerID'} = $ids_chunk;
        } else {
            $cond{'b.BannerID'} = $ids_chunk;
        }
    } elsif ($treat_as_bids) {
        $shard{bid} = $ids_chunk;
        if ($images) {
            $cond{'bi.image_id'} = SHARD_IDS;
        } else {
            $cond{'b.bid'} = SHARD_IDS;
        }
    } else {
        die "don't know how to treat incoming data";
    }
    my $data = get_all_sql(PRODUCTION_PPC(%shard), ['SELECT', join(',', @fields_to_fetch), @tables_sql, WHERE => \%cond]);
    if ($data && @$data) {
        for my $row (@$data) {
            # если кампания не приджойнилась, список из баннеров и групп бесполезен
            # так как неизвестно, в какой шард его загружать. поэтому пропускаем.
            next unless $row->{cid};

            my @data_to_print;
            if ($sync_campaigns) {
                push @data_to_print, {cid => $row->{cid}, bid => 0, pid => 0};
            }
            if ($sync_banners && $row->{bid}) {
                push @data_to_print, {cid => $row->{cid}, bid => $row->{bid}, pid => 0};
            }
            if ($sync_conditions && $row->{pid}) {
                push @data_to_print, {cid => $row->{cid}, bid => 0, pid => $row->{pid}};
            }
            if (@data_to_print > 0) {
                print join "\n", map { join "\t", ($_->{cid}, $_->{bid}, $_->{pid}, $priority) } @data_to_print;
                print "\n";
            }
        }
    }
}
