#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    prepare_mod_resync_file.pl

=head1 DESCRIPTION

    Формирует файл с данными для очереди переотправки в Модерацию (mod_resync_queue), подходящий для импорта через веб-интерфейс
    https://direct.yandex.ru/registered/main.pl?cmd=importIntoModerationResyncQueue
    Исходные данные получает из STDIN (должен быть указан тип данных --treat-as-(cids|bids)), результаты выводит в STDOUT

    Совсем огромные файлы веб-интерфейс не примет, так что придётся делить на части. Верхний предел проходит где-то в районе 2 млн строк.

    ВНИМАНИЕ!!! Скрипт по умолчанию ходит на чтение в продакшен (конфигурация ROProd)!

    Опции командной строки:
        --help
            выводит справку по использованию скрипта

        --treat-as-cids
        --treat-as-bids
            рассматривать данные из STDIN как номера кампаний (для --treat-as-cids)
            или номера баннеров bid (для --treat-as-bids)
            по одному номеру на строку
            одновременно может быть указан только один из этих параметров; пустые строки игнорируются

        --priority=<num>
            в генерируемом файле будут приоритеты <num> (-128..127, по-умолчанию 0)
            0 -- нормальный, <0 -- приоритет ниже обычного [отправится позже], >0 -- приоритет выше обычного [отправится раньше]
            > 100 -- переотправится в первую очередь вне зависимости от размера очереди

        --remoderate=<0|1>
            флаг принудительной ПРЕмодерации (по-умолчанию 0)
            0 - обрабатывать объекты автомодератором
            1 - ручная модерация

    ./protected/maintenance/prepare_mod_resync_file.pl --treat-as-cids < cids.txt | split -l 2000000 - data-to-import-
    ./protected/maintenance/prepare_mod_resync_file.pl --treat-as-bids --priority 101 --remoderate 1 < bids.txt | split -l 2000000 - data-to-import-
    ./protected/maintenance/prepare_mod_resync_file.pl --treat-as-cids --priority 101 < cids.txt | split -l 2000000 - data-to-import-

=cut

use strict;
use warnings;

BEGIN {
    # используем продакшен-базу
    $Settings::NO_SETTINGS_LOCAL = 1;
}

use Yandex::ListUtils;
use Yandex::Validate qw/is_valid_id/;

use Settings;
use ScriptHelper get_file_lock => undef, 'Yandex::Log' => undef, script_timer => undef;
use Moderate::ResyncQueue;

use utf8;

=head2 $ORDER_PACKET_LENGTH

    Количество кампаний/баннеров, по которым выбирать данные из БД за один раз

=cut

my $ORDER_PACKET_LENGTH = 1_000;

my $treat_as_cids = 0;
my $treat_as_bids = 0;

my $priority = 0;
my $remoderate = 0;

extract_script_params(
    'treat-as-cids' => \$treat_as_cids,
    'treat-as-bids' => \$treat_as_bids,
    'priority=i' => \$priority,
    'remoderate=i' => \$remoderate,
);

my @treats = ($treat_as_cids, $treat_as_bids);
die 'Должен быть обязательно указан только один из параметров: --treat-as-(cids|bids)' if scalar(grep {$_} @treats) != 1;
die 'Параметр priority должен быть целым числом в диапазоне -128..127' unless Moderate::ResyncQueue::is_valid_priority($priority);
die 'Параметр remoderate должен быть 0 или 1' unless Moderate::ResyncQueue::is_valid_remoderate($remoderate);

my @ids = grep { is_valid_id($_) } map { chomp($_); $_ } map { split qr/\s*,\s*/ } <STDIN>;
$Yandex::DBTools::DB_USER = sub { $_[0] =~ /^((sharded_)?unit_tests|ppchouse)/ ? undef : 'direct-ro' };

for my $ids_chunk (chunks \@ids, $ORDER_PACKET_LENGTH) {
    my $key;
    if ($treat_as_cids) {
        $key = 'cid';
    } elsif ($treat_as_bids) {
        $key = 'bid';
    } else {
        die "don't know how to treat incoming data";
    }

    my $data = Moderate::ResyncQueue::get_all_objects_by($key => $ids_chunk, priority => $priority, remoderate => $remoderate);

    if ($data && @$data) {
        for my $row (@$data) {
            my $extra = {};
            if (defined $row->{cid}) {
                $extra->{key} = 'cid';
                $extra->{value} = $row->{cid};
            } elsif (defined $row->{client_id}) {
                $extra->{key} = 'client_id';
                $extra->{value} = $row->{client_id};
            }
            print join "\n", join ("\t", ($row->{type}, $row->{id}, $row->{priority}, $row->{remoderate}, (%$extra ? ($extra->{key}, $extra->{value}) : ())));
            print "\n";
        }
    }
}
