#!/usr/bin/env perl

# $Id$

use my_inc "../..";

=pod

=encoding utf-8

=head1 NAME

release_workflow_monitor - отслеживание порядка принятия релизных тикетов

=head1 DESCRIPTION

    release_workflow_monitor.pl
    Смотрит последние 5 или N закрытых релизных тикетов и проверяет, что основные переходы выполнены разными людьми.
    Заводит тикет, если это не так и про это ещё не было тикета.

=head1 METADATA

<crontab>
    time: */30 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 250
        </leases>
    </switchman>
    package: scripts-switchman
    env: PROJECT_SPECIFIC=Direct
    params: --startrek-token-file=/etc/direct-tokens/startrek --only-environment=production --queue=DIRECTALERTS
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.release_workflow_monitor.working
    raw_events:     scripts.release_workflow_monitor.working
    ttl: 20m
    tag: direct_external_audit
    tag: direct_group_sre
</juggler>

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use List::AllUtils qw/any firstval uniq/;
use Path::Tiny;
use ScriptHelper get_file_lock => ['dont die'];
use Startrek::Client::Easy;
use YAML;

$Startrek::Client::Easy::USE_DEFAULT_ASSIGNEE = 0;

my $RECENT_RELEASES_CNT = 10;
my $ONLY_ENVIRONMENT = '';
my $TEST_STARTREK = 0;
my $ST_QUEUE_FOR_REPORTS = '';
my $STARTREK_TOKEN_FILE = '';

extract_script_params(
    'n|N=i' => \$RECENT_RELEASES_CNT,
    'e|only-environment=s' => \$ONLY_ENVIRONMENT,
    'test-startrek' => \$TEST_STARTREK,
    'queue=s' => \$ST_QUEUE_FOR_REPORTS,
    't|startrek-token-file=s' => \$STARTREK_TOKEN_FILE,
);

die 'empty startrek queue for reports' unless $ST_QUEUE_FOR_REPORTS;

if ($ONLY_ENVIRONMENT) {
    my $environment = path('/etc/yandex/environment.type')->slurp;
    chomp $environment;
    exit(0) unless ($environment eq $ONLY_ENVIRONMENT);
}

my %startrek_params;
if ($STARTREK_TOKEN_FILE) {
    $startrek_params{token} = Startrek::Client::Easy::get_startrek_client_token(file => $STARTREK_TOKEN_FILE);
}
if ($TEST_STARTREK) {
    $startrek_params{startrek} = 'test';
}

my $startrek = Startrek::Client::Easy->new(%startrek_params);

my $issues = $startrek->get(
    query => qq/Queue: DIRECT AND Type: "Release" AND (Status: Closed OR Status: "Ready to Deploy") AND Created: >= today() - 7d "Sort By": Key DESC/,
    perPage => $RECENT_RELEASES_CNT,
    array => 1
);

foreach my $issue (@$issues) {
    my $alert_issue = $startrek->get(
        query => qq/Queue: $ST_QUEUE_FOR_REPORTS AND Summary: "Release $issue->{key} workflow violation" "Sort By": Key DESC/,
        perPage => 1,
        array => 1
    );

    if (@$alert_issue) {
        $log->out("There is already an issue for $issue->{key}: $alert_issue->[0]->{key}");
        next;
    }
    my @key_states = ('testing', 'rmAcceptance', 'readyToDeploy');
    my @key_steps = map {$key_states[$_] . ' — ' . $key_states[$_ + 1]} (0 .. 2);
    my %key_changes = map {$_ => undef} @key_steps;
    my $last_id = 0;
    my ($workflow, $flows_chunk);
    for (1 .. 20) {
        if ($last_id) {
            $flows_chunk = $startrek->request('GET', "/issues/$issue->{key}/changelog?type=IssueWorkflow&id=$last_id&perPage=100");
        } else {
            $flows_chunk = $startrek->request('GET', "/issues/$issue->{key}/changelog?type=IssueWorkflow&perPage=100");
        }
        last unless @$flows_chunk;
        $last_id = $flows_chunk->[(scalar @$flows_chunk) - 1]->{id};
        push @$workflow, @$flows_chunk;
    }

    foreach my $flow (@$workflow) {
        unless (exists $flow->{fields} && any {$_->{field}->{id} eq 'status'} @{$flow->{fields}}) {
            next;
        }
        my $field_change = firstval {$_->{field}->{id} eq 'status'} @{$flow->{fields}};
        foreach my $i (0 .. 2) {
            if ($field_change->{from}->{key} eq $key_states[$i]
                && $field_change->{to}->{key} eq $key_states[$i + 1]
                && !defined $key_changes{$key_steps[$i]}) {
                $key_changes{$key_steps[$i]} = $flow->{updatedBy}->{id};
            }
        }
    }

    if ((scalar uniq grep {$_} values %key_changes) < 2) {
        my $description = "The following are the monitored steps and the people who made them in $issue->{key}:\r\n%%" . YAML::Dump(\%key_changes) . '%%';
        my $new_issue_key = $startrek->do(
            create => 1,
            queue => $ST_QUEUE_FOR_REPORTS,
            type => "bug",
            summary => "Release $issue->{key} workflow violation",
            description => $description,
        );
        $log->out("Created an issue $new_issue_key");
    }
}

juggler_ok();
