#!/usr/bin/perl

=pod

=encoding utf8

    $Id:$

=head1 NAME

    sandbox_service.pl

=head1 SYNOPSIS

    ./sandbox_service.pl create --uid 13456 --type client --currency 'RUB' --login test --email test@ya.ru --fio 'Ivan Ivanovich' --init --shared_account
    ./sandbox_service.pl check --uid 13456
    ./sandbox_service.pl drop --uid 13456
    ./sandbox_service.pl drop_fin_counter --uid 13456

    ./sandbox_service.pl create --uid 447395407 --type agency --currency 'RUB' --login agency-clients-test --email agency-clients-test@yandex.ru --fio 'Direct QA Agency'

    ./sandbox_service.pl save_app_access --data '{..}'

=head1 DESCRIPTION

    Консольная обертка для Sandbox::SandboxService. Служит для упрощения
    диагностики проблем с песочницей

=head1 METHODS

=cut

use Carp::Always;
use Data::Dumper;
use Getopt::Long;

use Direct::Modern;

use JSON;

use my_inc '../../';

use Sandbox::SandboxService;

$Data::Dumper::Deparse = 1;

$Yandex::DBShards::STRICT_SHARD_DBNAMES = 0;

my ($data, $uid, $enable_shared_account, $init_test_data, $help) = @_;

my $cmd = shift(@ARGV) or exit_with_error("no command specified");

my $login = 'testuser';
my $email = "$login\@yandex.ru";
my $fio = "Test User from script";
my $currency = 'RUB';
my $type = 'client';
Getopt::Long::GetOptions(
    'email=s' => \$email,
    'fio=s' => \$fio,
    'login=s' => \$login,
    'type=s' => \$type,
    'data=s' => \$data,
    'uid=s' => \$uid,
    'currency=s' => \$currency,
    'shared_account' => \$enable_shared_account,
    'init' => \$init_test_data,
    'help' => \$help
);

if ($help) {
    print_usage();
    exit(0);
}


my $result;
if($cmd eq 'create') {
    exit_with_error("type, uid and currecy must be specified for user creation") unless $uid && $type && $currency;

    # {'currency' => 'EUR','email' => 'hmepas@yandex.ru','fio' => "\x{416}\x{435}\x{43d}\x{435}\x{43a} \x{413}\x{43e}\x{440}\x{435}\x{43b}\x{43e}\x{432}",'init_test_data' => 1,'login' => 'hmepas','method' => 'CreateSandboxUser','type' => 'agency','uid' => 17804346 }
    # 'currency' => 'RUB', 'email' => 'testuser@yandex.ru', 'enable_shared_account' => undef, 'init_test_data' => 1, 'login' => 'testuser', 'type' => 'agency', 'uid' => 17804346
    # 'currency' => 'RUB', 'email' => 'testuser@yandex.ru', 'init_test_data' => 1, 'login' => 'testuser', 'type' => 'agency', 'uid' => 17804346
    $result = Sandbox::SandboxService->create_user({
        type => $type,
        login => $login,
        fio => $fio,
        uid => $uid,
        email => "$login\@yandex.ru",
        currency => $currency,
        ( defined $enable_shared_account ? (enable_shared_account => $enable_shared_account) : () ),
        init_test_data => $init_test_data // 0
    });
} elsif ($cmd eq 'check') {
    exit_with_error("no --uid specified") unless $uid;
    $result = Sandbox::SandboxService->get_current_state({ uid => $uid });
} elsif ($cmd eq 'drop') {
    exit_with_error("no --uid specified") unless $uid;
    $result = Sandbox::SandboxService->drop_user({ uid => $uid });
} elsif ($cmd eq 'drop_fin_counter') {
    exit_with_error("no --uid specified") unless $uid;
    $result = Sandbox::SandboxService->drop_fin_ops_counter({ uid => $uid });
} elsif ($cmd eq 'save_app_access') {
    exit_with_error("no --data") unless $data;
    $result = Sandbox::SandboxService->save_app_access({ data => from_json($data) });
} else {
    exit_with_error("command '$cmd` not recognized");
}

print Dumper($result);

sub exit_with_error {
    my ($err) = @_;
    print_usage($err);
    exit(1);
}

sub print_usage {
    my ($msg) = @_;
    say $msg if $msg;
    system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
}

1;
