#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    tt_deps.pl

=head1 SYNOPSIS

    tt_deps.pl -r OLD_REV:NEW_REV file
    tt_deps.pl -r OLD_REV file
    tt_deps.pl -r OLD_REV file -show-deps

=head1 DESCRIPTION

    Вычисляем зависимости шаблона file для ревизии OLD_REV
    и NEW_REV (если не указывать, будет взят HEAD).

    Для сохранившихся зависимостей выводим результат svn diff
    Появившиеся и пропавшие зависимости выводим в виде списка,
    если появившаяся зависимость была создана позже OLD_REF,
    то указываем ревизию, в которой эта зависимость была создана
    (это не значит, что в указанный момент времени file уже включал
    данную зависимость)

    Ключ -show-deps добавляет вывод списков зависимостей для старой
    и новой ревизий

    Ключ -only-blocks выводит только разницу в блоках

=cut

use strict;
use warnings;

use Cwd qw/abs_path/;
use File::Slurp;
use File::Temp qw/tempdir/;
use FindBin qw/$Bin/;
use List::MoreUtils qw/uniq/;
use List::Util qw/first/;
use XML::LibXML;
use YAML;
use Yandex::ListUtils qw/xdiff xisect/;
use Yandex::Shell qw/yash_qx/;

use Settings;
use ScriptHelper get_file_lock => undef, script_timer => undef;

use utf8;

my $BUNDLE_FILE = "$Bin/../../data/pages/direct/direct.js";
my $ONLY_BLOCKS = 0;
my $SVN_PREFIX = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/direct/perl/data';
my $TEMPDIR = tempdir(CLEANUP => 1);
$|++;
run() unless caller();


sub run
{
    my $diff = '';
    my $show_deps = 0;
    Getopt::Long::GetOptions(
        'r|diff=s' => \$diff,
        'show-deps' => \$show_deps,
        'svn=s' => \$SVN_PREFIX,
        'only-blocks' => \$ONLY_BLOCKS,
    );
    die "specify revisions" unless $diff;

    my $url = path2url($ARGV[0]);
    my ($old_rev, $new_rev) = split /:/, $diff;
    unless ($new_rev) {
        my $svn_info = yash_qx('svn', 'info', $SVN_PREFIX);
        ($new_rev) = $svn_info =~ /Revision: (\d+)/;
    }

    open STDERR, '>', '/dev/null';
    for ($old_rev, $new_rev) {
        next if -d "$TEMPDIR/$_";
        system('svn', 'co', $SVN_PREFIX, '-r', $_, "$TEMPDIR/$_", '--ignore-externals', '-q');
    }

    my $old_deps = get_deps($url, $old_rev);
    my $new_deps = get_deps($url, $new_rev);
    if ($show_deps) {
        print "old deps:\n";
        print Dump $old_deps;
        print "\n";
        print "new deps:\n";
        print Dump $new_deps;
        print "\n";
    }

    my $remained = xisect($old_deps, $new_deps);
    my $included = xdiff($new_deps, $remained);
    my $excluded = xdiff($old_deps, $remained);

    for my $url (@$included) {
        my $log = yash_qx('svn', 'log', $url, '--xml');
        my $root = XML::LibXML->load_xml(string => $log)->documentElement();
        my @entries = $root->findnodes('logentry');
        my $first = pop @entries;
        my $revision = $first->getAttribute('revision');
        my $created = '';
        if ($revision > $old_rev) {
            $created = " (created at $revision)";
        }
        my $short_url = substr($url, length($SVN_PREFIX) + 1);
        print(('=' x 67)."\n"."included $short_url$created\n\n");
    }
    for my $url (@$excluded) {
        my $short_url = substr($url, length($SVN_PREFIX) + 1);
        print(('=' x 67)."\n"."excluded $short_url\n\n");
    }
    for my $url (@$remained) {
        my $cmd = join ' ', ('diff', '-u', '-r', "$TEMPDIR/$old_rev/".url2path($url), "$TEMPDIR/$new_rev/".url2path($url));
        if (my $diff = `$cmd`) {
            print(('=' x 67)."\n".$diff."\n\n");
        }
    }
}


sub get_deps
{
    my $url = shift;
    my $rev = shift;

    my %seen = ();
    my %blocks_seen = ();
    my @queue = ($url);
    while (my $item = shift @queue) {
        $seen{$item} = 1;
        my $content = get_content($item, $rev);
        my $includes = includes($content);
        $blocks_seen{path2url("$Settings::ROOT/data/block/$_")} = 1 for @{blocks_includes($content)};
        my @result = ();
        for my $file (@$includes) {
            my $url = first {$_ && exists_in_repo($_, $rev)} map {path2url("$_/$file")} @$Settings::TT_INCLUDE_PATH;
            if ($url) {
                push @result, $url;
            }
        }
        push @queue, grep {!$seen{$_}} @result;
    }
    my @js_urls = grep {exists_in_repo($_, $rev)} map {s/tt2$/js/; $_} grep {m!data/block!} keys %seen;
    return $ONLY_BLOCKS ? [sort keys %blocks_seen] : [sort(@js_urls, keys %seen, keys %blocks_seen)];
}


sub includes
{
    my $content = shift;

    my @result = $content =~ /(?:\s)(?:INSERT|PROCESS|INCLUDE|WRAPPER)(?:[\s\n]+)([^\s%;,~]+)/g;
    foreach (map {split /\n/, $_} ($content =~ /(?:PROCESS)([^%;~]+)/sg)) {
        next unless /\+/;
        s/\s+|\+//g;
        push @result, $_ if $_;
    }
    for (@result) {
        s!^"(.*)"$!$1!;
        s!^'(.*)'$!$1!;
    }
    return [uniq @result];
}


sub blocks_includes
{
    my $content = shift;

    my $blocks_re = get_blocks_re();
    my @result = $content =~ /($blocks_re)/g;
    return [uniq @result];
}


sub get_blocks
{
    my $file = shift;
    my $content = read_file($file, binmode => ':utf8');
    my @blocks = $content =~ m{block/([^/]+)}g;
    return @blocks;
}


sub get_blocks_re
{
    return join "|", sort map {quotemeta($_)} get_blocks($BUNDLE_FILE);
}


sub path2url
{
    my $path = shift or die "no path provided";

    my $abs = abs_path($path);
    return undef unless $abs;

    my $rel = substr($abs, length("$Settings::ROOT/data/"));
    return "$SVN_PREFIX/$rel";
}


sub url2path
{
    my $url = shift or die "no url provided";

    my $rel = substr($url, length($SVN_PREFIX) + 1);
    return $rel;
}


sub get_content($$)
{
    my $url = shift;
    my $rev = shift;

    return scalar read_file("$TEMPDIR/$rev/".url2path($url), binmode => ':utf8');
}


sub exists_in_repo($$)
{
    my $url = shift;
    my $rev = shift;

    return -f "$TEMPDIR/$rev/".url2path($url);
}
