#!/usr/bin/perl

use my_inc "../..";


# $Id$

=head1 DESCRIPTION

    xiget -- замена xgettext, работаем с perl-кодом и tt-шаблонами
    Мотивация: xgettext не может указать контекст так, как нам хочется

    Ищем вызовы функции iget и стараемся максимально информативно сообщить о том, где эта фраза будет использована

    Собираем содержимое pot-файла и отдаем его в STDOUT

=head1 SYNOPSIS

    Обычное использование:
        protected/maintenance/xiget.pl > locale/direct.pot
    Просмотр предупреждений:
        protected/maintenance/xiget.pl --warn > /dev/null

    Можно передать опции:
        --help
            показать справку и завершиться
        --warn  
            выводить в STDERR разные подозрительные использования iget
            предупреждения будут (сейчас их 32), это нормально
        --use-cache
            использовать локальный кэш для фраз, найденный в perl-коде

=head1 COMMENTS
    
=cut

use strict;
use warnings;

use PPI;
use File::Slurp;
use List::Util qw/first/;
use List::MoreUtils qw/all any/;
use Digest::MD5 qw/md5_hex/;
use JSON;
use Encode;
use Parallel::ForkManager;

use Yandex::ListUtils;
use Yandex::Svn qw/svn_files/;
use Settings;
use ScriptHelper get_file_lock => undef, script_timer => undef;
use Yandex::I18n;
use Yandex::I18nTools;

use FilterSchema;

use utf8;
use open ':std' => ':utf8';

my $JSON = JSON->new->utf8;
my $WARN;
my $USE_CACHE;
my $PARALLEL = 8;
extract_script_params(
    'warn'      => \$WARN,
    'use-cache' => \$USE_CACHE,
    'parallel'  => \$PARALLEL,
);

my $PERL_RE   = qr/\w+\.(?:pm|pl)/;
my $TT_RE     = qr/\w+\.(?:html|tpl|tt2|tex|xjst|bemhtml)/;
my $TRANS_RE  = qr/\w+\.(?:trans)/;
my $JSCHM_RE  = qr/\w+\.json/;
my $ROOT      = $Settings::ROOT;
my $ROOT_LEGO = "$ROOT/data/lego/blocks";
my $ROOT_JSCHM = "$ROOT/data/json_schemas";
my @SVN_FILES = grep {-f} svn_files($ROOT);

my $PPI_CACHE = {};

# добавляем обработку файлов из data/lego/blocks
push @SVN_FILES, grep {m/\.tt2$/ && -f} svn_files($ROOT_LEGO) if -e $ROOT_LEGO && -r $ROOT_LEGO;

my $CORE_THRESHOLD = 2;

my @grep_iget_files = `grep -Rl --exclude-dir .svn iget $ROOT 2>/dev/null`;
push @grep_iget_files, `find $ROOT/protected/data/translation -name *.trans`;
push @grep_iget_files, `find $ROOT_JSCHM -name '*.json'`;
chomp for @grep_iget_files;

my $msgids = parallel_format_xiget(xisect(\@SVN_FILES, \@grep_iget_files)); 

print Yandex::I18nTools::get_po_header();
print join "\n", map { format_message($_) } group_by_msgid($msgids);

exit(0);


=head2 parallel_format_xiget

    в несколько потоков запустить над переданными файлами fomat_msgid и xiget

=cut
sub parallel_format_xiget {
    my ($files) = @_;
    my $pm = Parallel::ForkManager->new($PARALLEL);

    my ($failed, @msgids);
    $pm->run_on_finish(sub {
        my ($pid, $exit_code, $ident, $exit_signal, $core_dumped, $data) = @_;
        if ($exit_code) {
            $failed++;
        } else {
            push @msgids, @$data;
        }
    });

    for my $files_chunk (chunks $files, 10) {
        next if $failed;
        my $pid = $pm->start and next;
        my @chunk_msgids;
        for my $file (@$files_chunk) {
            push @chunk_msgids, map {format_msgid($_) } xiget($file);
        }
        $pm->finish(0, \@chunk_msgids);
    }
    $pm->wait_all_children;
    die "Fails in childs" if $failed;

    return \@msgids;
}



=head2 format_msgid

    Обработка msgid (экранирование, кодировка)

=cut

sub format_msgid
{
    my $message = shift;

    # экранируем неэкранированные двойные кавычки
    $message->{msgid} =~ s/(?<!\\)"/\\"/g;
    # убираем экранирование для одинарных кавычек
    $message->{msgid} =~ s/(?<!\\)\\'/'/g;

    if (!utf8::is_utf8($message->{msgid})) {
        utf8::decode($message->{msgid});
    }

    return $message;
}


=head2 group_by_msgid
    
    Аггрегируем location по msgid, это лучше, чем msguniq, т.к. убирает дубликаты в хинтах
    и оставляет больше хинтов, чем msguniq --use-first

=cut

sub group_by_msgid
{
    my ($messages) = @_;
    my %groups_by_msgid = ();
    foreach my $message (@$messages) {
        $groups_by_msgid{$message->{msgid}}->{location} ||= [];
        push @{ $groups_by_msgid{$message->{msgid}}->{location} }, @{ $message->{location} };
    }
    return map { {msgid => $_, location => [ xuniq {$_->{description} || $_->{controller} || $_->{core} || $_->{API} || $_->{filename}} @{ $groups_by_msgid{$_}->{location} } ]} } keys %groups_by_msgid;
}


=head2 format_message

    Отдаем переведенное сообщение, готовое для вставки в pot-файл

=cut

sub format_message
{
    my $message = shift;

    my @result = ();
    push @result, map { qq[#. $_] } _format_location($message->{location});

    my @splitted_mes = split "\n", $message->{msgid};

    push @result, qq/msgid "$splitted_mes[0]"/;
    push @result, qq[msgstr ""];
    push @result, qq[];
    
    return join "\n", @result;
}


=head2 _format_location

    Здесь собираем хинты

=cut

sub _format_location
{
    my $location = shift;

    return () if ref $location ne 'ARRAY';

    foreach (@$location) {
        if ($_->{description} && !utf8::is_utf8($_->{description})) {
            utf8::decode($_->{description});
        }
    }

    my @lines = ();

    if (any {$_->{translation}} @$location) {
        push @lines, 'translation: ' . join $Yandex::I18nTools::HINTS_SEPARATOR, sort map {$_->{filename} || ()} @$location;
    } elsif (@$location >= $CORE_THRESHOLD) {
        if (any {$_->{API}} @$location) {
            push @lines, 'API: core';
        } else {
            push @lines, 'common: core';
        }
    } elsif (all {$_->{template}} @$location) {
        push @lines, 'templates: ' . join $Yandex::I18nTools::HINTS_SEPARATOR, sort map {$_->{filename}} @$location;
    } else {
        my $controller_part = join $Yandex::I18nTools::HINTS_SEPARATOR, sort map {$_->{controller} . ($_->{description} ? '('.$_->{description}.')' : '')} grep {$_->{controller}} @$location;
        push @lines, 'controller: ' . $controller_part if $controller_part;

        my $core_part = join $Yandex::I18nTools::HINTS_SEPARATOR, sort map {($_->{core}) . ($_->{description} ? '('.$_->{description}.')' : '')} grep {$_->{core}} @$location;
        push @lines, 'core: ' . $core_part if $core_part;

        my $core_part_api = join $Yandex::I18nTools::HINTS_SEPARATOR, sort map {($_->{API}) . ($_->{description} ? '('.$_->{description}.')' : '')} grep {$_->{API}} @$location;
        push @lines, 'API: ' . $core_part_api if $core_part_api;
    }

    return @lines;
}


=head2 xiget
    
    Интерфейсная функция, которая вызывает нужную backend-функцию: xiget_perl, xiget_tt

=cut

sub xiget
{
    my $full_filename = shift;
    my $short_filename = _get_short_template_filename($full_filename);
    if ($full_filename =~ /$PERL_RE/ ) {
        return xiget_perl($full_filename, $short_filename);
    } elsif ($full_filename =~ /$TT_RE/) {
        return xiget_tt($full_filename, $short_filename);
    } elsif ($full_filename =~ /$TRANS_RE/) {
        return xiget_trans($full_filename, $short_filename);
    } elsif ($full_filename =~ /$ROOT_JSCHM.*$JSCHM_RE/) {
        return xiget_json_schema($full_filename, $short_filename);
    }
    return ();
}


=head2 xiget_perl
    
    Ищем строки для перевода в исходном коде на perl

=cut

sub xiget_perl 
{
    my ($filename, $short_filename) = @_;

    my $cached = _get_xiget_perl_cached($filename);

    if (!$cached) {
        $cached = [];
        my $doc = _get_ppi_doc($filename);
        my $iget_tokens = $doc->find( sub { $_[1]->isa('PPI::Token::Word') && $_[1]->content() =~ m/^(?:iget|iget_noop|piget|piget_array)$/ } );

        if (ref $iget_tokens eq 'ARRAY') {
            my $package_statements = $doc->find( sub { $_[1]->isa('PPI::Statement::Package') } );
            foreach my $iget_token (@$iget_tokens) {
                if ($iget_token->content() eq 'iget_noop') {
                    # обрабатываем все аргументы
                    my $sibling = $iget_token->snext_sibling();
                    if ($sibling->isa('PPI::Structure::List')) {
                        for my $string_token ($sibling->schild()->schildren()) {
                            my $string = $string_token->content();
                            next if $string eq ',';
                            push @$cached, {msgid => _get_msgid_from_token($string), location => [_get_location_in_perl($iget_token, $package_statements)]};
                        }
                    }
                } else {
                    my $sibling = $iget_token->snext_sibling();
                    my $string_token;
                    if ($sibling->isa('PPI::Structure::List')) {
                        my $first_inner_token = $sibling->schild(0)->schild(0);
                        if ($first_inner_token->isa('PPI::Token::Quote')) {
                            $string_token = $first_inner_token;
                        } else {
                            # !!! TODO писать такие бяки в лог или включить в тест на iget
                            # это не всегда ошибка
                            warn $filename . ':' . $iget_token->location()->[0] . "\t" . $sibling->schild()->content() . "\n" if $WARN;
                            next;
                        }
                    } elsif ($sibling->isa('PPI::Token::Quote')) {
                        # если написали iget без скобочек: iget "ля-ля-ля";
                        $string_token = $sibling;
                    } else {
                        # !!! TODO писать такие бяки в лог или включить в тест на iget
                        # это не всегда ошибка
                        warn $filename . ':' . $iget_token->location()->[0] . "\t" . "something suspicious\n" if $WARN;
                        next;
                    }
                    push @$cached, {msgid => _get_msgid_from_token($string_token->content()), location => [_get_location_in_perl($iget_token, $package_statements)]};
                }
            }
        }
        _set_xiget_perl_cached($filename, $cached);
    }
    return @$cached;
}


=head2 _get_xiget_perl_cached
=cut

sub _get_xiget_perl_cached
{
    my $filename = shift;

    return unless $USE_CACHE;

    my $hash = md5_hex(Encode::encode_utf8(read_file($filename, atomic => 1, binmode => ':utf8')));
    my $cache_filename = $ROOT.'/locale/.cache/'.$hash;
    if (-f $cache_filename) {
        my $cache = read_file($cache_filename, atomic => 1, binmode => ':utf8');
        return $JSON->decode($cache);
    }
    return;
}


=head2 _set_xiget_perl_cached
=cut

sub _set_xiget_perl_cached
{
    my $filename = shift;
    my $cache = shift;

    return unless $USE_CACHE;

    mkdir $ROOT.'/locale/.cache' unless -d $ROOT.'/locale/.cache';
    my $hash = md5_hex(Encode::encode_utf8(read_file($filename, atomic => 1, binmode => ':utf8')));
    my $cache_filename = $ROOT.'/locale/.cache/'.$hash;
    my $dump = $JSON->encode($cache);
    write_file($cache_filename, {atomic => 1, binmode => ':utf8'}, \$dump);
}


=head2 xiget_tt
    
    Ищем строки для перевода в tt-шаблонах

=cut

sub xiget_tt
{
    my ($filename, $short_filename) = @_;
    my @messages = ();

    my $text = read_file($filename);

    my $RE_squoted_str = qr/'(.+?(?<!\\))'/;
    my $RE_dquoted_str = qr/"(.+?(?<!\\))"/;
    my $RE_quoted_str = qr/$RE_squoted_str|$RE_dquoted_str/;
    my $RE_iget_str = qr/p?iget(?:_array|_noop)?\s*\(\s*$RE_quoted_str/;
    while ($text =~ /$RE_iget_str/msgo) {
        my $message = $1 || $2;
        push @messages, {msgid => $message, location => [{template => 1, filename => $short_filename}]};
    }

    return @messages;
}

=head2 xiget_trans

    Ищем строки для перевода в специально созданных для этого файлах, сериализованных json

=cut

sub xiget_trans
{
    my ($filename, $short_filename) = @_;
    my @messages = ();

    my $text = read_file($filename);

    ($short_filename) = $filename =~ m/translation\/(.+)\.trans/;

    foreach (@{$JSON->decode($text)}){
        $_ =~ s/\n/\\n/g;
        push @messages, {msgid => $_, location => [{filename=>$short_filename, translation=>1}]};
    }
    return @messages;
}

=head2 xiget_json_schema

    Ищем строки для перевода в схеме фильтров как значения параметров error и errorOnWrongType

=cut

sub xiget_json_schema
{
    my ($filename, $short_filename) = @_;

    my $iget_nodes = $FilterSchema::IGET_NODES;
    my @messages = ();

    eval {
        FilterSchema->new(file => $filename)->walk_by_error_texts(
            'regular',
            sub {
                my ($err_ref) = @_;
                if (defined $$err_ref) {
                    my $v = $$err_ref =~ s/\n/\\n/gr;
                    push @messages, {msgid => $v, location => [{filename=>$short_filename, translation=>1}]};
                }
            }
        );
    1} || warn $@;

    return @messages;
}


=head2 _get_short_template_filename

    Вырезаем из имени файла первую часть пути до корня рабочей копии; 
    кроме того, выкидываем data/t/, dataN и другие типичные префиксы

=cut

sub _get_short_template_filename
{
    my $filename = shift;
    $filename =~ s#$ROOT/##;
    $filename =~ s#^data[0-9]*/(?:t/)?##;
    return $filename;
}


=head2 _get_msgid_from_token
    
    Убираем наружные кавычки 

=cut

sub _get_msgid_from_token
{
    my $token = shift;
    $token =~ s/^(['"])(.*)\1$/$2/;
    return $token;
}


=head2 _get_location_in_perl

    Для вызовов из контроллеров возвращаем controller: Cmd(Description)
    Для вызовов вне контроллеров возвращаем core: ИмяМодуля::ИмяФункции 

=cut

sub _get_location_in_perl
{
    my ($token, $package_statements) = @_;

    my $location = {};
    my $iget_line_number = $token->location()->[0];
    my $package_name = _get_package_name_by_line_number($iget_line_number, $package_statements);

    my $parent= $token->parent();
    while ($parent && !_is_named_sub($parent)) {
        $parent = $parent->parent();
    }

    my $loc_name = $package_name !~ /API/ ? 'core' : 'API';

    if (my $sub_statement = $parent) {
        my $sub_name = $sub_statement->schild(1)->content();
        my @attributes = grep {$_->isa('PPI::Token::Attribute')} $sub_statement->schildren();
        if (my $description_attribute = first {/^Description/} @attributes) {
            ($location->{description}) = $description_attribute =~ /Description\((.+)\)/;
            $location->{description} =~ s/['"]//g;
        }
        
        my $cmd_attribute = first {/^Cmd/} @attributes;
        if ($cmd_attribute && $package_name !~ /API/) {
            ($location->{controller}) = $cmd_attribute =~ /Cmd\((.+)\)/;
        } else {
            # просто некоторая функция
            $location->{$loc_name} = $package_name . '::' . $sub_name;
        }
    } else {
        # просто некоторая функция
        $location->{$loc_name} = $package_name;
    }
    return $location;
}


=head2 _is_named_sub

    Определяем, является ли элемент узлом именованной (не анонимной) функции

=cut

sub _is_named_sub
{
    my $element = shift;
    return $element->isa('PPI::Statement::Sub')
        && $element->schild(1)->isa('PPI::Token::Word');
}


=head2 _get_package_name_by_line_number

    По номеру строки и найденным элементам PPI::Statement::Package определяем, какому модулю принадлежит строка

=cut

sub _get_package_name_by_line_number
{
    my ($line_number, $package_statements) = @_;
    if (ref $package_statements eq 'ARRAY') {
        if (my $nearest_package_statement = first { $_->location()->[0] < $line_number } reverse @$package_statements) {
            return $nearest_package_statement->schild(1)->content();
        }
    }
    return 'main';
}


=head2 _get_ppi_doc

    отдаем результат парсера (прозрачно кэшируем)

=cut

sub _get_ppi_doc
{
    my $filename = shift;
    unless ($PPI_CACHE->{$filename}) {
        $PPI_CACHE->{$filename} ||= PPI::Document->new($filename);
        die "can't parse file $filename: " . PPI::Document->errstr unless $PPI_CACHE->{$filename};
        $PPI_CACHE->{$filename}->index_locations();
    }
    return $PPI_CACHE->{$filename};
}
