#!/usr/bin/perl

use my_inc "..";


=head1 NAME

mk_regions.pl - обновить гео-регионы

=head1 DESCRIPTION

    Генерация странички со списком регионов, таблиц со списком регионов и соответствующих им офисов geo_regions,
    таблиц geo_translation с регионами geobase, их названиями и мапингом на регионы в Директе

    Принадлежность региона к офису определяется по списку регионов в поле serviced_regions таблицы yandex_offices в PPCDICT.

    $Id$

=head1 RUNNING

    При выполнении миграций запускать mk_regions.pl обычно имеет смысл сразу после выкладки кода. 
    Либо сначала выложить на ppc и запустить, а потом обновить и остальные серверы.

    Выполняется обычно полминуты-минуту.

    Принимает параметры (как минимум один из них должен быть указан):
        --files -- обновить шаблоны страниц со списком регионов
        --db -- обновить таблицы в БД (не обновляет таймзоны)
          дополнительно можно опцией --db-to-update указать одну или несколько БД, тогда будут обновлены только они
        --db-to-update <DB>, --dbu обновить только указанные инстансы. Полезно для тестовых БД. Можно указывать несколько раз
          Пример: --dbu ppc:1 --dbu ppcdict
          Если не указан, будут обновлены все инстансы.
        --timezones -- обновить таймзоны в БД
        --langs -- список языков, для которых надо генерировать файлы (ru|ua|tr|en)
          Если не указан список, подразумевается "все языки" (список захардкожен, TODO правильно было бы брать из настроек Директа)
        --new-regions-file - путь к файлу с id регионов, которые планируем добавить в геодерево
          Используется для предварительной проверки возможности добавления указанных регионов:
          валидность id, наличие флага bs, переводов и т.п.
        --report-file - файл в который будет выгружен отчет о новых регионах, используется вместе с параметром --new-regions-file
          Отчет формируется в виде таблицы, которую можно вставить в тикет
        --ya-path - путь к утилите ya для загрузки страницы с новым геодереом в MDS, используется вместе с параметром --files

    Для обновления метро в интерфейсе, нужно запустить: ./protected/prebuild/get_metro_js.pl

    ВНИМАНИЕ! Чтобы шаблоны data/regions_${lang}.html не получались на украинском, надо использовать версию 0.21 для yandex-du-i18n-perl,
    но просто так на hardy она не работает. Можно использовать бранч svn+ssh://svn.yandex.ru/direct-utils/yandex-lib-branches/i18n-hardy.
    Или сразу запускать скрипт так:
    PERL5LIB=/home/pankovpv/i18n-hardy/lib ./protected/mk_regions.pl --files --langs ru ua tr en

=head1 EXAMPLES

Для проверки списка новых регионов и загрузки обновленного геодерева в MDS:
    ./protected/mk_regions.pl --files --new-regions-file ~/new_regs.txt --ya-path ~/arcadia/ya

Для продакшена:
    ./protected/mk_regions.pl --files
    ./protected/mk_regions.pl --db
    ./protected/mk_regions.pl --timezones

=head1 internal

    Есть хитрая зависимость: 
    модуль geo_regions.pm генерируется этим скриптом (mk_regions.pl), но сам mk_regions.pl использует GeoTools, который использует geo_regions. 

    Поэтому: если в какой-то момент сборка geo_regions совсем сломалась и он получился пустой -- 
    mk_regions.pl не станет больше работать ("geo_regions.pm did not return a true value"). 

    Возможные сиюминутные решения: записать в geo_regions "1;" или восстановить его из svn.
    
    Полноценного решения, возможно, совсем нет: 
    в mk_regions.pl обрабатывает шаблон data/t/regions/all_regions.tmpl.html, а в шаблоне используется GeoTools::get_geo_names, 
    которая использует %geo_regions::GEOREG. 

    Все дерево регионов забирается из гео-базы, из украинского дерева (с Крымом в Украине) - актуальный урл указан в svn+ssh://svn.yandex.ru/direct-utils/geobase-pm/readme
    Отдельные тарнслокальные деревья формируются из него, перемещением Крыма в нужное место (в корень дерева или в Россию).
    Забираем в пакете geobase-pm

    Такие дела...

=cut


use Direct::Modern;

use Template;

use Encode qw( encode_utf8 decode_utf8 );
use FindBin qw/$Bin/;
use Path::Tiny;

use ScriptHelper;

use Settings;
use Yandex::Template::Provider::Encoded;
use Yandex::ListUtils qw/xisect xuniq nsort/;
use Yandex::Shell qw/yash_system/;
use YandexOffice;
use List::MoreUtils qw/uniq any/;
use Yandex::DateTime;
use geobase;
use Tools;
use TextTools;
use Yandex::DBTools;
use GeoTools;
use File::Slurp;
use JSON;
use Yandex::Clone qw/yclone/;
use List::Util qw/first/;

use Yandex::I18n;

use utf8;
use open ':std' => ':utf8';

# для добавления языка нужно дополнить структуры ниже
my %aliases = (
    ru => 'name',
    ua => 'ua_name',
    en => 'ename',
    tr => 'tr_name'
);

# кастомизируем названия регионов из гео-базы
my %predefined_names = (
    ru => {
        1  => "Москва и область",
        3  => "Центр",
        17 => "Северо-Запад",
        26 => "Юг",
        40 => "Поволжье",
        52 => "Урал",
        59 => "Сибирь",
        73 => "Дальний Восток",
        166  => "СНГ (исключая Россию)",
        102444 => "Северный Кавказ",
        20383 => "Савеловская (Серпуховско-Тимирязевская)",
        215103 => "Савеловская (Большая кольцевая)",
        11278 => "Берёзовский (Кемеровская область)",
        29397 => "Берёзовский (Свердловская область)",
        972 => "Дзержинск (Нижегородская область)",
        26018 => "Дзержинск (Минская область)",
    },
    ua => {
        3  => "Центр",
        17 => "Північний Захід",
        26 => "Південь",
        40 => "Поволжя",
        52 => "Урал",
        59 => "Сибір",
        73 => "Далекий Схід",
        166  => "СНД (за винятком Росії)",
        102444 => "Північний Кавказ",
        193330 => "Ходжейлийский район",
        193331 => "Ходжейли",
        189657 => "Ахангаранский район",
        11282 => "Кемеровская область (Кузбасс)",
        11278 => "Берёзовский (Кемеровская область)",
        29397 => "Берёзовский (Свердловская область)",
        972 => "Дзержинск (Нижегородская область)",
        26018 => "Дзержинск (Минская область)",
    },
    en => {
        166  => "CIS (except Russia)",
        972 => "Dzerzhinsk (Nizhny Novgorod District)",
        26018 => "Dzerzhinsk (Minsk District)",
        10987 => "Armavir (Krasnodar Krai)",
        105791 => "Armavir (Armenia)",
        11278 => "Berezovskiy (Kemerovo District)",
        29397 => "Berezovskiy (Sverdlovsk District)",
    },
    tr => {
        166  => "CIS (Rusya dahil değil)",
        193330 => "Xo‘jayli District",
        193331 => "Xojayli",
        189657 => "Okhangaron District",
        11282 => "Kemerovskaya oblastı",
        11278 => "Beriozovski (Kemerovskaya oblastı)",
        29397 => "Beriozovski (Sverdlovskaya oblastı)",
        10987 => "Armavir (Krasnodarski krayı)",
        105791 => "Armavir Bölgesi (Armenia)",
        10522 => "Belgrad",
        104492 => "Belgrad Bölgesi",
        132 => "Hayfa",
        103901 => "Hayfa Bölgesi",
        131 => "Tel Aviv",
        103905 => "Tel Aviv Bölgesi",
        122056 => "İmereti",
        122063 => "Şida Kartli",
        122058 => "Mtsheta-Mtianeti",
        122060 => "Samegrelo-Zemo Svaneti",
        122055 => "Guria",
        122059 => "Raca-Leçhumi ve Kvemo Svaneti",
        122062 => "Kvemo Kartli",
        122061 => "Samtshe-Cavaheti",
        122054 => "Acara Özerk Cumhuriyeti",
        122057 => "Kaheti",
        196979 => 'Denau district',
        196980 => "Danau",
    }
);

my @skipped_names = (
    'Городской округ',
    'Міський округ',
    'Городское поселение',
    'Сельское поселение',
    'Муниципальное образование',
);

my %undefined_names = (
    name => "Не определeн", ename => 'Undefined', 
    ua_name => 'Не определен', tr_name => 'Belirsiz'
);
my %all_names = (
    name => "Все", ename => 'All', 
    ua_name => 'Всі', tr_name => 'Tümü',
);
# Замена городов в таймзонах
my %TZ_CITY_TRANSLATE = (
    Kamchatka => 'Petropavlovsk',
    Aqtobe => 'Aktobe',
);

# Некая грубая настройка
# Список регионов первого уровня
my @top_regions = ( 225, 166, 111, 183, 241, 10002, 10003, 138 );

my $region_undefined = {
    id => 0, type => 0, 
    %undefined_names,
    media => 1, oldname => 'Все', geo_flag => 0, 
    parents => [], level => 0, childs => \@top_regions
};

# Исключения для медиа регионов
my $MEDIA_MIN_TYPE = 4;
my %MEDIA_EXCL = map {$_ => 1} (1, 10174);

# Определение порядка сортировки регионов
my %ORDERS = (
    3 => 1, 17 => 2, 26 => 3, 40 => 3, 59 => 4, 73 => 5, # Регионы России
    1 => 1, 213 => 1, # Моск. обл, Москва
    10174 => 1, # Ленинградская область
);

# неуникальные названия регионов, к которым надо дописать название страны
my @NEED_ALIAS = (
    20526, 20527, 20086,
    11220, # Троицк
    24702, # Ленинский район
    98626, # Кировский район
    121048, # Железногорский городской округ
    20297, # Пушкинский район
    11471, # Алушта
    10942, # Печора
    116765, # Феодосія
    116767, # Судак
    10298, # Петропавловск (в Казахстане)
);

# регионы на которые действует транслокальность (пока только Крым)
my %TRANSLOCAL_REGIONS = (
    977 => {
        translocal_parent => {
            ru => 225,
            ua => 187,
        },
        new_parents => {
            ru => {
                to_remove => {166 => 1, 187 => 1}, # убираем "Украина", "СНГ (исключая Россию)"
                to_add => [225], # добавляем "Россия"
            },
            # для Украины остаются старые parents
        },
    }
);

# Существуют станции метро, находящиеся вне административных границ города, к метрополитену которого они относятся. В geobase.pm такие станции не содержат город в качестве родителя, поэтому для них требуется указывать город явно.
my %CITY_FOR_METRO = (
    20325 => 2,     # Девяткино => Санкт-Петербург
);

# сюда собираем чайлдов регионов
my %CHILDS;

my ($UPDATE_FILES, $UPDATE_DB, @DBS_TO_UPDATE, $UPDATE_TIMEZONES, @SUPPORTED_LANGS, $DATA_PATH, $REPORT_PATH, $YA_PATH);
extract_script_params(
    "files|f"   => \$UPDATE_FILES,
    "db|d|b"    => \$UPDATE_DB,
    "db-to-update|dbu=s@"     => \@DBS_TO_UPDATE,
    "timezones|t" => \$UPDATE_TIMEZONES,
    "langs=s{,}" => \@SUPPORTED_LANGS,
    "new-regions-file=s" => \$DATA_PATH,
    "report-file=s" => \$REPORT_PATH,
    "ya-path=s" => \$YA_PATH,
);


@SUPPORTED_LANGS = qw(ru ua tr en) if scalar @SUPPORTED_LANGS == 0;


unless (($UPDATE_FILES || $UPDATE_DB || $UPDATE_TIMEZONES || $DATA_PATH) && @SUPPORTED_LANGS) {
    die "please use '$0 --files' to update files or '$0 --db' to update db or --new-regions-file for regions checks\n";
}
if (my @unsupported = grep {$_ !~ /^(ua|tr|ru|en)$/} @SUPPORTED_LANGS) {
    die 'unsupported langs ' . join " ", @unsupported;
}

# готовим geo_translation до удаления регионов из %geobase::Region
my @to_geo_translation;
if ($UPDATE_DB) {
    while (my ($id, $h) = each %geobase::Region) {
        next unless $h->{parents};
        # для каждого региона геобазы ищем ближайшего родственника в директе
        for my $p ($id, reverse @{ $h->{parents} }) {
            if (exists $geo_regions::GEOREG{$p}) {
                push @to_geo_translation, [$id, $p, $geobase::Region{$p}->{name}];
                last;
            }
        }
    }
}



# отдельный костыль, который убирает городские округа по названию
# и несколько регионов по списку (https://st.yandex-team.ru/DIRECT-48422)
my %regions_for_delete = map {($_ => 1)} (
    114619,     # "Новомосковский административный округ"
    114620,     # "Троицкий административный округ"

    20524,      # Запад/Украина/СНГ/Евразия/Земля
    20525,      # Восток/Украина/СНГ/Евразия/Земля
    20526,      # Юг/Украина/СНГ/Евразия/Земля
    20527,      # Центр/Украина/СНГ/Евразия/Земля
    20528,      # Север/Украина/СНГ/Евразия/Земля
    980,        # Страны Балтии/Европа/Евразия/Земля
    1004,       # Ближний Восток/Азия/Евразия/Земля
);

my $skip = join '|' => map {quotemeta} @skipped_names;
my $skip_re = qr/^(?:$skip) | (?:$skip)$/i;

for my $geo_id (keys %geobase::Region) {
    my $region = $geobase::Region{$geo_id};
    if (
        $regions_for_delete{$geo_id} ||
        $region->{name} =~ $skip_re && !$region->{bs}
    ) {
        # my $parent_id = $region->{path}->[-1];
        # $log->out("Skipping $geo_id $geobase::Region{$parent_id}->{name} / $region->{name}");
        $regions_for_delete{$geo_id} = 1;
        delete $geobase::Region{$geo_id};
    }
}
# удаленные регионы удаляем из родителей и детей
for my $geo_id (keys %geobase::Region) {
    my $region = $geobase::Region{$geo_id};
    $region->{parents} = [grep {!$regions_for_delete{$_}} @{$region->{parents}}];
    $region->{chld}    = [grep {!$regions_for_delete{$_}} @{$region->{chld}}];
    $region->{path}    = [grep {!$regions_for_delete{$_}} @{$region->{path}}];
    $region->{suggest} = [grep {!$regions_for_delete{$_}} @{$region->{suggest}}];
}

# БД, в которых создается geo_regions
my @MAIN_DBS = (
    dbnames(PPC(shard => 'all')),
    PPCDICT,
);

# БД, в которых создается geo_translation
my @all_dbs = (
    @MAIN_DBS,
);
my %supported_db = map { $_ => 1 } @all_dbs;
@DBS_TO_UPDATE = @all_dbs if scalar @DBS_TO_UPDATE == 0;
for my $db (@DBS_TO_UPDATE){
    die "usupported db $db, stop\n" unless $supported_db{$db};
}

my $tree_html_link;
if ($UPDATE_FILES) {
    my @acts = (
        ###############################################
        # template, perl module
        { tmpl => "$Bin/../data/t/regions/all_regions.tmpl.pm",
          out => "$Bin/../protected/geo_regions.pm",
          encoding => 'utf8',
        },
        (map {
            my $tmpl = $_;
            map {
                {
                    tmpl => "$Bin/../data/t/regions/${tmpl}.tmpl.html",
                    out => "$Bin/../data/t/regions/${tmpl}_${_}.html",
                    encoding => 'utf8',
                    lang => $_,
                    translocal_regions_src_name => 'regions_translocal_ua', # ключ в %TMPL под которым находится нужное дерево регионов
                },
                {
                    tmpl => "$Bin/../data/t/regions/${tmpl}.tmpl.html",
                    out => "$Bin/../data/t/regions/${tmpl}_${_}_for_ru.html",
                    encoding => 'utf8',
                    lang => $_,
                    translocal_regions_src_name => 'regions_translocal_ru',
                }
            } @SUPPORTED_LANGS
         } 'all_regions', 'all_regions2'
        ),
        
        (map {
            my ($tmpl, $out, %extra) = @$_;
            map {{
                    tmpl => "$Bin/../data/t/regions/${tmpl}.tmpl.html",
                    out => "$Bin/../data/t/regions/${out}_${_}.html",
                    encoding => $extra{encoding} || 'utf8',
                    lang => $_,
                    translocal_regions_src_name => 'regions_translocal_ua', # ключ в %TMPL под которым находится нужное дерево регионов
                    %extra
                },
                {
                    tmpl => "$Bin/../data/t/regions/${tmpl}.tmpl.html",
                    out => "$Bin/../data/t/regions/${out}_${_}_for_ru.html",
                    encoding => $extra{encoding} || 'utf8',
                    lang => $_,
                    translocal_regions_src_name => 'regions_translocal_ru',
                    %extra   
                }
            } @SUPPORTED_LANGS
         } ['all_regions', 'media_regions', extdata => {media => 1}],
           ['all_regions', '../../regions', twice => 1]
        )
            
            # не найдено где шаблон применяется, но страницы доступны снаружи напрямую
            # после выкладки DIRECT-13828 можно будет удалить
            # ['all_regions', '../advq_regions',
            #    twice => 1,
            #    extdata => {advq => 1},
            #    encoding => 'cp1251']

    );

    # Строим чайлдов
    for my $id (sort {$a <=> $b} keys %geobase::Region) {
        my $h = $geobase::Region{$id};
        next if !defined $h->{parents} || !@{ $h->{parents} };
        push @{ $CHILDS{ $h->{parents}->[-1] } }, $id;
    }

    # Хэш для шаблона
    # Получаем дерево всех интересующих нас категорий
    my %TMPL = (
        regions => process_regions(\@top_regions, \@SUPPORTED_LANGS),
        supported_langs => \@SUPPORTED_LANGS,
        name_aliases => [get_aliases(@SUPPORTED_LANGS)],
    );
    
    # Удаляем ветки без привязок
    my @arr;
    simplify( \@arr, \%TMPL );
    set_media(@{$TMPL{regions}});
    set_translocal(\%TMPL);
    
    # создаем отдельные транслокальные деревья
    create_translocal_region_tree(\%TMPL);

    # собираем %$georeg -- в точности такой же хеш, как %geo_regions::GEOREG 
    # Для geo_regions.pm %GEOREG собирается рекурсивным обходом $TMPL{regions} средствами Template::Toolkit, 
    # и почти такой же TT-обход есть в all_regions.tmpl.html 
    # Кажется, что логично иметь это преобразование ($TMPL{regions} ==> %GEOREG) в perl-коде
    my $georeg = make_georeg($TMPL{regions_translocal_ua});
    $TMPL{georeg} = $georeg;
    $TMPL{regions_for_suggest_csv} = join ", ", regions_for_suggest($georeg); 
    $TMPL{georeg_for_ru} = make_georeg($TMPL{regions_translocal_ru});
    $TMPL{georeg_api} = make_georeg($TMPL{regions}); # дерево для API с Крымом в корне
    
    # добавляем алиасы для неуникальных названий регионов
    for my $reg_id ( @NEED_ALIAS ) {
        next if $regions_for_delete{$reg_id};
        for my $name_key (get_aliases(@SUPPORTED_LANGS)) {
            my $georeg = $TMPL{georeg}->{$reg_id};
            my $alias = $georeg->{$name_key} . q{ (} . $TMPL{georeg}->{ $georeg->{parents}->[-1] }->{$name_key} . q{)};
            $TMPL{georeg_uniq_alias}->{$reg_id}->{$name_key} = $alias;
            $TMPL{georeg_uniq_name}->{$name_key}->{$alias} = $reg_id;
        }
    }

    my $has_dupe_err;
    # проверяем, не осталось ли дублей
    for my $name_key (get_aliases(@SUPPORTED_LANGS)) {
        my %name_count;
        while ( my ($reg_id, $data) = each %{$TMPL{georeg}} ) {
            my $uname = ( $TMPL{georeg_uniq_alias}->{$reg_id} ? $TMPL{georeg_uniq_alias}->{$reg_id}->{$name_key} : undef ) || $data->{$name_key};
            die "reg ($reg_id) has no name" unless $uname;
            push @{$name_count{ $uname }}, $reg_id;
        }
        while( my ($name, $ids) = each %name_count ) {
            next if @$ids < 2;
            print "Duplicate region $name_key = $name (@{$ids})\n";
            $has_dupe_err = 1;
        }
    }
    die "There were some dupes" if $has_dupe_err;

    # соберем данные о метро
    my @metro_city = (1, 2, 143); # Москва Спб Киев
    my $MOSCOW = 213;
    # хеш по region_id

    # выбираем все записи из %Region с типом 9 (метро) и 14 (монорельс)
    $TMPL{metro} = {
        map {  
            my $city = $_;
            $city => [ sort { $a->{name} cmp $b->{name} } 
                       map {{ region_id => $_, name => _tune_metro_name($geobase::Region{$_}->{name}) }}
                       grep { (($geobase::Region{$_}->{type}||0) == 9 || ($geobase::Region{$_}->{type}||0) == 14) 
                              && (@{ (xisect [$city], $geobase::Region{$_}->{parents}) } || (exists $CITY_FOR_METRO{$_} && $CITY_FOR_METRO{$_} == $city))} 
                       keys %geobase::Region 
                     ],
        } @metro_city
    };

    # отдельно собираем список стран
    $TMPL{country_regions} = [map {{
        region_id => $_,
        copy_names($geobase::Region{$_}, \@SUPPORTED_LANGS)
    }} grep {($geobase::Region{$_}->{type}||0) == 3 || (($geobase::Region{$_}->{type}||0) == 12 && exists $GeoTools::ALLOWED_OVERSEAS_TERRITORIES{$_})} keys %geobase::Region];

    # получение region_id по имени
    map { my $id = $_; map { $TMPL{georeg_uniq_name}->{$_}->{$geobase::Region{$id}->{$_}} = $id } get_aliases(@SUPPORTED_LANGS)} (@metro_city, $MOSCOW);

    for my $translocal_type ('ru', 'ua') {
        my $regions_name_key = "regions_translocal_$translocal_type";
        for my $lang (@SUPPORTED_LANGS) {
            Yandex::I18n::init_i18n($lang, check_file => 1);
            locale_sort($TMPL{$regions_name_key}, $lang);
            my $regions_tree = regions_tree($TMPL{$regions_name_key}, $lang);
            my $filename_suff = $translocal_type eq 'ru' ? "_for_$translocal_type" : "";
            write_file("$Bin/../data/regions_${lang}${filename_suff}.json", {binmode => ':utf8', atomic => 1}, to_json($regions_tree, { canonical => 1 }));
        }
    }
    yash_system("make -C $Bin/../data3/ regions-tree");

    my $t = new Template( LOAD_TEMPLATES => [ Yandex::Template::Provider::Encoded->new( encoding => 'utf8'
                                                                                , ABSOLUTE => 1
                                                                                , EVAL_PERL => 1
                                                                                , INCLUDE_PATH => [ "$Bin/../data/t/regions" ]
                                                                                , ) ],
                          PRE_DEFINE => {
                                     iget => \&iget,
                                     get_geo_names => \&GeoTools::get_geo_names,
                                     },
                          FILTERS => {
                              js                         => \&js_quote,
                              url                        => \&url_quote,
                          }
             );

    $TMPL{original_regions} = yclone($TMPL{regions}); # для использования в шаблонах ключа "regions"

    for my $act (@acts) {
        my $enc = $act->{encoding} || 'cp1251';

        open(my $fh, ">:encoding($enc)", $act->{out}) || die "Can't open $act->{out}: $!";
        $TMPL{extdata} = $act->{extdata};
        # Если надо - настраиваем интернационализацию
        if ($act->{lang}) {
            Yandex::I18n::init_i18n($act->{lang}, check_file => 1);
            $TMPL{lang} = $act->{lang};
        }
        
        $TMPL{regions} = yclone($act->{translocal_regions_src_name} ? $TMPL{ $act->{translocal_regions_src_name} } : $TMPL{original_regions});
        locale_sort($TMPL{regions}, $act->{lang} || 'ru');
        if ($act->{twice}) {
            my $tmp;
            $t->process($act->{tmpl}, \%TMPL, \$tmp) or die $t->error();
            $t->process(\$tmp, \%TMPL, $fh) or die $t->error();
        } else {        
            $t->process($act->{tmpl}, \%TMPL, $fh) or die $t->error();
        }
    }
    if ($YA_PATH) {
        my $upload_output = `cat $Bin/../data/regions_ru.html | sed '1i<meta charset="UTF-8">' | $YA_PATH upload --mds --ttl 14 --json-output`;
        eval {
            my $processed_output = from_json($upload_output);
            $tree_html_link = $processed_output->{download_link};
        };
    }
}

if ($DATA_PATH) {
    my $new_regions_ids = { map {$_ => 1} grep {$_} path($DATA_PATH)->lines({ chomp => 1}) } or $log->die("Can't open new regions file $DATA_PATH: $@");
    if (%$new_regions_ids) {
        my $new_regions_cnt = scalar keys %$new_regions_ids;
        $log->out("Got $new_regions_cnt new regions ids from file");

        # выкачиваем новую геобазу
        `[ ! -d "/tmp/ttl_7d" ] && mkdir -p "/tmp/ttl_7d" && wget -O '/tmp/ttl_7d/new_geobase.pm' 'http://geoexport.yandex.ru/?format=pm&new_parents=977%3A187' && sed -i "1s/.*/package new_geobase;/" /tmp/ttl_7d/new_geobase.pm`;
        use lib '/tmp/ttl_7d/';
        require new_geobase;

        # творческий copy-paste с GeoTools::_sort_regions_by_tree
        my $ids = [nsort keys %$new_regions_ids];
        my $parents = [uniq map { @{ $new_geobase::Region{$_}->{parents} // [] } } @$ids];
        my $sorted_reg_list = [ grep { ! %{$new_geobase::Region{$_} // {}} } @$ids ];
        my $sort_regions_by_tree;
        $sort_regions_by_tree = sub {
            my ($reg_list, $tree_node_id) = @_;

            if (any { $_ == $tree_node_id } @$reg_list) {
                push @$sorted_reg_list, $tree_node_id;
            }
            my %reg_list_extended = map { $_ => 1 } (@$parents, @$reg_list);

            my @childs = nsort grep { $reg_list_extended{$_} } @{ $new_geobase::Region{$tree_node_id}->{chld} // [] };
            foreach (@childs) {
                $sort_regions_by_tree->($reg_list, $_);
            }
        };
        $sort_regions_by_tree->($ids, 10000); # Земля, а не "весь мир"

        my @new_regions_lines;
        for my $reg_id (@$sorted_reg_list) {
            my $reg;
            # id региона есть в геобазе
            $reg->{is_exist} = exists $new_geobase::Region{$reg_id}  && %{ $new_geobase::Region{$reg_id} }? 1 : 0;
            if ($reg->{is_exist}) {
                # стоит ли флаг bs в новой геобазе
                $reg->{bs_new_geobase} = $new_geobase::Region{$reg_id}->{bs};
                # есть переводы названия на ru ua tr en
                $reg->{en_name} = $new_geobase::Region{$reg_id}->{en_name};
                $reg->{tr_name} = $new_geobase::Region{$reg_id}->{tr_name};
                $reg->{ua_name} = $new_geobase::Region{$reg_id}->{ua_name};
                $reg->{name} = $new_geobase::Region{$reg_id}->{name};
                # для генерации нового геодерева представим что флаг bs уже стоит в геобазе
                $geobase::Region{$reg_id}->{bs} = 1;
                my $parent_id = $new_geobase::Region{$reg_id}->{parents}->[-1];
                $reg->{parent} = $new_geobase::Region{$parent_id}->{name} . " ($parent_id)";
            }

            my $red_line = 0;
            unless ($reg->{is_exist} && $reg->{name} && $reg->{en_name} && $reg->{tr_name} && $reg->{ua_name}) {
                $red_line = 1;
            }
            push @new_regions_lines, '|| '. ($red_line? "??" : "") . $reg_id . ($red_line? "??" : "") .'| '
                                    . ($reg->{is_exist}) .'| '
                                    . ($reg->{name} || '') .'| '
                                    . ($reg->{en_name} || '') .'| '
                                    . ($reg->{ua_name} || '') .'| '
                                    . ($reg->{tr_name} || '') .'| '
                                    . ($reg->{bs_new_geobase} || '') .'| '
                                    . ($reg->{parent} || '')
                                    . "||\n";
        }
        my $disclamer = "Регион может быть добавлен в геодерево Директа если он существует, есть переводы его названия на английский, "
                        . "украинский и турецкий, в геобазе установлен в 1 флаг bs.\n"
                        . "Если id региона отмечен ??красным?? цветом, значит кроме флага bs у него не хватает еще чего-то для добавления в геодерево Директа (например, переводов)\n"
                        . "Значение полей:\n"
                        . "* id - идентификатор региона в геобазе\n"
                        . "* есть в геобазе - если 1, то регион с таким номером существует\n"
                        . "* Название - название региона на русском языке\n"
                        . "* EN - перевод на английский\n"
                        . "* UA - перевод на украинский\n"
                        . "* TR - перевод на турецкий\n"
                        . "* флаг bs - установлен в 1 если на регион разрешен таргетинг в Директе\n";
        my $start = "#|\n";
        my $header = "**|| id| есть в геобазе| Название| EN| UA| TR| флаг bs| Родитель ||**\n";
        my $end = "|#\n";
        my ($report, $path);
        if ($REPORT_PATH) {
            $report = path($REPORT_PATH);
            $path = $REPORT_PATH;
        } else {
            $log->out("please use '--report-file' param to write report in file");
            $report = Path::Tiny->tempfile({realpath => 1}, TEMPLATE => "regions_report-XXXXXXXX", DIR => "/tmp/temp-ttl/ttl_2d", UNLINK => 0);
            $path = $report->realpath;
        }
        my $link = $tree_html_link ? "\nПример нового дерева: $tree_html_link\n" : '';
        if (!$REPORT_PATH) {
            my $html_header = '<!DOCTYPE html><html lang="ru"><head><meta charset="UTF-8"><title>Отчёт</title></head><body><pre>';
            my $html_footer = '</pre></body></html>';
            $report->spew($html_header, $start, $header, @new_regions_lines, $end, $disclamer, $link, $html_footer);
        } else {
            $report->spew($start, $header, @new_regions_lines, $end, $disclamer, $link);
        }
        $log->out("Written report in $path");
        if (!$REPORT_PATH) {
            my $upload_output = `$YA_PATH upload --mds --ttl 14 --json-output $path`;
            eval {
                my $processed_output = from_json($upload_output);
                print "\n\nReport is uploaded to $processed_output->{download_link}\n";
            };
        }
    } else {
        $log->die("Can't find new regions ids in file $DATA_PATH");
    }
}

if ($UPDATE_DB) {
    my @to_geo_regions;

    while (my ($id, $h) = each %geobase::Region) {
        next unless $h->{parents};
        next if $id < 0;

        # костыль: для удалённых регионов (не всех) парентом указан -1, а в БД это поле положительное
        # поэтому для таких меняем парента на 20001 - в геобазе это корень для всего удалённого
        my $parent = $h->{parents}->[-1];
        $parent = 20001  if $parent && $parent < 0;
        push @to_geo_regions, {
            region_id => $id,
            parent_id => $parent,
            type => $h->{type},
            copy_with_pred_names($h, $id, \@SUPPORTED_LANGS),
            office_id => get_office_by_geo($id)->{office_id},
        };
    }

    for my $db (@DBS_TO_UPDATE) {
        do_mass_insert_sql($db, "REPLACE LOW_PRIORITY INTO geo_translation (region, direct_region, region_name) VALUES %s", \@to_geo_translation);
    }

    push @to_geo_regions, {
        region_id => $region_undefined->{id},
        parent_id => $region_undefined->{parents}->[-1],
        type => $region_undefined->{type},
        copy_with_pred_names($region_undefined, $region_undefined->{id}, \@SUPPORTED_LANGS),
        office_id => get_office_by_geo($region_undefined->{id})->{office_id},
    };

    my @fields = (get_aliases(@SUPPORTED_LANGS), qw/region_id parent_id type office_id/);
    my $insert = sprintf q[INSERT INTO geo_regions(%s) VALUES %%s
            ON DUPLICATE KEY UPDATE %s],
        join(',', @fields),
        join(',', map {"$_ = VALUES($_)"} (get_aliases(@SUPPORTED_LANGS), qw/parent_id type office_id/));

    my %db_needs_update = map { $_ => 1 } @DBS_TO_UPDATE;
    for my $db (grep {$db_needs_update{$_}} @MAIN_DBS) {
        do_mass_insert_sql($db, $insert,
            [map {[@{$_}{@fields}]} @to_geo_regions]);
    }

    # записываем в ppc_properties информацию о дате изменения дерева регионов
    if ( $db_needs_update{ppcdict} ){
        do_insert_into_table( PPCDICT, 'ppc_properties',
            { name => 'geo_regions_update_time', value__dont_quote => 'NOW()' },
            on_duplicate_key_update => 1
        );
    }
}

if ($UPDATE_TIMEZONES) {
    generate_timezones(@SUPPORTED_LANGS);                       
}

###############################################################

sub copy_with_pred_names {
    
    my ($source, $id, $langs) = @_;
    
    return map {
        my $alias = $aliases{$_};
        ($alias => $predefined_names{$_}->{$id} || _tune_region_name($source->{$alias}))
    } @$langs
}


sub _tune_region_name {
    my ($name) = @_;

    # rename Городской округ ХХХ -> ХХХ (городской округ)
    state $rename_re = join '|' => map {quotemeta} @skipped_names;
    $name =~ s/^($rename_re) (.+)$/"$2 (" . lc($1) . ")"/e;

    return $name;
}



# Создание дерева регионов
sub process_regions {
    
    my ($childs, $langs) = @_;
    
    my @res;
    for my $id (@$childs) {
        next if !defined $geobase::Region{$id};
        push @res, {
            id => $id,
            copy_with_pred_names($geobase::Region{$id}, $id, $langs),
            (map {($_ => $geobase::Region{$id}->{$_})} qw/main type parents/),
            regions => process_regions($CHILDS{$id}, $langs),
        };
    }
    return \@res;
}

# Удаляем ветки без привязок
sub simplify {
    my ( $ar, $data, $lvl ) = @_;
    $data->{lvl} = $lvl;
    $lvl ||= 0;
    my $sum = $data->{id} && $geobase::Region{$data->{id}}->{bs} ? 1 : 0;
    my @new_regions = ();
    for my $reg ( @{ $data->{regions} } ) {
        my @arr;
        my $cnt = simplify( \@arr, $reg, $lvl + 1 );
        if ( $cnt > 0 && @arr ) {
            $sum += $cnt;
            push @new_regions, @arr;
        }
    }
    $data->{sum} = $sum;

    # Преобразуем названия региона "Саратовская область" -> "Саратов и область"
    if ($lvl > 2) {
        my ($main_region) = grep {$_->{main}} map {$geobase::Region{$_}} @{$geobase::Region{$data->{id}}->{chld}};
        if ($main_region) {
            magicfy_region_name($data, $main_region);
        }
    }

    @{$ar} = ( $data );
    $data->{regions} = \@new_regions;
    return $sum;
}

sub set_media {
    my @regs = @_;
    for my $r (@regs) {
        set_media(@{$r->{regions}});
        $r->{media} = $r->{type} > 0 && $r->{type} <= $MEDIA_MIN_TYPE || $MEDIA_EXCL{$r->{id}} ? 1 : 0;
        $r->{media_regions} = grep {$_->{media}} @{$r->{regions}};
    }
}

# --------------------------------------------------------------------
# выносим спорные территории из дерева в корень (пока только Крым)
sub set_translocal {
    my $regions = shift;
    my $translocal_regions = shift // [];
    my $lvl = shift // 0;

    my $translocal_region_index;
    for my $i ( 0 .. $#{ $regions->{regions} } ) {
        if ($regions->{regions}->[$i]->{id} && exists $TRANSLOCAL_REGIONS{$regions->{regions}->[$i]->{id}}) {
            $regions->{regions}->[$i]->{is_translocal} = 1;
            $regions->{regions}->[$i]->{translocal_parent} = $TRANSLOCAL_REGIONS{$regions->{regions}->[$i]->{id}}->{translocal_parent};
            $translocal_region_index = $i;
        } else {
            set_translocal($regions->{regions}->[$i], $translocal_regions, $lvl + 1);
        }
    }

    if (defined $translocal_region_index) {
        my $translocal_region = splice @{$regions->{regions}}, $translocal_region_index, 1;
        push @$translocal_regions, $translocal_region;
    }

    # добавляем транслокальные регионы в корень дерева
    if ($lvl == 0 && @$translocal_regions) {
        push @{$regions->{regions}}, @$translocal_regions;
    }
}

# --------------------------------------------------------------------
# поиск транслокальных регионов, копирование их в отдельные ветки под ключами regions_translocal_*
sub create_translocal_region_tree {
    my $regions = shift;

    my $translocal_regions = [];
    for my $root_region_index (0 .. $#{ $regions->{regions} }) {
        my $root_region = $regions->{regions}->[$root_region_index];
        if ($root_region->{is_translocal}) {
            for my $translocal_type (keys %{ $root_region->{translocal_parent} }) {
                # "regions_translocal_ru", "regions_translocal_ua"
                $regions->{"regions_translocal_$translocal_type"} = yclone($regions->{regions});
                my $translocal_region = splice @{ $regions->{"regions_translocal_$translocal_type"} }, $root_region_index, 1;
                fix_translocal_regions($translocal_region, $translocal_region->{id}, $translocal_type);
                move_translocal_regions($regions->{"regions_translocal_$translocal_type"}, $translocal_region, $translocal_type);
            }
        }
    }
}

# --------------------------------------------------------------------
# исправление содержимого транслокального региона (пока только parents)
sub fix_translocal_regions {
    my ($translocal_region, $translocal_region_id, $translocal_type) = @_;

    return unless exists $TRANSLOCAL_REGIONS{ $translocal_region_id }->{new_parents};

    my $parents = $translocal_region->{parents};

    # удаляем не нужных родителей
    my $new_parents = [
        grep {! $TRANSLOCAL_REGIONS{$translocal_region_id}->{new_parents}->{$translocal_type}->{to_remove}->{$_}}
        @$parents
    ];

    # добавляем нужных
    push @$new_parents, @{$TRANSLOCAL_REGIONS{$translocal_region_id}->{new_parents}->{$translocal_type}->{to_add}}
        if exists $TRANSLOCAL_REGIONS{$translocal_region_id}->{new_parents}->{$translocal_type}->{to_add};

    $translocal_region->{parents} = $new_parents;

    for my $child_region (@{$translocal_region->{regions}}) {
        fix_translocal_regions($child_region, $translocal_region_id, $translocal_type);
    }
}

# --------------------------------------------------------------------
# перемещаем транслокальные регионы внутрь дерева
sub move_translocal_regions {
    my $regions = shift;
    my $translocal_region = shift;
    my $translocal_type = shift;

    for my $region (@$regions) {
        if ($translocal_region->{translocal_parent}->{$translocal_type}
            && $region->{id} == $translocal_region->{translocal_parent}->{$translocal_type}
           )
        {
            push @{ $region->{regions} }, $translocal_region;
            return 1;
        }

        if (ref($region->{regions}) eq 'ARRAY') {
            my $result = move_translocal_regions($region->{regions}, $translocal_region, $translocal_type);
            return 1 if $result; # переместили 1 регион, больше не требуется обходить рекурсивно
        }
    }

    return 0;
}

# Магическое преобразование региона
# "Саратовская область" -> "Саратов и область"
# действует
#  - только для России
#  - только для русского и украинского языка
#  - только для областей
#  - 
sub magicfy_region_name {
    my ($parent, $main) = @_;
    if ( 
        (grep {$_==225 || $_==187 || $_ == 159 || $_==149} @{$parent->{parents}})
        && ($parent->{name} =~ /^\w+\s+область/ 
            || $parent->{name} =~ /^\S+\s+область/ &&
                 $parent->{name} =~ /^\Q$main->{name}\E/
            )
    ) {
        # Пропускаем район или округ
        if ($main->{type} == 10) {
            my ($new_main_candidate) = grep {$_->{main}} map {$geobase::Region{$_}} @{$main->{chld}};
            $main = $new_main_candidate if $new_main_candidate;
        }

        my $name = $main->{name};
        $name =~ s/\s+\(.*?\)//g;

        if (substr($parent->{name}, 0, 2) eq substr($main->{name}, 0, 2)) {
            $parent->{name} = "$name и область";
            $parent->{ua_name} = "$main->{ua_name} і область" if defined $main->{ua_name};
        } else {
            $parent->{name} = "$name и $parent->{name}";
            $parent->{ua_name} = "$main->{ua_name} і $parent->{ua_name}" if defined $main->{ua_name} && defined $parent->{ua_name};
        }
    }
}



=head2 block_regions

    функция для рекурсивного обхода в make_georeg, калька с BLOCK block_regions из all_regions.tmpl.pm

=cut

sub block_regions
{
    my ($result, $regions, $path) = @_;
    $regions ||= [];
    
    for my $r (@$regions){
        my $local_path = [ @$path, $r->{id} ];
        $result->{$r->{id}} = {
            (map {
                my $alias = $aliases{$_};
                ($alias => scalar @{$r->{regions}} || !$r->{"main_$alias"}  ? $r->{$alias} : $r->{"main_$alias"})
            } @SUPPORTED_LANGS),
            media => $r->{media},
            geo_flag => grep( {$_ == 225} @$path )&& $r->{lvl} > 2 || grep( {$_ == 187} @$path) && $r->{lvl} > 2 ? 1 : 0,
            parents => [ @$path ],
            level => scalar @$path,
            childs => [ map { $_->{id}+0 } @{$r->{regions} || []}],
            type => $r->{type}, 
        };

        block_regions($result, $r->{regions}, $local_path);
    }

    return;
}


=head2 make_georeg

    из многоярусного дерева регионов собирает одноуровневый хеш со связями через списки детей и родителей

=cut

sub make_georeg
{
    my $all_regions = shift;

    my $georeg = {};

    my $reg_0 = {
        %all_names,
        media => 1, 
        geo_flag => 0, 
        parents => [], 
        level => 0, 
        childs => [ map {$_->{id}+0} @$all_regions ], 
    };

    $georeg->{0} = $reg_0;

    block_regions($georeg, $all_regions, [0] );

    return $georeg;
}


=head2 regions_for_suggest

    обрабатывает одноуровневый хеш регионов ($georeg), возвращает список регионов, которые мы подсказываем в "быстром выборе". 
    Это должны быть: 
    для России, Украины, Казахстана и Белоруссии -- области (кроме Московкой и Ленинградской -- они уже всегда есть в быстром выборе), 
    в остальном -- страны

=cut

sub regions_for_suggest
{
    my $georeg = shift;

    my @res;

    for my $id (keys %$georeg){
        my $reg = $georeg->{$id};

        next if $id =~ /^(1|10174|225|187)$/;

        if ( grep { /^(225|187|149|159)$/ } @{$reg->{parents}} ){
            next unless $reg->{type} == 5;
            push @res, $id;
            next;
        }

        next unless ($reg->{type} || 0) == 3;
        push @res, $id;
    }
    
    return sort { $a <=> $b} @res;
}


=head2 copy_names($source, $langs)

    копирование названия регионов из источника($source)
    по списку поддерживаемых яызков

    на выходе:
    (
        name_tr => "",
        name_en => "",
        name_ru => ""
    )

=cut

sub copy_names {

    my ($source, $langs) = @_;
    
    return map {
       ("name_${_}" => $source->{$aliases{$_}})
    } @$langs;
}

sub get_aliases {@aliases{@_}}

# сортировка дерева регионов с учетом языка
sub locale_sort {
    
    my ($tree, $lang) = @_;

    my $name = (get_aliases($lang))[0];
    {
        use locale;
        Yandex::I18n::init_i18n($lang, check_file => 1);
        foreach my $r (@$tree) {
            _sort_child_regions($r, $name);
        }
    }
}

sub _sort_child_regions {
    
    my ($region, $name) = @_;

    $region->{regions} = [sort {
        ($ORDERS{$a->{id}} || 1000) <=> ($ORDERS{$b->{id}} || 1000)
            || lc($a->{$name}) cmp lc($b->{$name})
    } @{$region->{regions}}];
    foreach my $r (@{$region->{regions}}) {
        _sort_child_regions($r, $name)
    }
}

# Создание списка таймзон и запись в ppcdict
sub generate_timezones {
    my @langs = @_;

    # модифицируем для удобства геобазу
    while(my ($id, $v) = each %geobase::Region) {
        $v->{id} = $id;
        $v->{parents_hash} = { map {$_ => 1} @{$v->{parents}} };
    }
    my @timezones;
    my @names = get_aliases(@langs);
    
    my $manual_timezones = {
        # Украина: не показываем крымскую зону
        187 => {
            skip => [ qw# Europe/Moscow # ],
        },
        # Россия
        225 => {
            # упразднённые зоны, которые попадаются в геобазе
            skip => [ qw# Asia/Sakhalin Asia/Novosibirsk Asia/Magadan # ],
            # новые зоны, которые (возможно) ещё не проросли в геобазу
            add =>  [ qw# Asia/Srednekolymsk # ],
        },
    };

    # цикл по странам
    for my $c (grep {$_->{type} && $_->{type} == 3} values %geobase::Region) {

        # таймзоны бьются на группы
        my $group_nick = $c->{id} == 225
            ? 'russia'
            : $c->{parents}->[-1] == 166
            ? 'cis'
            : 'world';
        # -1 скрытые
        my @childs = grep {
                         $_->{type}
                         && $_->{type} != -1
                         && $_->{parents_hash}->{$c->{id}}
                     } values %geobase::Region;
        my %childs_by_ename = map {(my $qq = $_->{ename}) =~ s/_/ /g; $qq => $_} @childs;

        my @all_country_timezones =
            grep {tz_offset($_) % 3600 == 0}
            uniq
            grep {$_}
            map {$_->{timezone}}
            sort {$a->{id} <=> $b->{id}}
            @childs;

        # Для ручных - исправляем 
        my @country_timezones;
        if ( my $modify = $manual_timezones->{$c->{id}} ) {
            my %skip = map {($_ => 1)} @{$modify->{skip}};
            @country_timezones =
                uniq
                grep { !$skip{$_} }
                (@all_country_timezones, @{$modify->{add}});
        }
        # ... для остальных - выбираем уникальные
        else {
            @country_timezones = xuniq {tz_offset($_)} @all_country_timezones;
        }

        for my $tz (@country_timezones) {
            my ($continent, $city_ename) = split /\//, $tz;
            $city_ename = $TZ_CITY_TRANSLATE{$city_ename} || $city_ename;
            my $city = $childs_by_ename{$city_ename} || (grep {$_->{timezone} eq $tz} @childs)[0];

            my %tz = (
                country_id => $c->{id}, timezone => $tz,
                group_nick => $group_nick
            );
            for my $n (@names) {
                my $tzname = $c->{$n};
                if (@country_timezones > 1 && $city) {
                    $tzname = $group_nick eq 'russia'
                        ? $city->{$n}
                        : $tzname . ", $city->{$n}";
                }
                $tz{$n} = $tzname;
            }
            push @timezones, \%tz;
        }
    }

    my @fields = (get_aliases(@langs), qw/country_id timezone group_nick/);
    my $insert = sprintf q[INSERT INTO geo_timezones(%s) VALUES %%s
            ON DUPLICATE KEY UPDATE %s],
        join(',', @fields),
        join(',', map {"$_ = VALUES($_)"} get_aliases(@langs), 'group_nick', 'timezone');

    do_mass_insert_sql(PPCDICT, $insert, [map {[@{$_}{@fields}]} @timezones]);  

    # записываем в ppc_properties информацию о дате изменения таймзон
    do_insert_into_table(PPCDICT, 'ppc_properties',
                         { name => 'geo_timezones_update_time',
                           value__dont_quote => 'NOW()'
                         }, on_duplicate_key_update => 1
                        );
}

sub regions_tree
{
    my $regions = shift;
    my $lang = shift;

    return undef unless @$regions;
    my $name_field = {en => 'ename', ru => 'name', ua => 'ua_name', tr => 'tr_name'}->{$lang};
    return [map { +{name => $_->{$name_field}, id => $_->{id}, inner => regions_tree($_->{regions}, $lang)} } @$regions];
}

# TODO (yandex-du-dumper-perl): следующие две функции вызывать из модуля Yandex::Dumper, когда/если он появится

# см. также: CPAN-модуль Data::Recursive::Encode
sub encode_utf8_recursive {
    my ($struct) = @_;

    unless ( defined $struct ) {
        return undef;
    }

    unless ( ref $struct ) {
        return encode_utf8($struct);
    }

    if ( ref $struct eq 'ARRAY' ) {
        return [ map { encode_utf8_recursive($_) } @$struct ];
    }

    if ( ref $struct eq 'HASH' ) {
        return { map { encode_utf8($_) => encode_utf8_recursive( $struct->{$_} ) } keys %$struct };
    }

    die "Unrecognised structure: $struct";
}

sub _tune_metro_name {
    my ($name) = @_;

    $name =~ s!ё!е!g;
    $name =~ s/^\s*|\s*$//g;
    $name =~ s/\(Московское центральное кольцо\)/(МЦК)/;

    return $name;
}
