#!/usr/bin/env perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */20 * * * *
    <switchman>
        group:  scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:            13h
    tag: direct_group_internal_systems
    raw_events: scripts.moderateSyncDiags.working.common
    raw_events: scripts.moderateSyncDiags.working.performance
</juggler>

<crontab>
    time: */20 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

=cut

=head1 DESCRIPTION

    Привозит из Модерации и БК причины отклонения в PPCDICT.moderate_diags

=cut

use strict;
use warnings;
use utf8;
use feature q/state/;

use File::Slurp;
use Text::Diff;


use Yandex::SendMail qw/send_alert/;
use Yandex::Shell;
use Yandex::I18n;

use Moderate::JSONRPC::Client;
use BannerStorage;

use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;
use Property;
use Yandex::DBTools;

use Moderate::Settings;
use Moderate::Tools;

# workaround for: Can't call method "_put_session" on an undefined value at /usr/lib/perl5/AnyEvent/Handle.pm
# it's a race condition in the object destroying order
use Carp::Always;

# скрипт выдает false-negatives (требует обновлений, когда все в порядке)
my $BROADCAST_EMAIL = 'direct-maintenance@yandex-team.ru';
local $Yandex::SendMail::ENVELOPE_SENDER = $BROADCAST_EMAIL;
my $mod_reasons_file = "$Settings::ROOT/protected/data/translation/moderate_reasons.trans";
my $MIN_REASONS_NUMBER = 20;

my $sql = "INSERT INTO moderate_diags (diag_id, type, text, textFull, bad_reason, unban_is_prohibited, token)
           VALUES %s
           ON DUPLICATE KEY UPDATE
               text = IF(values(token) IS NULL, text, values(text)),
               textFull = IF(values(token) IS NULL, textFull, values(textFull)),
               bad_reason = values(bad_reason),
               unban_is_prohibited = values(unban_is_prohibited),
               token = values(token)";

sub _update_moderate_diags {
    my ($type, $diags) = @_;

    my @data;
    foreach my $v (@$diags) {
        my $decline_shows = $v->{decline_shows} && $v->{decline_shows} eq 'Yes' ? 'Yes' : 'No';
        my $unban_is_prohibited = $v->{unban_is_prohibited} && $v->{unban_is_prohibited} eq 'Yes' ? 'Yes' : 'No';
        # unban_is_prohibited пришло на замену decline_shows
        # используем unban_is_prohibited вместо decline_shows в качестве значения для
        # bad_reason, потому что модерация уже не поддерживает актуальное значение в
        # decline_shows, а Директ ещё не перешёл на unban_is_prohibited
        my $bad_reason = $type eq 'common' ? $unban_is_prohibited : $decline_shows;
        push @data, [
            $v->{diag_id},
            $type,
            $v->{text},
            $v->{text_full}     || undef,
            $bad_reason,
            $unban_is_prohibited,
            $v->{token},
        ];
    }

    eval {
        do_mass_insert_sql(
            PPCDICT,
            $sql,
            \@data
        );
    };

    if ($@) {
        $log->die("$@");
    }
}

extract_script_params();

$log->out('START');

my $moderate_json_rpc_url = Moderate::Tools::get_moderation_jsonrpc_url();
$log->out("Fetching reasons from $moderate_json_rpc_url");

my $jsonrpc = Moderate::JSONRPC::Client->new(
    Moderate::Tools::get_moderation_jsonrpc_url({use_cache => 1}),
    { timeout => 60 }
);
my $result = $jsonrpc->getDiags();
$log->out('Got reasons:', $result);
my $common_ok = $result && %$result && (scalar keys %$result >= $MIN_REASONS_NUMBER) ? 1 : 0;
_update_moderate_diags(common => [map {values %$_} values %$result]);

$log->out("Fetching reasons from $BannerStorage::BS_REST_API_URL");
$result = BannerStorage::get_rejection_reasons();
$log->out('Got reasons:', $result);
my $performance_ok = $result && @$result && (scalar @$result >= $MIN_REASONS_NUMBER) ? 1 : 0;
_update_moderate_diags(performance => [map { {diag_id => $_->{id}, text => $_->{name}, text_full => $_->{text}} } @$result]);

my $prop = Property->new('moderate_new_reasons_email_time');
my $last_update_time = $prop->get() || 0;
$log->out("moderate_new_reasons_email_time property is $last_update_time");
my $delta_update_time = time() - $last_update_time;

if (! $last_update_time || $delta_update_time > 60*60*24) {
    $log->out('Checking for diff');
    my $new_reasons_file = yash_qx("$Settings::ROOT/protected/maintenance/mk_moderate_reasons.pl --no-file --force");

    my $old_reasons_file = read_file($mod_reasons_file);
    my $diff = diff(\$old_reasons_file, \$new_reasons_file);

    utf8::decode($diff);

    # в случае наличия непредвиденных причин отклонения, пишем письмо в рассылку
    if ($diff) {
        $log->out("Sending diff alert to $BROADCAST_EMAIL");
        my $text = <<'TEXT';
        На модерации добавлены новые причины отклонения.
        Для их перевода необходимо подготовить бету с продакшен-базой, запустить генерацию и закоммитить файл с текстами:
        mk conf_prod
        ./protected/maintenance/mk_moderate_reasons.pl
        commit protected/data/translation/moderate_reasons.trans
        mk conf_devtest
TEXT
        $text .= "===$mod_reasons_file\n--- file\n+++ DB\n$diff";
        send_alert($text , 'New unsynced moderate reasons', $BROADCAST_EMAIL);
    }
    $prop->set(time);
}

$log->out("Writing juggler data");
if ($common_ok) {
    juggler_ok(service_suffix => 'common');
}
if ($performance_ok) {
    juggler_ok(service_suffix => 'performance');
}

$log->out('FINISH');
