#!/usr/bin/perl

=head1 DESCRIPTION

   Переводим на раздельное управление ставками все кампании с ручным управлением в которых есть ограничение ставки в сети
   В каких таблицах обновляем ставки
   bids
   bids_base

=head1 USAGE

    Запуск в холостом режиме, без записи в базу
    LOG_TEE=1 ./protected/one-shot/102225_calc_priceContext_and_add_to_resync.pl --dry-run [--shard 1 ] [--cid 1234 ]

    Запуск с записью в базу
    LOG_TEE=1 ./protected/one-shot/102225_calc_priceContext_and_add_to_resync.pl

    Перезапуск с конкретного cid'a. Будет выполняться в шарде указанного cid'a.
    LOG_TEE=1 ./protected/one-shot/102225_calc_priceContext_and_add_to_resync.pl --cid 12345

    Запуск в конкретном шарде
    LOG_TEE=1 ./protected/one-shot/102225_calc_priceContext_and_add_to_resync.pl --shard 1

    Перезапуск для всех указанных шардов/cid'ов
    LOG_TEE=1 ./protected/one-shot/102225_calc_priceContext_and_add_to_resync.pl --resume '[{"shard":1,"cid":123}, {"shard": 2, "cid": 456}]'

=cut

use Direct::Modern;

use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils;

use my_inc '../..', for => 'protected';
use BS::ResyncQueue;
use ScriptHelper 'Yandex::Log' => [ date_suf => '%Y%m%d', auto_rotate => 1, tee => $ENV{LOG_TEE}, lock => 1 ];
use Settings;
use LogTools;
use ShardingTools;
use List::MoreUtils qw/uniq/;
use JSON qw/to_json from_json/;
use Currencies;

my $FEATURE_NAME = "context_limit_inside_strategies";
my $DRY_RUN;
my $SHARD;
my $START_CID;
my $RESUME_INFO;
my $BIDS_CHUNK_SIZE = 50000;
my $CAMPAIGNS_CHUNK_SIZE = 500;

my $ONE_ITERATION = undef;
my $ONLY_CID;

my $STOP_FILE_LOCATION = "$Settings::ROOT/protected/run/set_independent_bids_and_calc_price_context.stop";

my $run_params ="@ARGV";
extract_script_params(
    'dry-run'       => \$DRY_RUN,
    'shard:i'       => \$SHARD,
    'cid:i'         => \$START_CID,
    'cid-chunk:i'   => \$CAMPAIGNS_CHUNK_SIZE,
    'bid-chunk:i'   => \$BIDS_CHUNK_SIZE,
    'resume:s'      => \$RESUME_INFO,
    'one-iteration' => \$ONE_ITERATION,
    'only-cid:i'  => \$ONLY_CID
);

if ( -e $STOP_FILE_LOCATION ) {
    $log->die("not running, found stop file at $STOP_FILE_LOCATION");
}
my $SHARDS_AND_CIDS = {};
$log->out('start');
$log->out("run with params: $run_params");

if ($ONLY_CID) {
    $START_CID = $ONLY_CID;
    $ONE_ITERATION = 1;
    $CAMPAIGNS_CHUNK_SIZE = 1;
    $SHARDS_AND_CIDS->{get_shard(cid => $START_CID)} = $START_CID;
    $log->out("running for only cid :$START_CID");
} elsif ($START_CID) {
    $SHARDS_AND_CIDS->{get_shard(cid => $START_CID)} = $START_CID;
    $log->out("starting with cid: $START_CID" );
} elsif ($RESUME_INFO) {
    $log->out("strarting with resume params: $RESUME_INFO" );
    $SHARDS_AND_CIDS->{$_->{shard}} = $_->{cid} foreach @{from_json($RESUME_INFO)};
} elsif ($SHARD) {
    $SHARDS_AND_CIDS->{$SHARD} = undef;
}

my @shards_to_process = keys %$SHARDS_AND_CIDS;
@shards_to_process = ppc_shards() unless @shards_to_process;

my $currency_min_prices =  {map { $_ => $Currencies::_CURRENCY_DESCRIPTION{$_}->{MIN_PRICE} } keys %Currencies::_CURRENCY_DESCRIPTION};

my $feature_percent = get_one_field_sql(PPCDICT, [
    'select json_extract(settings, "$.percent") from features',
    WHERE => { feature_text_id => $FEATURE_NAME },
    LIMIT => 1,
]);
die "Feature <$FEATURE_NAME> is not set to all clients. Stopped"  if $feature_percent != 100;

$log->out("processing shards: @shards_to_process" );
$log->out("cid chunk size: $CAMPAIGNS_CHUNK_SIZE, bid chunk size: $BIDS_CHUNK_SIZE");
$log->out("to stop, touch $STOP_FILE_LOCATION");
$log->out({currency_min_prices => $currency_min_prices});

foreach_shard_parallel shard => \@shards_to_process, sub {
    my ($shard) = @_;

    my $next_cid = $SHARDS_AND_CIDS->{$shard} || get_one_field_sql(PPC(shard => $shard), "select min(cid) from campaigns");

    my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");
    while(1) {
        $log->out("selecting campaigns from cid $next_cid");

        my $campaigns_where = {
            "archived" => "No",
            "platform" => "both",
            "autobudget" => "No",
            _OR => {"co.strategy__is_null" => 1, "co.strategy__ne" => "different_places"},
            "ContextPriceCoef__lt" => 100,
        };

        if ($ONLY_CID) {
            $campaigns_where->{cid} = $next_cid;
        } else {
            $campaigns_where->{cid__ge} = $next_cid;
        }

        my $cids = get_one_column_sql(PPC(shard => $shard), ["select cid from campaigns c join camp_options co using (cid)",
            WHERE => $campaigns_where,
            "ORDER BY cid",
            "LIMIT $CAMPAIGNS_CHUNK_SIZE"
        ]);

        $log->out("fetched campaigns to update: " . scalar(@$cids));
        last unless @$cids;

        my $cids_str = join(', ', @$cids);
        my $campaigns_info = get_all_sql(PPC(shard => $shard), ["
        SELECT
            c.cid,
            c.currency,
            ContextPriceCoef,
            ContextLimit,
            ifnull(bids_cnt_query.cnt, 0) as bids_count,
            ifnull(bids_base_cnt_query.cnt, 0) as bids_base_count
        FROM
            campaigns c
            JOIN camp_options using (cid)
            LEFT JOIN (select cid, count(*) cnt from bids where cid in ($cids_str) group by cid) bids_cnt_query on c.cid = bids_cnt_query.cid
            LEFT JOIN (select cid, count(*) cnt from bids_base where cid in ($cids_str) group by cid) bids_base_cnt_query on c.cid = bids_base_cnt_query.cid",
            WHERE => {
                "c.cid" => $cids,
             },
        ]);

        $log->out({campaigns_info => $campaigns_info});
        modify_price_context_on_shard($shard, $campaigns_info);

        $next_cid = $cids->[-1] + 1;

        $log->out("Resume info: {\"shard\": $shard,  \"cid\": $next_cid}");
        if ( -e $STOP_FILE_LOCATION ) {
            $log->out("terminating early, found stop file at $STOP_FILE_LOCATION");
            last;
        }

        last if (defined $ONE_ITERATION);
    }
    $log->out(" finished.");

};

sub log_prices_and_get_cids {
    my($campaigns_bids_rows, $table, $log_price_info,  %opts) = @_;
    my @cids;
    my @log;
    foreach (@$campaigns_bids_rows) {
        push @log, {
            cid       => $_->{cid},
            pid       => $_->{pid},
            id        => $_->{id},
            type      => "update_" . $table . "_price_context",
            price     => $_->{price},
            price_ctx => $_->{price_context},
            currency  => $_->{currency}
        };
        push @cids, $_->{cid};
    }
    if (@log) {
        LogTools::log_price(\@log) if ($opts{log_price});
        $log->out({ $log_price_info => \@log });
    }
    return [uniq @cids];
}

#принимает на вход массив [{cid => , bids_count => , bb_count => }]. Разделяет его на чанки размером не больше $bids_count_limit
#если ставки в кампании не умещаются в чанк, делает чанк с одной кампанией.
sub get_cids_chunks($$) {
    my ($cids, $bids_count_limit) = @_;

    my $sum = 0;
    my @cid_chunks = ();
    my @next_chunk = ();
    foreach my $cid_info (@$cids) {
        my $next_sum = $sum + $cid_info->{bids_count} + $cid_info->{bids_base_count};

        if ($next_sum <= $bids_count_limit || $sum == 0 ) {
            $sum = $next_sum;
        } else {
            $sum = 0;
            push @cid_chunks,[@next_chunk];
            @next_chunk = ();
        }
        push @next_chunk, $cid_info->{cid};
    }
    push @cid_chunks, [@next_chunk] if (@next_chunk);
    return \@cid_chunks;
}

sub select_bids($$) {
    my ($shard, $cids) = @_;

    my $campaigns_bids = get_all_sql(PPC(shard => $shard), [ "
            SELECT
              c.cid,
              c.currency,
              b.pid,
              b.id,
              b.price,
              b.price_context
            FROM campaigns c
              LEFT JOIN bids b on c.cid = b.cid",
        WHERE => {
            "c.cid" => $cids,
        },
        "ORDER BY id",
    ]);

    my $campaigns_bids_base = get_all_sql(PPC(shard => $shard), [ "
            SELECT
              c.cid,
              c.currency,
              bb.pid,
              bb.bid_id as id,
              bb.price,
              bb.price_context
            FROM campaigns c
              LEFT JOIN bids_base bb on c.cid = bb.cid",
        WHERE => {
            "c.cid" => $cids,
            'bb.bid_type__ne' => 'keyword',
        },
        "ORDER BY bid_id",
    ]);

    return ($campaigns_bids, $campaigns_bids_base);
}

sub modify_price_context_on_shard {
    my ($shard, $campaigns_info) = @_;
    foreach my $cids ( @{get_cids_chunks($campaigns_info, $BIDS_CHUNK_SIZE)}) {
        $log->out("processing cids: @$cids");

        my ($campaigns_bids, $campaigns_bids_base) = select_bids($shard, $cids);

        my $bids_fetched_cnt = scalar @$campaigns_bids;
        my $bids_base_fetched_cnt = scalar @$campaigns_bids_base;
        my $warning = ($bids_fetched_cnt + $bids_base_fetched_cnt > $BIDS_CHUNK_SIZE)?  "Oversized chunk!": "";
        $log->out("$warning Selected rows -> bids: $bids_fetched_cnt; bids_base: $bids_base_fetched_cnt;");

        #пишем текущие ставки в лог
        my $cids_for_bids_update = log_prices_and_get_cids($campaigns_bids, "bids", "log_price_before_update");
        my $cids_for_bids_base_update = log_prices_and_get_cids($campaigns_bids_base, "bids_base", "log_price_before_update");

        unless ($DRY_RUN) {
            $log->out("updating price_context");

            do_in_transaction {
                my $profile = Yandex::Trace::new_profile('strategies:update_price_context');

                do_sql(PPC(shard => $shard), [ "update bids b
                        join campaigns c using (cid)
                        join camp_options co using (cid)
                    set
                        b.price_context = GREATEST(price * c.ContextPriceCoef / 100," . sql_case('c.currency', $currency_min_prices) . ")",
                    "where" => {
                        "c.platform"             => "both",
                        "c.autobudget"           => "No",
                        "c.ContextPriceCoef__lt" => 100,
                        _OR => {"co.strategy__is_null" => 1, "co.strategy__ne" => "different_places"},
                        "b.cid"                  => $cids_for_bids_update } ]) if (@$cids_for_bids_update);

                do_sql(PPC(shard => $shard), [ "update bids_base bb
                        join campaigns c using (cid)
                        join camp_options co using (cid)
                    set
                        bb.price_context = GREATEST(price * c.ContextPriceCoef / 100," . sql_case('c.currency', $currency_min_prices) . ")",
                    "where" => {
                        "c.platform"             => "both",
                        "c.autobudget"           => "No",
                        "c.ContextPriceCoef__lt" => 100,
                        "bb.bid_type__ne" => "keyword",
                        _OR => {"co.strategy__is_null" => 1, "co.strategy__ne" => "different_places"},
                        "bb.cid"                 => $cids_for_bids_base_update } ]) if (@$cids_for_bids_base_update);

                $log->out("updating strategy params");
                do_sql(PPC(shard => $shard), [ "update campaigns c
                        join camp_options co using (cid)
                    set
                        c.ContextPriceCoef = 100,
                        c.ContextLimit = 0,
                        co.strategy = 'different_places'",
                    "where" => {
                            "c.platform"         => "both",
                            "c.autobudget"       => "No",
                            "c.archived"         => "No",
                            _OR => {"co.strategy__is_null" => 1, "co.strategy__ne" => "different_places"},
                            "c.cid"              => $cids } ]);
                };

                $log->out("adding adgroups to resync_queue");
                push @$campaigns_bids, @$campaigns_bids_base;

                my @resync_data = map {{
                    cid      => $_->{cid},
                    pid      => $_->{pid},
                    bid      => 0,
                    priority => 85 }} xuniq {$_->{pid}} @$campaigns_bids;
                $log->out({ resync_data_adgroups => \@resync_data });

                $log->out("adding campaigns to resync_queue");
                my @resync_campaigns = map {{
                    cid      => $_->{cid},
                    pid      => 0,
                    bid      => 0,
                    priority => 85 }} xuniq {$_->{cid}} @$campaigns_bids;
                $log->out({ resync_data_campaigns => \@resync_campaigns });

                push @resync_data, @resync_campaigns;
                bs_resync(\@resync_data);
            }

            ($campaigns_bids, $campaigns_bids_base) = select_bids($shard, $cids);
            #сохраняем новые ставки в log_price
            log_prices_and_get_cids($campaigns_bids, "bids", "log_price_after_update", log_price => 1);
            log_prices_and_get_cids($campaigns_bids_base, "bids_base", "log_price_after_update", log_price => 1);
        }
}


