#!/usr/bin/perl

use my_inc '../..';

=head1 DESCRIPTION

    Миграция - полная копия deploy/archive/20151007_MODIFY-convert-unconverted-campaigns.pl
    за исключением обновленной копипасты после коммента:
    "# дальше сплошная копипаста из Client::ConvertToRealMoneyTasks::convert_client_inplace"
    
    - доконвертирует ранее несконвертированные по разным причинам кампании
    - если кампания по ошибке не подключена к ОС - подключаем
    - у явно указанных клиентов поправляем work_currency, если все кампании клиента успешно сконвертированы

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::TimeCommon;
use Yandex::Balance qw(balance_get_orders_info);
use Yandex::ListUtils qw(xsort chunks);
use Yandex::HashUtils;

use Settings;
use ScriptHelper;
use ShardingTools qw(ppc_shards);
use Client::ConvertToRealMoneyTasks ();

use Campaign;
use Client;
use Client::ConvertToRealMoney;
use Common;
use CampAutoPrice::Common;
use Currency::Rate;
use DBLog;
use Campaign::Types qw(get_camp_kind_types);

my $OPERATOR_UID = 0;
my @CLIENT_IDS_FIX_WORK_CURRENCY = (1417150);
{
    no warnings 'once';
    $Client::ConvertToRealMoneyTasks::log = $log;
}

$log->out('START');

for my $shard (ppc_shards()) {
    my $shard_prefix = "[shard_$shard]";
    my $msg_prefix_guard = $log->msg_prefix_guard($shard_prefix);

    $log->out('Fetching campaigns to fix');
    my $campaigns = get_all_sql(PPC(shard => $shard), ['
        SELECT c.cid, q.ClientID,
               q.uid, q.convert_type,
               q.state, q.new_currency,
               q.country_region_id, q.email,
               q.start_convert_at
        FROM currency_convert_queue q
        INNER JOIN users u ON q.ClientID = u.ClientID
        INNER JOIN campaigns c ON u.uid = c.uid
    ',  WHERE => {
        'q.convert_type' => 'MODIFY',
        'q.state' => 'DONE',
        'c.currencyConverted' => 'No',
        'c.type' => get_camp_kind_types('currency_convert'),
        _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
    }]);
    $log->out('Got ' . scalar(@$campaigns) . ' campaigns to fix:', $campaigns);

    for my $camp (@$campaigns) {
        my $cid = $camp->{cid};
        my $cid_msg_prefix_guard = $log->msg_prefix_guard("$shard_prefix\t[cid_$cid]");
        $log->out('Converting campaign');
        _convert_campaign_inplace($camp);
        $log->out('Check campaign binding to wallet');
        _check_bind_to_wallet($cid);
    }
}

for my $client_id (@CLIENT_IDS_FIX_WORK_CURRENCY) {
    my $msg_prefix_guard = $log->msg_prefix_guard("[ClientID $client_id]");
    my $currency = get_one_field_sql(PPC(ClientID => $client_id), ["select new_currency from currency_convert_queue",
                                                                     where => {ClientID => $client_id,
                                                                               convert_type => 'MODIFY',
                                                                               state => 'DONE'}]);
    next unless $currency;
    my $client_current_currency = get_one_field_sql(PPC(ClientID => $client_id), 'select IFNULL(work_currency, "YND_FIXED") from clients where ClientID = ?', $client_id);
    my $client_not_converted_camps = get_one_field_sql(PPC(ClientID => $client_id), ['select count(*) 
                                                                                        from campaigns c
                                                                                        join users u using(uid)',
                                                                                       where => {'c.type' => get_camp_kind_types('currency_convert'),
                                                                                                 'c.currency__ne' => $currency,
                                                                                                 'u.ClientID' => $client_id}]);
    if ($client_current_currency eq 'YND_FIXED' && !$client_not_converted_camps) {
        $log->out("Current client work_currency: $client_current_currency");
        $log->out("Updating client work_currency to: $currency");
        my $update_ok = do_update_table(PPC(ClientID => $client_id), 'clients', {work_currency => $currency}, where => {ClientID => $client_id});
        $log->out("Successful update of client work_currency") if $update_ok;

    }
}

$log->out('FINISH');


=head2 _convert_campaign_inplace

=cut

sub _convert_campaign_inplace {
    my ($camp) = @_;

    my $cid = $camp->{cid};
    my $client_id = $camp->{ClientID};
    my $data = hash_cut $camp, qw(ClientID uid convert_type state new_currency country_region_id email start_convert_at);
    my $currency = $data->{new_currency};
    my $client_msg_prefix = "[copy_convert_bids_and_stop_campaigns] [ClientID $client_id]";

    # дальше сплошная копипаста из Client::ConvertToRealMoneyTasks::convert_client_inplace

    my $rbac = Client::ConvertToRealMoneyTasks::_get_rbac_object();
    my $cond = {#'c.uid' => $client_chief_uid, # не копипаста, закомментил в миграции
                'c.type' => get_camp_kind_types('currency_convert'),
                'c.currencyConverted' => "No",
                _TEXT => 'IFNULL(c.currency, "YND_FIXED") = "YND_FIXED"',
                'c.cid' => $cid, # не копипаста, добавлено в миграции
    };

    my $campaigns_data = Common::get_user_camps_by_sql($cond, {shard => {ClientID => $client_id}, include_empty_campaigns => 1});
    my $campaigns = $campaigns_data->{campaigns};
    $log->out('Got ' . scalar(@$campaigns) . ' campaigns from DB to convert');

    my @cids = map {$_->{cid}} @$campaigns;
    clear_auto_price_queue(\@cids, error_str => 'currency convert');

    my $rate = convert_currency(1, 'YND_FIXED', $currency);
    my $rate_nds = convert_currency(1, 'YND_FIXED', $currency, with_nds => 1);

    my $migrate_datetime = Client::ConvertToRealMoneyTasks::_get_migrate_date($data->{start_convert_at}, $data->{convert_type});
    my $first_date_in_real_currency = mysql_round_day($migrate_datetime);

    $log->out('Fetching chips cost/spent from Balance to update in DB');
    for my $cids_chunk (chunks \@cids, 10) {
        my $cids_chunk_str = join(',', @$cids_chunk);
        $log->out("Fetching chips cost/spent for cids chunk: $cids_chunk_str");
        my $balance_info_chunk = balance_get_orders_info($cids_chunk);
        my @for_insert;
        for my $order (@$balance_info_chunk) {
            $log->out($order);
            my $cid = $order->{ServiceOrderID};
            my $chips_cost = $order->{CompletionFixedMoneyQty} // $order->{CompletionMoneyQty};
            my $chips_spent = $order->{CompletionFixedQty} // $order->{completion_qty} // 0;
            push @for_insert, [$cid, $chips_cost, $chips_spent];
        }
        $log->out("Updating chips cost/spent in DB for cids chunk $cids_chunk_str");
        do_mass_insert_sql(PPC(ClientID => $client_id), '
            INSERT INTO campaigns_multicurrency_sums
                (cid, chips_cost, chips_spent) VALUES %s
            ON DUPLICATE KEY UPDATE
                chips_cost = VALUES(chips_cost),
                chips_spent = VALUES(chips_spent)
        ', \@for_insert);
    }

    $log->out('Begin converting campaigns');

    my $campaigns_multicurrency_sums_data = get_hashes_hash_sql(PPC(ClientID => $client_id), ['SELECT cid, sum, chips_cost FROM campaigns_multicurrency_sums', WHERE => {cid => \@cids}]);
    for my $campaign (@$campaigns) {
        my $cid = $campaign->{cid};
        $log->msg_prefix("$client_msg_prefix [cid $cid]");

        my $converted_campaign_params = Client::ConvertToRealMoney::get_converted_campaign_params($campaign, $currency, $rate, keep_ProductID => 1, copy_last_bill_sums => 1, save_conversion_strategy => 1);

        # берём сохранённые суммы для кампании в валюте, которые получаем в нотификациях от Баланса
        my $campaign_multicurrency_sums_data = $campaigns_multicurrency_sums_data->{$cid};
        my $new_sum_spent = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{chips_cost} // 0 : 0;
        my $new_sum = ($campaign_multicurrency_sums_data) ? $campaign_multicurrency_sums_data->{sum} // 0 : 0;

        $converted_campaign_params->{campaigns}->{sum} = $new_sum;
        $converted_campaign_params->{campaigns}->{sum_spent} = $new_sum_spent;
        $converted_campaign_params->{campaigns}->{balance_tid} = 0;
        $log->out('Converted params:', $converted_campaign_params);

        $log->out("Logging new campaign sum $new_sum");
        DBLog::log_balance($data->{ClientID}, $data->{uid}, $cid, $new_sum, 0, 0, $campaign->{mediaType}, $currency);

        $log->out('Converting campaign');

        do_in_transaction {
            # лочим валюту в campaigns. в других местах можно также выбрать кампанию с FOR UPDATE и добиться последовательного выполнения транзакции на построчном локе.
            my $campaign_currency = get_one_field_sql(PPC(ClientID => $client_id), ['SELECT currency FROM campaigns', WHERE => {cid => $cid}, 'FOR UPDATE']);
            $log->die("Campaign $cid is already in $campaign_currency. That's VERY strange =)") if $campaign_currency && $campaign_currency ne 'YND_FIXED';

            Client::ConvertToRealMoney::convert_campaign_bids([$cid], 'YND_FIXED', $currency);
            Client::ConvertToRealMoney::convert_campaign_secondary_options([$cid], 'YND_FIXED', $currency);
            Client::ConvertToRealMoney::convert_campaign_payments_info([$cid], 'YND_FIXED', $currency);

            if ($converted_campaign_params->{camp_options} && %{$converted_campaign_params->{camp_options}}) {
                do_update_table(PPC(ClientID => $client_id), 'camp_options', $converted_campaign_params->{camp_options}, where => {cid => $cid});
            }
            do_update_table(PPC(ClientID => $client_id), 'campaigns', {
                currencyConverted => 'Yes',
                # прогноз сконвертировали по курсу, но ставки могли вырасти (больше минимальная ставка), да и шаг торгов другой — надёжнее пересчитать
                autobudgetForecastDate => undef,
                statusBsSynced => 'No',
                %{$converted_campaign_params->{campaigns}}
            }, where => {cid => $cid});

            my $additional_table = "campaigns_$campaign->{type}";
            if (exists $converted_campaign_params->{$additional_table}) {
                # сейчас здесь только campaigns_performance
                do_update_table(PPC(ClientID => $client_id), $additional_table,
                    $converted_campaign_params->{$additional_table}, where => {cid => $cid});
            }

            # записываем суммы до конвертации и после, чтобы показывать их на странице успеха
            my $money_before = $campaign->{total};
            my $money_after = $new_sum - $new_sum_spent;
            $log->out("Writing correspondence data for campaign $cid: money_before = $money_before, money_after = $money_after");
            do_insert_into_table(PPC(ClientID => $client_id), 'currency_convert_money_correspondence', {
                ClientID => $client_id,
                old_cid => $cid,
                new_cid => $cid,
                money_before => $money_before,
                money_after => $money_after,
                was_archived => ($campaign->{archived} && $campaign->{archived} eq 'Yes') ? 1 : 0,
            }, on_duplicate_key_update => 1, key => 'ClientID');

            # multicurrency: monitor.agency_clients_month_stat
            # multicurrency: ppc.stopped_camp_users

            # конвертируем грубый прогноз приблизительно
            do_update_table(PPC(ClientID => $client_id), 'camp_rough_forecast', {'forecast_sum__dont_quote' => "forecast_sum*$rate", forecast_time__dont_quote => 'forecast_time'}, where => {cid => $cid});
        };

        $log->out('Done converting campaign');
    }
    $log->msg_prefix($client_msg_prefix);
    $log->out('Done converting campaigns');
}

=head2 _check_bind_to_wallet

=cut

sub _check_bind_to_wallet {
    my $cid = shift;

    my $camp = get_one_line_sql(PPC(cid => $cid), ['select uid, 
                                                           IFNULL(AgencyID, 0) AgencyID, 
                                                           type, 
                                                           IFNULL(currency, "YND_FIXED") currency, 
                                                           wallet_cid
                                                      from campaigns',
                                                     where => {cid => $cid,
                                                               wallet_cid => 0,
                                                               type => get_camp_kind_types('under_wallet')}]);
    return unless $camp;

    my $wallet_cid = get_one_field_sql(PPC(uid => $camp->{uid}), 'select cid 
                                                            from campaigns
                                                           where type = "wallet"
                                                             and IFNULL(currency, "YND_FIXED") = ?
                                                             and IFNULL(AgencyID, 0) = ?
                                                             and uid = ?',
                                                                 $camp->{currency}, $camp->{AgencyID}, $camp->{uid});
    return unless $wallet_cid;

    my $camps_under_wallet_qty = get_one_field_sql(PPC(uid => $camp->{uid}), ['select count(*) 
                                                                                 from campaigns',
                                                                                where => { uid => $camp->{uid},
                                                                                           wallet_cid => $wallet_cid }
                                                                             ]);
    return unless $camps_under_wallet_qty;


    $log->out("Binding campaign to wallet_cid=$wallet_cid");
    do_update_table(PPC(cid => $cid), 'campaigns'
                       , {wallet_cid => $wallet_cid
                          , statusBsSynced => 'No'
                         }
                       , where => {
                           cid => $cid
                       }
                   );

    $log->out("Send campaign to balance for binding with wallet");
    # перепосылаем в биллинг кампании, чтобы проставить признак общий счет
    my $rbac = Client::ConvertToRealMoneyTasks::_get_rbac_object();
    my $camp_balance_response = eval { create_campaigns_balance($rbac, $OPERATOR_UID, [$cid], force_group_order_transfer => 1) };
    if (! $camp_balance_response || $camp_balance_response->{error} || ! $camp_balance_response->{balance_res}) {
        $log->die("Can't bind campaign to wallet in balance");
    } else {
        $log->out("Campaign successfully sended to balance for binding with wallet");
    }
}
