#!/usr/bin/env perl

use my_inc "../..";

=head1 DESCRIPTION

Перепосылка фидов (по source = file) в БЛ, чтобы заполнить target_domain

Параметры:

    shard-id -- шард, на котором запускать (по умолчанию на всех)
    limit    -- обработать столько фидов и завершиться (по умолчанию не задан)

Можно перезапускать.

=cut

use Direct::Modern;
use JSON;
use open ':std' => ':utf8';

use Direct::Feeds;
use ScriptHelper 'Yandex::Log' => 'messages';

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::Trace qw//;
use ShardingTools qw/ppc_shards foreach_shard_parallel_verbose/;
use Settings;

my ($SHARDS, $LIMIT);
extract_script_params(
    'shard-id:i@' => \$SHARDS,
    'limit:i'  => \$LIMIT,
);

my $MAX_FEED_FILE_SIZE = 512 * 1024 * 1024;
my $BL_REQUEST_TIMEOUT = 30 * 60;

$log->out('START');
my $json = JSON->new();
my $BL_CHUNK_SIZE = Direct::Feeds::get_bl_chunk_size();
my $CHUNK_SIZE = Direct::Feeds::get_bl_select_chunk_size();
my @shards = defined $SHARDS ? @$SHARDS : ppc_shards();
my %shards_set = map {$_ => undef} @shards;
$log->out("Working with shards: ". join ', ', @shards);

foreach_shard_parallel_verbose($log, sub {
    my ($shard) = @_;
    return unless exists $shards_set{$shard};
    my $feeds = get_all_sql(PPC(shard => $shard), [
        "SELECT
            f.feed_id, f.ClientID,
            f.business_type,
            f.url,
            cll.feed_max_file_size,
            f.feed_type
        FROM feeds f
        LEFT JOIN client_limits cll ON f.ClientID = cll.ClientID
        ",
        where => {
            'f.update_status' => 'Done',
            'f.source' => 'file',
            _OR => [target_domain => '', target_domain__is_null => 1]
        },
        $LIMIT ? (
            "ORDER BY feed_id",
            "LIMIT $LIMIT"
        ) : ()
    ]);

    my $current_chunk_size = 0;
    my $req;
    for my $feed (@$feeds) {
        $log->out("Resending feed $feed->{feed_id}");
        my $url = Yandex::HTTP::make_url($Settings::BANNER_LAND_URL, {
            cmd => 'yml2directinf',
            url => $feed->{url},
            business_type => $feed->{business_type},
            last_valid_feed_type => $feed->{feed_type},
            max_file_size => ($feed->{feed_max_file_size} || $MAX_FEED_FILE_SIZE),
            max_file_size_type => 'bytes',
        });
        $current_chunk_size++;
        $req->{$feed->{feed_id}} = {
            url => $url,
        };
        if ($current_chunk_size >= $CHUNK_SIZE) {
            _feeds_to_bl($shard, $req);
            $current_chunk_size = 0;
            undef %$req;
        }
    }
    if ($req && %$req) {
        _feeds_to_bl($shard, $req);
    }
});

$log->out('FINISH');

sub _feeds_to_bl {
    my ($shard, $req) = @_;
    my $profile = Yandex::Trace::new_profile('bmapi:yml2directinf', obj_num => scalar keys %$req);
    my $res = http_parallel_request(GET => $req,
        content_ref => 1,
        handle_params => { keepalive => 1 },
        ipv6_prefer => 1,
        max_req => $BL_CHUNK_SIZE,
        num_attempts => 3,
        persistent => 0,
        timeout => $BL_REQUEST_TIMEOUT,
    );
    undef $profile;

    for my $feed_id (keys %$req) {
        if (!$res->{$feed_id}) {
            $log->out("Got no results for feed $feed_id");
            next;
        }
        my $page = $res->{$feed_id};

        if (!$page->{is_success}) {
            $log->out(["Request to BL failed", {
                feed_id => $feed_id,
                url => $page->{headers}->{URL},
                response_status => $page->{headers}->{Status},
                response_reason => $page->{headers}->{Reason},
            }]);
            next;
        }
        my $content_ref = $page->{content_ref};
        my $bl_data = eval {$json->decode($$content_ref)};
        if ($@) {
            $log->out("Failed to decode JSON for feed $feed_id");
            next;
        }

        my $target_domain = Direct::Feeds::extract_domain($bl_data);
        if (!$target_domain) {
            $log->out("No target domain for feed $feed_id");
            if ($bl_data->{errors} && @{$bl_data->{errors}}) {
                $log->out(["Feed $feed_id errors:", $bl_data->{errors}]);
            }
            next;
        }

        do_update_table(PPC(shard => $shard), 'feeds', {
                LastChange__dont_quote => 'NOW()',
                target_domain => $target_domain
            },
            where => {
                feed_id => $feed_id,
                _OR => [target_domain => '', target_domain__is_null => 1]
        });
        $log->out("Feed $feed_id has a target_domain $target_domain now");
    }
}
