#!/usr/bin/env perl

=head1 SYNOPSIS

    ./50047-get_banners_data.pl [<input_file>] [-o <output_file>]

=head1 DESCRIPTION

    Скрипт для выгрузки информации о баннерах по задаче
    https://st.yandex-team.ru/DIRECT-50047

    input_file -- файл со списком bid'ов. Если не задан, список читается из стандартного ввода
    output_file -- xlsx-файл для записи результатов, по умолчанию banners_data.xlsx

=cut

use my_inc '../..';

use Direct::Modern;

use Getopt::Long;

use Settings;
use User;

use RBAC2::Extended;
use RBACDirect qw/rbac_mass_get_manager_of_agencies/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_merge/;

use Excel::Writer::XLSX;
use List::MoreUtils qw/any uniq/;
use List::Util qw/sum/;

my $output_file = 'banners_data.xlsx';
GetOptions(
    'o|output-file=s' => \$output_file,
) || die "can't parse options, stop";

my $rbac = RBAC2::Extended->get_singleton(1);

my %agg_stat;
my $logins_count = 0;
my $camps_count = 0;
my @bids;
while (my $line = <>) {
    my ($bid) = $line =~ /^(\d+)$/;
    push @bids, $bid if $bid;
}

print "Bids => ".scalar(@bids)."\n";

my %users;
my %managers;
my %agencies;
my $workbook = Excel::Writer::XLSX->new($output_file);
my $worksheet = $workbook->add_worksheet();
$worksheet->write_row(0, 0, [
    '№ объявления',
    '№ кампании',
    'Логин клиента',
    'Email клиента',
    'Логин менеджера',
    'ФИО менеджера',
    'Логин агентства',
    'Название агентства',
    'Email агентства',
]);
$worksheet->set_column(0, 3, 20);
my $row_num = 1;

my $iter = sharded_chunks_iter(bid => \@bids, chunk_size => 10000);
while (my ($shard, $bids) = $iter->()) {
    #bid/cid/логин клиента/емейл клиента/логин менеджера/ФИО менеджера/логин агентства/название агентства/емейл агентства
    my $banners = get_all_sql(PPC(shard => $shard), ["SELECT b.bid, b.cid, uid, ManagerUID, AgencyUID, c.statusActive, b.statusActive, b.statusShow, b.statusModerate
                                                      FROM banners b JOIN campaigns c ON b.cid=c.cid", 
                                                     where => { 'b.bid' => $bids, 
                                                                'c.statusActive' => 'Yes',
                                                                'b.statusActive' => 'Yes',
                                                                'b.statusModerate' => 'Yes',
                                                                'b.statusShow' => 'Yes'}]);
    my @uids = grep {!$users{$_}} uniq grep {$_} map {$_->{uid}} @$banners;
    my @agencyUids = grep {!$agencies{$_}} uniq grep {$_} map {$_->{AgencyUID}} @$banners;
    my $manager_for_agency = rbac_mass_get_manager_of_agencies($rbac, \@agencyUids);
    for (grep {$_->{AgencyUID}} @$banners) {
        die if $_->{ManagerUID};
        $_->{ManagerUID} = $manager_for_agency->{$_->{AgencyUID}};
    }
    my @managerUids = grep {!$managers{$_}} uniq grep {$_} map {$_->{ManagerUID}} @$banners;
    
    my $new_users = get_users_data(\@uids, [qw/login email ClientID/]);
    hash_merge \%users, $new_users;
    my $new_managers = get_users_data(\@managerUids, [qw/login fio/]);
    hash_merge \%managers, $new_managers;
    my $new_agencies = get_users_data(\@agencyUids, [qw/login fio email/]);
    hash_merge \%agencies, $new_agencies;

    ## no critic (Freenode::DollarAB)
    foreach my $b (@$banners) {
        $worksheet->write_row($row_num++, 0, [
            $b->{bid},
            $b->{cid},
            $users{$b->{uid}}->{login},
            $users{$b->{uid}}->{email},
            $managers{$b->{ManagerUID}||0}->{login} || '-',
            $managers{$b->{ManagerUID}||0}->{fio} || '-',
            $agencies{$b->{AgencyUID}||0}->{login} || '-',
            $agencies{$b->{AgencyUID}||0}->{fio} || '-',
            $agencies{$b->{AgencyUID}||0}->{email} || '-',
        ]);
    }
    {
        local $| = 1; print ".";
    }
}

$workbook->close();

print "\nResult rows count =>".$row_num."\n";
