#!/usr/bin/perl

=head1 USAGE

    LOG_TEE=1 ./protected/one-shot/63976_migrate_resources_video_to_creatives.pl [options...] <mapping filename>

=head1 DESCRIPTION

    Читает из файла маппинг ClientID, media_resource_id => creative_id и переделывает все старые видеодополнения
    записанные в banner_resources в новые видеодополнения выполненные в виде креативов

    Параметры
        <mapping file> - путь к файлу с маппингом

    Опции:
        --help - справка
        --check-only - только проверить маппинг из файла, ничего не записывать в БД
        --incomplete-mapping    - подразумевать что маппинг не полный, и не ругаться на его отсутствие для каких-то найденных старых видео

    Опции только для тестирования на бете:
        --create-fake-creatives - в маппинге указаны фейковые creative_id, проверить что настоящих креативов с такими id не существует,
                                    записать их в perf_creatives и выйти
        --clean-fake-creatives  - удалить все креативы указанные в маппинге из perf_creatives
        --client-id - обрабатывать креативы только указанных клиентов (можно указать несколько раз)
                      включает опцию --incomplete-mapping

=cut

use Direct::Modern;

use JSON;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::ListUtils qw/enumerate_iter/;
use Yandex::Retry qw/relaxed_guard/;

use my_inc '../..';

use EnvTools;
use ScriptHelper;
use Settings;

my $check_only = 0;
my $create_fake_creatives = 0;
my $clean_fake_creatives = 0;
my $incomplete_mapping = 0;
my @client_ids;
extract_script_params(
    'check-only' => \$check_only,
    'create-fake-creatives' => \$create_fake_creatives,
    'clean-fake-creatives' => \$clean_fake_creatives,
    'incomplete-mapping' => \$incomplete_mapping,
    'client-id=i@' => \@client_ids,
);
if (@client_ids) {
    $incomplete_mapping = 1;
}

my $mapping_filename = $ARGV[0];
unless ($mapping_filename && -f $mapping_filename) {
    usage();
}
if (!is_beta()) {
    if ($create_fake_creatives) {
        $log->die("--fake-creatives is beta only!");
    }
    if ($clean_fake_creatives) {
        $log->die("--clean-fake-creatives is beta only!");
    }
    if (@client_ids) {
        $log->die("--client-id is beta only!");
    }
}

my $SLEEP_COEF = 1;

$log->out('START');

my $mapping = read_mapping($mapping_filename, \@client_ids);

if ($create_fake_creatives) {
    create_fake_creatives($mapping);
    $log->out('FINISH');
    exit;
} elsif ($clean_fake_creatives) {
    clean_fake_creatives($mapping);
    $log->out('FINISH');
    exit;
} else {
    unless (check_creative_mapping($mapping)) {
        # маппинг будет проверяться на бете (в roprod конфигурации), и до тех пор, пока маппинг не сойдется, в проде запускать не будем
        # поэтому ок тут просто упасть
        $log->die("mapping creative check failed");
    }
}

unless ($check_only) {
    migrate_videos($mapping);
}

$log->out('FINISH');

sub read_mapping {
    my ($mapping_filename, $client_ids) = @_;
    my $result = {};

    open(my $fh, "<:utf8", $mapping_filename);
    unless ($fh) {
        $log->die("can't open $mapping_filename");
    }

    my $clients = @$client_ids ? { map { $_ => undef } @$client_ids } : undef;
    while (<$fh>) {
        chomp;
        my ($client_id, $media_resource_id, $creative_id) = $_ =~ /^(\d+)\s+(\d+)\s+(\d+)$/;

        unless ($creative_id) {
            $log->die("bad row format '$_' at $.");
        }

        next if defined $clients && !exists $clients->{$client_id};

        if (defined($result->{$client_id}{$media_resource_id})) {
            $log->die("double mapping for client $client_id media_resource_id $media_resource_id at $.");
        }

        $result->{$client_id}{$media_resource_id} = $creative_id;
    }
    close($fh);

    return $result;
}

sub create_fake_creatives {
    my ($mapping) = @_;

    my $fake_creatives_already_exist = 0;
    foreach_shard(ClientID => [keys %$mapping], sub {
        my ($shard, $client_ids) = @_;
        my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        my $fake_creatives = [ map { values %{$mapping->{$_}} } @$client_ids ];

        my $existing_fake_creatives = get_one_column_sql(PPC(shard => $shard), ["SELECT creative_id FROM perf_creatives", WHERE => {creative_id => $fake_creatives}]);
        if (@$existing_fake_creatives) {
            $fake_creatives_already_exist = 1;
            $log->out("some creatives are not fake!", $existing_fake_creatives);
        }
    });

    if ($fake_creatives_already_exist) {
        $log->out("not creating any creatives");
        return;
    }

    foreach_shard(ClientID => [keys %$mapping], sub {
        my ($shard, $client_ids) = @_;
        my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        # [creative_id, ClientID, creative_type, stock_creative_id]
        my @mass_insert_values;

        for my $c (@$client_ids) {
            push @mass_insert_values, map { [$_, $c, 'video_addition', $_] } values %{$mapping->{$c}};
        }
        my $inserted = do_mass_insert_sql(PPC(shard => $shard), "INSERT INTO perf_creatives(creative_id, ClientID, creative_type, stock_creative_id) VALUES %s", \@mass_insert_values);
        $log->out("created $inserted fake creatives");
    });
}

sub clean_fake_creatives {
    my ($mapping) = @_;

    my $fake_creatives_already_exist = 0;
    foreach_shard(ClientID => [keys %$mapping], sub {
        my ($shard, $client_ids) = @_;
        my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        my $fake_creatives = [ map { values %{$mapping->{$_}} } @$client_ids ];

        my $deleted = do_delete_from_table(PPC(shard => $shard), 'perf_creatives', where => {creative_id => $fake_creatives});
        $log->out("deleted $deleted fake creatives");
    });
}

sub check_creative_mapping {
    my ($mapping) = @_;

    # проверяем, что все creative_id из маппинга существуют в perf_creatives и принадлежат указанным ClientID
    my $creative_check_ok = 1;
    foreach_shard(ClientID => [keys %$mapping], sub {
        my ($shard, $client_ids) = @_;
        my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        my %expected_creative_clients;
        for my $c (@$client_ids) {
            $expected_creative_clients{$_} = $c for values %{$mapping->{$c}};
        }

        my $creatives = get_hashes_hash_sql(PPC(shard => $shard), [
               "SELECT creative_id, ClientID, creative_type
                FROM perf_creatives",
                WHERE => {
                    creative_id => [keys %expected_creative_clients],
                },
            ]);

        for my $creative_id (keys %expected_creative_clients) {
            my $expected_client_id = $expected_creative_clients{$creative_id};

            my $creative = $creatives->{$creative_id};
            unless (defined($creative)) {
                $creative_check_ok = 0;
                $log->out("creative $creative_id not found!");
                next;
            }

            my $actual_client_id = $creative->{ClientID};
            unless ($actual_client_id == $expected_client_id) {
                $creative_check_ok = 0;
                $log->out("creative $creative_id has bad ClientID! expected $expected_client_id, got $actual_client_id");
                next;
            }

            my $actual_creative_type = $creative->{creative_type};
            unless ($actual_creative_type eq 'video_addition') {
                $creative_check_ok = 0;
                $log->out("creative $creative_id has bad type! expected video_addition, got $actual_creative_type");
            }
        }
    });

    return $creative_check_ok;
}

sub migrate_videos {
    my ($mapping) = @_;

    foreach_shard(ClientID => [keys %$mapping], sub {
        my ($shard, $client_ids) = @_;
        my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        my $last_chunk_max_id = 0;

        # цикл до тех пор, пока запрос не пройдет по всем строчкам из banner_resources
        while (1) {
            my $relax = relaxed_guard(times => $SLEEP_COEF);

            my $add_cond = {};
            if ($incomplete_mapping) {
                $add_cond = {
                    'c.ClientID' => $client_ids,
                };
            }

            # выбираем баннеры у которых есть старое видео
            # ClientID, used_resources нужны для маппинга
            # resource_id для записи в лог перед удалением
            # pid и cid для новой записи в banners_performance
            # заодно проверяем, есть ли у баннера *уже* запись в banners_performance. Если есть, будем ругаться
            # statusModerate и statusActive для вычисления статуса модерации нового видео
            my $banners = get_all_sql(PPC(shard => $shard), [
                   "SELECT c.ClientID, br.resource_id, br.used_resources, br.bid, b.pid, b.cid, bp.banner_creative_id, b.statusModerate, b.statusActive
                    FROM banner_resources br
                        STRAIGHT_JOIN banners b ON b.bid = br.bid
                        STRAIGHT_JOIN campaigns c ON c.cid = b.cid
                        LEFT JOIN banners_performance bp ON bp.bid = br.bid
                    ",
                    WHERE => {
                        'br.resource_id__int__gt' => $last_chunk_max_id,
                        %$add_cond,
                    },
                   "ORDER BY br.resource_id
                    LIMIT 1000",
                ]);
            last unless @$banners;
            $last_chunk_max_id = $banners->[-1]{resource_id};

            my @new_banners_performance_records;
            my @records_to_delete_from_banner_resources;
            my @bids_to_auto_accept;
            for my $banner (@$banners) {
                my $bid = $banner->{bid};

                if (!defined($banner->{ClientID})) {
                    $log->out("banner $bid has no campaigns.ClientID");
                    next;
                }

                if ($banner->{banner_creative_id}) {
                    $log->out("banner $bid has both banner_resources and banners_performance");
                    next;
                }

                my $media_resources = from_json($banner->{used_resources});
                if (@$media_resources != 1) {
                    $log->out("banner $bid has ".scalar(@$media_resources)." used resources. Expected 1");
                    next;
                }
                my $mrid = $media_resources->[0];

                my $creative_id = $mapping->{$banner->{ClientID}}{$mrid};
                unless (defined($creative_id)) {
                    unless ($incomplete_mapping) {
                        $log->out("banner $bid has no creative mapping! ClientID ".$banner->{ClientID}.", media_resource_id $mrid");
                    }
                    next;
                }

                my $v_status_moderate = 'Ready';
                if ($banner->{statusModerate} eq 'Yes') {
                    push @bids_to_auto_accept, [$bid];
                } elsif ($banner->{statusModerate} eq 'No' && $banner->{statusActive} eq 'No') {
                    $v_status_moderate = 'Yes';
                } elsif ($banner->{statusModerate} eq 'New') {
                    $v_status_moderate = 'New';
                }
                push @new_banners_performance_records, {%{hash_cut($banner, qw/bid pid cid/)}, creative_id => $creative_id, statusModerate => $v_status_moderate};
                push @records_to_delete_from_banner_resources, hash_cut($banner, qw/resource_id bid used_resources/);
            }

            next unless @new_banners_performance_records;

            # массив для do_mass_insert_sql
            # столбцы идут в таком порядке: banner_creative_id, bid, pid, cid, creative_id, statusModerate
            my @mass_insert_values;
            my $banner_creative_ids = get_new_id_multi('banner_creative_id', scalar(@new_banners_performance_records));
            my $it = enumerate_iter($banner_creative_ids, \@new_banners_performance_records);
            while (my ($i, $banner_creative_id, $r) = $it->()) {
                push @mass_insert_values, [$banner_creative_id, @$r{qw/bid pid cid creative_id statusModerate/}];
            }

            do_in_transaction {
                $log->out("deleting from banner_resources: ".to_json(\@records_to_delete_from_banner_resources));
                do_delete_from_table(PPC(shard => $shard), 'banner_resources', where => {bid => [ map { $_->{bid} } @records_to_delete_from_banner_resources ]});

                $log->out("inserting into banners_performance records with ids: ".to_json([map { $_->[0] } @mass_insert_values]));
                do_mass_insert_sql(PPC(shard => $shard), 'INSERT IGNORE INTO banners_performance(banner_creative_id, bid, pid, cid, creative_id, statusModerate) VALUES %s', \@mass_insert_values);

                if (@bids_to_auto_accept) {
                    $log->out("auto accepting bids on moderation: ".to_json([map { $_->[0] } @bids_to_auto_accept]));
                    do_mass_insert_sql(PPC(shard => $shard), 'INSERT IGNORE INTO auto_moderate(bid) VALUES %s', \@bids_to_auto_accept);
                }
            }
        }

        $log->out("Done");
    });
}
