#!/usr/bin/perl

=head1 DESCRIPTION

    Удаление ретаргетингов из всех фильтров смарт фидов.

    Производится с помощью директовских моделей, поэтому статусы синхронизации с БК сбрасываются не через
    ленивую очередь, а напрямую.
    Однако в данном случае это не должно быть большой проблемой, т.к. будет затронуто не очень большое количество
    баннеров/групп:
    $ direct-sql pr:ppc:all '
        select count(*) from bids_performance join banners using (pid)
        where ret_cond_id is not null and is_deleted = 0' -b | grep -v count | jq -s add
    118399

=head1 USAGE

    Запуск в холостом режиме, без записи в базу
    LOG_TEE=1 ./protected/one-shot/75389-clear-perf-filter-retargetings.pl --dry-run

    Обрабатывать фильтры пачками не более 10 штук. По умолчанию - 100
    LOG_TEE=1 ./protected/one-shot/75389-clear-perf-filter-retargetings.pl --chunk-size 10

    Обрабатывать фильтры только из группы 12345678
    При этом известные поломанные фильтры из $KNOWN_BROKEN_FILTERS не пропускаются
    LOG_TEE=1 ./protected/one-shot/75389-clear-perf-filter-retargetings.pl --pid 12345678

    Обрабатывать фильтры только из шарда 5
    LOG_TEE=1 ./protected/one-shot/75389-clear-perf-filter-retargetings.pl --shard-id=5

=cut

use Direct::Modern;
use my_inc "../..";

use List::MoreUtils qw/uniq/;

use Settings;
use ScriptHelper;

use Yandex::DBTools;
use Yandex::DBShards;
use Direct::AdGroups2::Performance;
use Direct::PerformanceFilters;
use ShardingTools;

# 57372 - фильтр, созданный для фида с типом GoogleMerchant. Потом тип фида поменялся на YandexMarket,
# и условие перестало проходить валидацию. Тикет DIRECT-76588
my $KNOWN_BROKEN_FILTERS = [57372];

my $CHUNK_SIZE = 100;
my $DRY_RUN = 0;
my $only_pid;
my $only_shard;

extract_script_params(
    'chunk-size=i' => \$CHUNK_SIZE,
    'dry-run' => \$DRY_RUN,
    'pid=i' => \$only_pid,
    'shard-id=i' => \$only_shard,
);

if ($only_pid && $only_shard) {
    die "can't specify both pid and shard-id\n";
}

my @shards;
if ($only_pid) {
    @shards = get_shard(pid => $only_pid);
    $log->out("working only on pid $only_pid");
} elsif ($only_shard) {
    @shards = ($only_shard);
    $log->out("working only on shard $only_shard");
} else {
    @shards = ppc_shards();
}

for my $shard (@shards) {
    do_shard($shard);
}

sub do_shard {
    my ($shard) = @_;

    my $shard_prefix_guard = $log->msg_prefix_guard("[shard=$shard]");
    my %only_pid_cond;
    my %skip_broken_cond;
    if ($only_pid) {
        $only_pid_cond{pid} = $only_pid;
    } else {
        $skip_broken_cond{perf_filter_id__not_in} = $KNOWN_BROKEN_FILTERS;
    }


    $log->out('start');

    my $last_perf_filter_id = 0;
    while (1) {
        my $perf_filter_ids = get_one_column_sql(PPC(shard => $shard), [
                "SELECT perf_filter_id
                FROM bids_performance",
                WHERE => {
                    perf_filter_id__gt => $last_perf_filter_id,
                    ret_cond_id__is_not_null => 1,
                    is_deleted => 0,
                    %only_pid_cond,
                    %skip_broken_cond,
                },
                "ORDER BY perf_filter_id
                LIMIT $CHUNK_SIZE"
            ]);
        last if !@$perf_filter_ids;

        $last_perf_filter_id = $perf_filter_ids->[-1];

        # здесь мы иногда можем падать, т.к. в продакшене встречаются битые фильтры
        # у которых тип фида не совместим с условием в фильтре
        # Такое случается, если условие было создано с фидом одного типа, который потом был изменен
        # тикет DIRECT-76588
        my $filters = eval {Direct::PerformanceFilters->get_by(perf_filter_id => $perf_filter_ids, shard => $shard)->items_by('id')};
        if (!$filters) {
            my $id_range;
            if (@$perf_filter_ids > 1) {
                $id_range = $perf_filter_ids->[0].' - '.$perf_filter_ids->[-1];
            } else {
                $id_range = $perf_filter_ids->[0];
            }
            $log->out("WARNING: error when fetching PerformanceFilter models from database. perf_filter_id in [$id_range] are skipped!: $@");
            next;
        }

        my @adgroup_ids = uniq map { $_->adgroup_id() } values %$filters;
        my $adgroups = Direct::AdGroups2::Performance->get_by(adgroup_id => \@adgroup_ids)->items_by('id');

        $log->out(($DRY_RUN ? 'NOT ' : '').'deleting retargetings from perf filters', [keys %$filters]);
        for my $filter (values %$filters) {
            my $adgroup = $adgroups->{$filter->adgroup_id()};
            if (!$adgroup) {
                $log->out("WARNING: adgroup for filter ".$filter->id()." not found (".$filter->adgroup_id().")! filter is skipped");
                next;
            }
            $filter->adgroup($adgroup);
            $filter->ret_cond_id(undef);
        }

        if (!$DRY_RUN) {
            Direct::PerformanceFilters->new(items => [values %$filters])->update();
        }
    }

    $log->out('finish');
}
