#!/usr/bin/perl

=head1 DESCRIPTION

    Восстановление поисковых ставок, сброшенных в 0, по файлу.
    В файле должен быть список JSON такого формата

        {"old_price":"20.00","cid":31425896,"bid_id":11598811886,"pid":3046328567}
        {"old_price":"20.00","cid":31425896,"bid_id":11607215499,"pid":3046328567}
        {"old_price":"20.00","cid":31425896,"bid_id":11607215500,"pid":3046328567}
        {"old_price":"20.00","cid":31425896,"bid_id":11607215501,"pid":3046328567}

    Поле old_price должно содержать значение ставки, которое было до поломки

    Ставка восстанавливается только если текущая поисковая ставка равна нулю, и стратегия
    либо только в сетях, либо автобюджетная.

    Если стратегия автобюджетная, ставка меняется в таблице bids_manual_prices
    Если стратегия ручная в сетях, ставки меняются в таблицах bids и bids_base.

    При измененнии ставок в bids и bids_base, у них сбрасывается статус синхронизации с БК

=head1 USAGE

    Запуск в холостом режиме, без записи в базу
    LOG_TEE=1 ./protected/one-shot/78654-restore-bids.pl --dry-run --file ~/bids_to_restore.txt

    Запуск с записью в базу
    LOG_TEE=1 ./protected/one-shot/78654-restore-bids.pl --file ~/bids_to_restore.txt

    Указание одной ставки через параметры
    LOG_TEE=1 ./protected/one-shot/78654-restore-bids.pl --test-bid '{"old_price":"20.00","cid":31425896,"bid_id":11607215501,"pid":3046328567}'

=cut

use Direct::Modern;
use my_inc "../..";

use List::MoreUtils qw/natatime all uniq/;
use Path::Tiny;
use JSON;

use Settings;
use ScriptHelper;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry qw/relaxed_guard/;

my $DRY_RUN = 0;

extract_script_params(
    'file=s' => \my $filename,
    'test-bid=s' => \my $test_bid,
    'dry-run' => \$DRY_RUN,
);

my @bids;
if ($filename) {
    @bids = path($filename)->lines({chomp => 1});
} elsif($test_bid) {
    @bids = $test_bid;
} else {
    usage();
}
my $bids_it = natatime 1_000, @bids;

$log->out("START");
$log->out("working on ".scalar(@bids)." bids");
while (my @bids_chunk = $bids_it->()) {
    my $relaxed_guard = relaxed_guard times => 3;

    @bids_chunk = map { from_json($_) } @bids_chunk;
    for my $bid (@bids_chunk) {
        if (!(all { $_ =~ /^(cid|pid|bid_id|old_price)$/ } keys %$bid) || scalar(keys %$bid) != 4) {
            die "malformed old bid ".to_json($bid);
        }
    }
    foreach_shard cid => \@bids_chunk, sub {
        my ($shard, $shard_bids) = @_;

        restore_bids_in_shard($shard, $shard_bids);
    }
}
$log->out("FINISH");

sub restore_bids_in_shard {
    my ($shard, $shard_bids) = @_;
    my $shard_prefix_guard = $log->msg_prefix_guard("[shard=$shard".($DRY_RUN ? ', dry run' : '').']');

    my @cids = uniq map { $_->{cid} } @$shard_bids;
    my $old_bids = { map { $_->{bid_id} => $_ } @$shard_bids };
    my @bid_ids = keys %$old_bids;

    my $rows = get_all_sql(PPC(shard => $shard), ["
        SELECT
            c.ClientID,
            c.cid,
            c.platform,
            c.autobudget,
            bi.id AS bid_id,
            bi.price AS bid_price,
            bim.price AS manual_bid_price
        FROM
            campaigns AS c
            JOIN bids AS bi ON bi.cid = c.cid
            LEFT JOIN bids_manual_prices bim ON bim.cid = bi.cid AND bim.id = bi.id",
        WHERE => {
            'c.cid' => \@cids,
            'bi.id' => \@bid_ids,
        }
    ]);

    my @manual_prices_to_restore;
    my @prices_to_restore;
    for my $row (@$rows) {
        my $bid_id = $row->{bid_id};
        my $client_id = $row->{ClientID};
        my $old_bid = $old_bids->{$bid_id};
        my $old_price = $old_bid->{old_price};

        if ($row->{autobudget} ne 'No') {
            if ($row->{manual_bid_price} == 0) {
                $log->out("restoring manual price for bid $bid_id to $old_price");
                push @manual_prices_to_restore, $old_bid;
            } elsif ($row->{manual_bid_price} eq $old_price) {
                $log->out("bid $bid_id was already restored, ClientID $client_id");
            } else {
                $log->out("ClientID $client_id changed price for bid $bid_id");
            }
        } else {
            if ($row->{bid_price} == 0) {
                if ($row->{platform} eq 'context') {
                    $log->out("restoring price for bid $bid_id to $old_price");
                    push @prices_to_restore, $old_bid;
                } else {
                    $log->out("weird zero price for bid $bid_id, platform ".$row->{platform}.", ClientID $client_id");
                }
            } elsif ($row->{bid_price} eq $old_price) {
                $log->out("bid $bid_id was already restored, ClientID $client_id");
            } else {
                $log->out("ClientID $client_id changed price for bid $bid_id");
            }
        }
    }

    if (!$DRY_RUN) {
        do_in_transaction sub {
            if (@manual_prices_to_restore) {
                do_mass_update_sql(PPC(shard => $shard), 'bids_manual_prices', 'id',
                    {
                        map { $_->{bid_id} => {price => $_->{old_price}} } @manual_prices_to_restore
                    },
                    where => {
                        cid => \@cids,
                        price => 0,
                    }
                );
            }

            if (@prices_to_restore) {
                do_mass_update_sql(PPC(shard => $shard), 'bids', 'id',
                    {
                        map {
                            $_->{bid_id} => {
                                price => $_->{old_price},
                                statusBsSynced => 'No'
                            }
                        } @prices_to_restore
                    },
                    where => {
                        cid => \@cids,
                        price => 0,
                    }
                );

                do_mass_update_sql(PPC(shard => $shard), 'bids_base', 'bid_id',
                    {
                        map {
                            $_->{bid_id} => {
                                price => $_->{old_price},
                                statusBsSynced => 'No'
                            }
                        } @prices_to_restore
                    },
                    where => {
                        cid => \@cids,
                        price => 0,
                    }
                );
            }
        }
    }
}
