#!/usr/bin/perl

=pod

=encoding utf8

    $Id:$

=head1 NAME

    85634-fill-adgroups_cpm_banner-criterion_type.pl

=head1 SYNOPSIS

    # работаем по одной кампании, в заданном шарде
    LOG_TEE=1 ./protected/one-shot/85634-fill-adgroups_cpm_banner-criterion_type.pl --sleep-coef 0 --sleep-seconds  0--shard=15 --cid=123556

    # пишем логи, но ничего не меняем
    LOG_TEE=1 ./protected/one-shot/85634-fill-adgroups_cpm_banner-criterion_type.pl --dry-run --chunk-size=100

    # с минимального cid-а ускоряет работу скрипта, т.к. по cid-ам есть индекс, а по типу нет и без минимального
    # cid-а получаем фулскан по всей бд
    LOG_TEE=1 ./protected/one-shot/85634-fill-adgroups_cpm_banner-criterion_type.pl --min-cid 12345

=head1 DESCRIPTION

    Скрипт для первичного заполнения таблицы adgroups_cpm_banner на момент пока там всего две колонки.

    Можно перезапускать.

    После работы скрипта не плохо проверить что в adgroups_cpm_banner не появились подвисшие записи
    direct-sql pr:ppc:all 'select g_cpm.pid from adgroups_cpm_banner g_cpm left join phrases g using(pid) where g.pid is null'

    Т.к. каждая последующая выборка делается с последнего pid-а то, возможно
    ситуация когда будут обработаны не все группы, если в момент работы скрипта
    таковые будут созданы без записи в adgroups_cpm_banner, поэтому после
    запуска результат работы скрипта стоит проверить запуском с --dry-run

    Запрос для проверки результата:

    direct-sql pr:ppc:all '
            SELECT
                c.cid, g.pid,
                IF(ret.pid IS NULL, "keyword", "user_profile") AS expected_type,
                g_cb.criterion_type AS actual_type
            FROM campaigns c
                JOIN phrases g using (cid)
                LEFT JOIN bids_retargeting ret ON ret.pid = g.pid
                LEFT JOIN adgroups_cpm_banner g_cb ON g_cb.pid = g.pid
            WHERE c.type IN ("cpm_banner", "cpm_deals") and adgroup_type = "cpm_banner"
                AND ( IF(ret.pid IS NULL, "keyword", "user_profile") <> g_cb.criterion_type OR g_cb.criterion_type IS NULL )
            LIMIT 25'

=head1 METHODS

=cut

use Direct::Modern;

use Time::HiRes;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Retry;

use my_inc '../..', for => 'protected';
use ScriptHelper 'Yandex::Log' => [ date_suf => '%Y%m%d', auto_rotate => 1, tee => $ENV{LOG_TEE}, lock => 1 ];
use Settings;
use ShardingTools;

my $MIN_CID;
my $CHUNK_SIZE = 1000;
my $SLEEP_COEF = 2;
my $SLEEP_SECONDS = 10;
my $CID_ARG;
my $DRY_RUN;
my $SHARD;

extract_script_params(
    'sleep-seconds=i' => \$SLEEP_SECONDS,
    'sleep-coef=i' => \$SLEEP_COEF,
    'dry-run' => \$DRY_RUN,
    'cid=i' => \$CID_ARG,
    'min-cid=i' => \$MIN_CID,
    'shard=i' => \$SHARD,
    'chunk=i' => \$CHUNK_SIZE,
);

$log->die("you cant use both cid and min-cid arguments") if $CID_ARG && $MIN_CID;

$log->out('start');

my $shards = [ $SHARD // ppc_shards() ];

foreach_shard_parallel shard => $shards, sub {
    my ($shard) = @_;
    do_shard($shard);
};

$log->out('finish');

sub do_shard {
    my ($shard) = @_;

    my $db = PPC( shard => $shard );
    my $shard_prefix_guard = $log->msg_prefix_guard("shard=$shard");
    $log->out('start');

    my $last_pid;
    while(1) {
        my $groups = get_all_sql($db, [
            'SELECT
                c.cid, g.pid, 
                IF(kw.pid IS NULL, 0, 1) as has_keywords,
                IF(ret.pid IS NULL, 0, 1) as has_segments,
                IF(kw_arc.pid IS NULL, 0, 1) as has_keywords_archived
            FROM campaigns c
                JOIN phrases g using (cid)
                LEFT JOIN bids kw ON kw.pid = g.pid
                LEFT JOIN bids_retargeting ret ON ret.pid = g.pid
                LEFT JOIN bids_arc kw_arc ON kw_arc.cid = c.cid AND kw_arc.pid = g.pid
                LEFT JOIN adgroups_cpm_banner g_cb ON g_cb.pid = g.pid',
            WHERE => {
                ( $MIN_CID ? ('g.cid__ge' => $MIN_CID) : () ),
                ( $CID_ARG ? ('g.cid' => $CID_ARG) : () ),
                ( $last_pid ? ('g.pid__gt' => $last_pid) :() ),
                type => [qw/cpm_banner cpm_deals/],
                adgroup_type => "cpm_banner",
                'g_cb.pid__is_null' => 1
            },
            "GROUP BY g.pid",
            "ORDER BY g.pid",
            "LIMIT $CHUNK_SIZE"
        ]);

        if (!@$groups) {
            $log->out("nothing selected quiting..");
            last;
        }

        $last_pid = $groups->[$#$groups]->{pid};
        $log->out("last pid in chunk " . $last_pid);

        my ($keyword_adgroups, $profile_adgroups, $undefined_adgroups, $without_targeting_adgroups) = ([], [], [], []);
        foreach my $g (@$groups) {
            my $status = $g->{has_keywords} + $g->{has_keywords_archived} + $g->{has_segments};
            if ($status == 1) {
                if ($g->{has_segments}) {
                    push @$profile_adgroups, $g;
                } else {
                    push @$keyword_adgroups, $g;
                }
            } elsif ($status == 0) { # у группы нет ни таргетингов, ни записи в adgroups_cpm_banner, такого быть не должно
                push @$without_targeting_adgroups, $g;
            } elsif ($status > 1) { # у группы таргетинги разных типов - не косистентное состяние
                push @$undefined_adgroups, $g;
            }
        }

        $log->out('Broken adgroups with no targetings and no subtype', $without_targeting_adgroups) if @$without_targeting_adgroups;
        $log->out('Broken adgroups with targetings of several types', $undefined_adgroups) if @$undefined_adgroups;
        $log->die("Amount of broken records exceeds CHUNK_SIZE=$CHUNK_SIZE, script can't be finished properly") if (@$without_targeting_adgroups+@$undefined_adgroups) >= $CHUNK_SIZE;

        if(!@$keyword_adgroups && !@$profile_adgroups) {
            $log->out('no more adgroups to fill, quiting..');
            last;
        }

        my $pids = [ map { $_->{pid} } grep { $_->{has_keywords} } @$keyword_adgroups ];
        my $arc_pids = [ map { $_->{pid} } grep { $_->{has_keywords_archived} } @$keyword_adgroups ];
        my $user_profile_pids = [ map { $_->{pid} } @$profile_adgroups];

        $log->out('Adgroups with keywords targeting', $pids) if @$pids;
        $log->out('Archived adgroups with keywords targeting', $arc_pids) if @$arc_pids;
        $log->out('Adgroups with user profiles targeting', $user_profile_pids) if @$user_profile_pids;

        if(!$DRY_RUN) {
            $log->out("filling keyword based adgroups");
            fill_adgroups_cpm_banner($db, 'keyword' => $pids, 'bids kw USING(pid)') if @$pids;
            $log->out("filling keyword based archived adgroups");
            fill_adgroups_cpm_banner($db, 'keyword' => $arc_pids, 'bids_arc kw_arc ON g.cid = kw_arc.cid AND g.pid = kw_arc.pid') if @$arc_pids;
            $log->out("filling user profile based adgroups");
            fill_adgroups_cpm_banner($db, 'user_profile' => $user_profile_pids, 'bids_retargeting ret ON ret.pid = g.pid') if @$user_profile_pids;
        }

        # спим все равно, не смотря на relaxed_guard в меняющий методах, чтобы не очень агресивно селектить
        # спим фиксированное количество секунд, потом что селекты могут
        # драматично отличатся по времени - полчаса первый и секунды на прогретой базе
        Time::HiRes::sleep($SLEEP_SECONDS) if $SLEEP_SECONDS;
    }

    $log->out('finish');
}

sub fill_adgroups_cpm_banner {
    my ($db, $type, $pids, $join) = @_;
    my $relax = relaxed_guard(times => $SLEEP_COEF);
    my $rows_affected = do_sql($db, [
            "INSERT IGNORE INTO adgroups_cpm_banner (pid, criterion_type)
            SELECT g.pid, '$type' as type FROM phrases g JOIN $join",
            WHERE => { 'g.pid' => $pids }]) // 0;
    $log->out("\t..$rows_affected rows inserted into adgroups_cpm_banner as $type");
}

1;
