#!/usr/bin/perl

=head1 NAME

    DIRECT-101306: исправление statusModerate для смарт-групп

=head1 DESCRIPTION

    Проставляем всем смарт-группам, которые не промодерированы и не черновики
    статус промодерированности (statusModerate=Yes, statusPostModerate=Yes).

    Принимает параметры:

    --client-id - ClientID клиента, чьим группам необходимо поправить статусы модерации
    --file - файл в tsv с одной колонкой, содержащей ClientID; скрипт исправит статусы модерации у групп этих клиентов
    --dry-run - не изменять данные, лишь написать в лог, какие данные будут изменены

=cut

use Direct::Modern;

use Yandex::DBTools;


use my_inc '../..';

use File::Slurp;
use ScriptHelper;
use Settings;
use Yandex::DBShards qw/sharded_chunks/;
use Yandex::Retry qw/relaxed_guard/;

$log->out('START');

my $SLEEP_COEF = 1;

extract_script_params(
    'file=s' => \my $client_ids_file,
    'client-id=i' => \my $only_client_id,
    'dry-run' => \my $DRY_RUN,
);

my @client_ids;
if ($client_ids_file) {
    @client_ids = read_file($client_ids_file, chomp => 1);
} elsif ($only_client_id) {
    @client_ids = ($only_client_id);
} else {
    die "please specify either --file or --client-id";
}

for my $chunk (sharded_chunks ClientID => \@client_ids) {
    my ($shard, $client_ids) = ($chunk->{shard}, $chunk->{ClientID});
    process_shard_clientIds($shard, $client_ids);
}

sub process_shard_clientIds {
    my $shard = shift;
    my $client_ids = shift;

    $log->out({shard => $shard, client_ids => $client_ids});
    my $adgroups_for_update = get_all_sql(PPC(shard => $shard), [
        q/SELECT p.pid, p.statusModerate, p.statusPostModerate
        FROM campaigns c
        JOIN phrases p ON (p.cid = c.cid)/,
        where => {
            'c.type'           => 'performance',
            'c.ClientID'       => $client_ids,
            'p.adgroup_type'   => 'performance',
            'p.statusModerate__not_in' => [ qw/New Yes/ ]
        }]);

    for my $adgroup (@$adgroups_for_update) {
        $log->out({ adgroup => $adgroup });

        my $num_rows = 0;
        if (defined $DRY_RUN && $DRY_RUN) {
            $log->out("Skip update for dry run");
        } else {
            my $relax = relaxed_guard(times => $SLEEP_COEF);
            $num_rows = do_update_table(PPC(shard => $shard), 'phrases',
               { statusModerate => 'Yes', statusPostModerate => 'Yes', statusBsSynced => 'No' },
               where => {
                   pid                => $adgroup->{pid},
                   statusModerate     => $adgroup->{statusModerate},
                   statusPostModerate => $adgroup->{statusPostModerate}
               });
        }

        if ($num_rows > 0) {
            $log->out("AdGroup ".$adgroup->{pid}." updated");
        } else {
            $log->out("AdGroup ".$adgroup->{pid}." skipped");
        }
    }
}

$log->out('FINISH');

