#!/usr/bin/perl

=head1 DESCRIPTION

Скрипт для поиска и перепривязки записей в maps, потерянных при копировании кампаний между клиентами, находящимися на разных шардах 

Параметры:

    --сid - идентификатор кампании, если нужно починить только связанные с ней адреса
    --shard-id  шард (по умолчанию - все)
=cut

use Direct::Modern;
use my_inc '../..';

use List::MoreUtils qw/uniq/;
use Try::Tiny qw/try catch/;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::ListUtils qw/chunks/;

use ShardingTools;
use ScriptHelper;
use Settings;

use CommonMaps;
use BS::ResyncQueue;


my $SLEEP_COEF = 1;
my $CHUNK_SIZE = 1000;

extract_script_params(
    'shard-id=i' => \my @shards,
    'cid=i' => \my @cids,
    'sleep-сoef=f' => \$SLEEP_COEF,
);

if (@cids) {
    my $shard_by_cid = get_shard_multi(cid => \@cids);

    @shards = uniq values %$shard_by_cid;

    die "Can't define shards by cids: ".(join(', ', @cids)) unless @shards;
}

@shards = ppc_shards()  if !@shards;

my @map_fields = qw/x y x1 y1 x2 y2/;

$log->out('START');

for my $shard (@shards) {
    $log->msg_prefix("Shard #$shard");

    my $rg = relaxed_guard times => $SLEEP_COEF;
    my $broken_aids = get_one_column_sql(PPC(shard => $shard),
    [q/SELECT DISTINCT aid FROM 
        (SELECT aid FROM addresses a LEFT JOIN maps m ON a.map_id = m.mid
            WHERE a.map_id > 0 AND m.mid is null
        UNION ALL
         SELECT aid FROM addresses aa LEFT JOIN maps ma ON aa.map_id_auto = ma.mid
            WHERE aa.map_id_auto > 0 AND ma.mid is null
        ) s/,
         @cids ? (q/JOIN vcards v ON (v.address_id = s.aid)/, WHERE => {'v.cid' => \@cids}) : (),
        'ORDER BY' => 'aid']);
    $log->out('Found '.(0+@$broken_aids).' broken addresses');

    for my $chunk (chunks($broken_aids, $CHUNK_SIZE)){
        repair_addresses($shard, $chunk);    
    }

    $log->msg_prefix(undef);
}


$log->out('FINISH');

sub repair_addresses {
    my ($shard, $aids) = @_;

    my $broken_banners = get_all_sql(PPC(shard => $shard),
        [q/SELECT b.bid, b.cid, a.aid, a.map_id, a.map_id_auto
            FROM addresses a 
             LEFT JOIN vcards v ON (v.address_id = a.aid)
             LEFT JOIN banners b ON (b.vcard_id = v.vcard_id)/,
        
        WHERE => {'a.aid' => $aids}]
    );

    my @map_ids = uniq grep {defined $_} map {$_->{map_id}, $_->{map_id_auto}} @$broken_banners;

    if (@map_ids) {
        my $stat;
        foreach my $row (@$broken_banners){
            $stat->{$row->{cid} // 0}++;
        }
        $log->out('Will be repaired: '.(join ', ', map {"campaign $_: $stat->{$_} banners"} sort {$a<=>$b} keys %$stat));
    }
    else {
        $log->out('Do nothing');
        return;
    }

    my $map_info = get_hashes_hash_sql(PPC(shard => 'all'), 
        ['SELECT mid,',sql_fields(@map_fields),' FROM maps', WHERE => {mid => \@map_ids} ]
    );

    my @wrong_map_ids;

    foreach my $map_row (values %$map_info){
        try {
            my $new_mid = CommonMaps::save_map_point(PPC(shard => $shard), hash_cut($map_row, @map_fields));
            $map_row->{mid} = $new_mid;
        }
        catch {
            my $error = shift;
            $log->out("Can't resolve new map id for mid: $map_row->{mid}, cause: $error");
            push @wrong_map_ids,  $map_row->{mid};
        };

        foreach my $mid (@wrong_map_ids){
            $log->out("Skipping mid: $mid");
            delete $map_info->{$mid};
        }
    }

    my $adresses_to_update;
    my $updated_banners;
    foreach my $row (@$broken_banners){
        next unless $row->{cid};
        my $row_changed;
        foreach my $field (qw/map_id map_id_auto/){
            my $map_id = $row->{$field};
            next unless $map_id;
            next unless $map_info->{$map_id};
            $adresses_to_update->{$row->{aid}} //= {};
            $log->out("address $row->{aid}: $field changed ".$row->{$field}.'->'.$map_info->{$map_id}->{mid})
                unless $adresses_to_update->{$row->{aid}}->{$field};
            $adresses_to_update->{$row->{aid}}->{$field} = $map_info->{$map_id}->{mid};

            unless ($row->{bid}){
                $log->out('banner missed for address '.$row->{aid});
                next;
            }
            $log->out("banner $row->{bid} will be resended to bs") unless $updated_banners->{$row->{bid}};
            $updated_banners->{$row->{bid}} //= $row->{cid};
        }
    }
    
    try {
        do_mass_update_sql(PPC(shard => $shard), addresses => aid => $adresses_to_update);
        BS::ResyncQueue::bs_resync([map { {
                cid => $updated_banners->{$_},
                bid => $_,
                priority => BS::ResyncQueue::PRIORITY_ONE_SHOT_RESYNC_BANNERS_ON_BR_FIX}
            } keys %$updated_banners]);
    }
    catch {
        my $error = shift;
        $log->out('Error while records updating - aid:[ '.(join ', ', keys %$adresses_to_update)." ], cause: $error" );
    };
}

