#!/usr/bin/perl

use my_inc '../..';

=head1 NAME

    add_clients_to_curr_convert.pl

=head1 DESCRIPTION

    Cтавит в очередь на конвертацию клиентов по списку логинов из переданного файла.
    Для агентских субклиентов необходимо указать логин агенства!
    Для попадания в очередь у клиентов должен быть проставлен тизер мультивалютности.

    Формат tsv файла:
    <login>
    <login>    <agency_login>

    Поле <agency_login> - нужно только для субклиентов агенств

=head1 RUNNING

    Запускать с параметрами:
        --data-file - путь к файлу с логинами клиентов
    Опционально:
        --convert-date - дата конвертации, по умолчанию будет ближайшая возможная

    ./protected/one-shot/add_clients_to_curr_convert.pl --date '2016-09-30 00:00:00' --data-file /home/gukserg/logins.txt

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils;
use Yandex::TimeCommon qw/unix2human mysql2unix/;
use Yandex::HashUtils qw/hash_merge/;

use Path::Tiny;
use List::MoreUtils qw/part/;

use Client;
use Client::ConvertToRealMoney;
use RBACElementary;
use ScriptHelper;
use Settings;

use Primitives;
use PrimitivesIds qw/get_clientids get_login2clientid/;

my ($data_path, $convert_date);
extract_script_params(
    'convert-date=s' => \$convert_date,
    'data-file=s' => \$data_path,
);

my $NO_AGENCY = 'NO_AGENCY';
# после того как поставили в очередь указанное количество клиентов на определенную дату
# ставим последующих клиентов на следующий день
my $NEXT_DAY_CONVERT_CHUNK = 5000;


$log->out('START');

my $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts();
if ($convert_date) {
    if (mysql2unix($convert_date) < $convert_start_ts) {
        # дата конвертации не должна быть раньше ближайшей возможной
        $log->die("Bad convert-date: $convert_date");
    } else {
        $convert_start_ts = mysql2unix($convert_date);
    }
}

my @lines = path($data_path)->lines({ chomp => 1}) or $log->die("Can't open data file $data_path: $@");

$log->out('Got '. scalar @lines .' logins from file');

# распределяем клиентам по агенствам, самоходы попадают в $NO_AGENCY
my %agency2logins;
for my $line (@lines) {
    my ($login, $agency_login) = split "\t", $line;
    push @{ $agency2logins{$agency_login || $NO_AGENCY} }, $login if $login;
}

my $total_clients_count = 0;
for my $agency_login (keys %agency2logins) {

    $log->out("Working with agency $agency_login") if $agency_login;

    my $logins = $agency2logins{$agency_login};
    my $agency_id;
    if ($agency_login ne $NO_AGENCY) {
        my $agency_uid = get_uid_by_login($agency_login) or $log->die("agency $agency_login not found");
        $agency_id = rbac_get_agency_clientid_by_uid( $agency_uid) or $log->die("agency $agency_login ClientID not found");
    }

    my $login2clientid = get_login2clientid(login => $logins);
    my $bad_logins = xminus($logins, [keys %$login2clientid]);
    if (defined $bad_logins && @$bad_logins) {
        $log->out({SKIP => 1, not_found_logins => $bad_logins}) ;
    }

    my $clientid2login = { reverse %$login2clientid };
    my $clids = [ keys $clientid2login ];

    my $clid2currencies = mass_get_client_currencies($clids);
    my ($good_clids, $bad_clids) = part { $clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' ? 0 : 1 } @$clids;
    if (defined $bad_clids && @$bad_clids) {
        $log->out({SKIP => 1, bad_work_currency_logins => [ map {$clientid2login->{$_}} @$bad_clids ]} ) ;
    }

    my $clients = get_all_sql(PPC(ClientID => $good_clids), ['
        SELECT ctfmt.ClientID
             , cfcc.country_region_id
             , cfcc.currency
             , u.email
        FROM clients_to_force_multicurrency_teaser ctfmt
        LEFT JOIN currency_convert_queue q ON ctfmt.ClientID = q.ClientID
        LEFT JOIN client_firm_country_currency cfcc ON ctfmt.ClientID = cfcc.ClientID
        LEFT JOIN users u ON ctfmt.ClientID = u.ClientID
     ', WHERE => {
            'ctfmt.ClientID' => SHARD_IDS,
            'q.ClientID__is_null' => 1,
        }, '
        GROUP BY ctfmt.ClientID
        HAVING
                COUNT(DISTINCT cfcc.country_region_id) = 1
            AND COUNT(DISTINCT cfcc.currency) = 1
    ']);

    my $not_selected_clids = xminus($good_clids, [ map { $_->{ClientID} } @$clients ]);
    if (@$not_selected_clids) {
        $log->out({SKIP => 1, not_selected_logins => [ map {$clientid2login->{$_}} @$not_selected_clids ]});
    }

    $log->out('Selected '. scalar @$clients .' clients for add to currency convert');

    if (@$clients) {
        my $client_ids = [ map { $_->{ClientID} } @$clients ];
        my $clid2nds = mass_get_client_NDS($client_ids, fetch_for_ynd_fixed_too => 1);
        my $clid2first_agency = Primitives::mass_get_client_first_agency($client_ids);
        my $clients_data = mass_get_clients_data($client_ids, [qw/allow_create_scamp_by_subclient/]);
        my $clid2chief_uid = {};
        for my $clids_chunk (chunks($client_ids, 5_000)) {
            # разбиваем на чанки, чтобы не падать по памяти при большом количестве клиентов
            hash_merge($clid2chief_uid, RBACElementary::rbac_get_chief_reps_of_clients($clids_chunk));
        }

        my @convert_requests;
        my $clients_count = 0;
        for my $client (@$clients) {
            my $client_id = $client->{ClientID};
            my $login = $clientid2login->{$client_id};

            my $prefix_guard = $log->msg_prefix_guard("[login=$login,ClientID=$client_id". ($agency_id ? ",agency=$agency_login" : '') ."]");

            # сдвигаем дату конвертации на один день на каждый чанк
            my $chunk_convert_start_ts = $convert_start_ts + 24 * 3600 * int($total_clients_count/$NEXT_DAY_CONVERT_CHUNK);
            my $convert_start_datetime = unix2human($chunk_convert_start_ts - 60);
            $log->out("convert_start_datetime $convert_start_datetime");

            my $client_nds = $clid2nds->{$client_id};
            my $client_currencies = get_client_currencies($client_id, allow_initial_currency => 0);
            my $error = Client::can_convert_to_real_money(
                ClientID => $client_id,
                NDS => $client_nds,
                client_currencies => $client_currencies,
                client_chief_uid => $clid2chief_uid->{$client_id},
                ignore_request_in_queue => 1,
            );
            if (defined $error) {
                $log->out("SKIP - cannot convert client into real currency: $error");
                next;
            }

            my $currency = $client->{currency};
            my $country = $client->{country_region_id};

            $clients_count++;
            $total_clients_count++;

            # единственность стран и валют и то, что они такие -- проверили в исходном SQL-запросе
            my @country_currencies = ({region_id => $country, currency => $currency});
            my $convert_type = Client::ConvertToRealMoney::get_convert_type($client_id, $currency, $client_nds, \@country_currencies);
            my $to_convert = {
                ClientID => $client_id,
                uid => undef, # SMS никому не пишем
                convert_type => $convert_type,
                new_currency => $currency,
                country_region_id => $country,
                # отдельные письма пишем только самоходам, агентству общее письмо не пишем
                email => $agency_id ? undef : $client->{email},
                start_convert_ts => $chunk_convert_start_ts,
            };
            $log->out($to_convert);
            push @convert_requests, $to_convert;
        }

        $log->out("Got $clients_count to add in currency convert queue");
        if (@convert_requests) {
            $log->out("Add to currency_convert_queue...");
            Client::ConvertToRealMoney::mass_queue_currency_convert(\@convert_requests, ignore => 1);
        }
    }
}

$log->out('FINISH');
