#!/usr/bin/perl

use my_inc '../..';

=head1 NAME

autofill-mandatory-group-names.pl - заполняет незаданные phrases.group_name автоматически сгенерированными значниями.

=head1 SYNOPSIS

Аргументы:
    --shard X - работать только с шардом X. Можно указывать опцию несколько раз для нескольких шардов.
    --shard-min-pid X:Y - на шарде X начинать обработку с phrases.pid=Y, а не сначала таблицы. Можно указывать несколько раз.
                          Таким образом можно продолжить обработку со старого места, указав для каждого шарда pid последней
                          записи из логов.

=head1 DESCRIPTION

Работает параллельно на всех шардах. Бъёт диапазон pid'ов на куски по 100_000 групп, один кусок обрабатывается за несколько секунд.

=cut

use Direct::Modern;

use List::MoreUtils qw/all/;
use Try::Tiny;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry;

use Settings;
use ScriptHelper;
use ShardingTools;

my $CHUNK_SIZE = 100_000;

$log->out('START');

my (@shard_min_pids, @shards);
my $relaxed_times = 2;
extract_script_params(
    'relaxed-times=i', \$relaxed_times,
    'shard-min-pid=s' => \@shard_min_pids,
    'shard=i' => \@shards,
);

my %shard_min_pids = (
    map { split /:/ } @shard_min_pids
);
my %valid_shards = map { $_ => 1 } ppc_shards();
@shards = grep { $valid_shards{$_} } @shards;
@shards = keys %valid_shards unless @shards;

# foreach_shard_parallel теряет отправляемые на STDOUT/STDERR
# сообщения об ошибках. А так хоть будет что в логи записать.
sub protect(&) {
    my $sub = shift;
    return sub {
        my @args = @_;
        try {
            $sub->(@args);
        } catch {
            my $err = $_;
            $log->out($err);
            die $err;
        }
    };
}

my $results = foreach_shard_parallel(
    shard => \@shards,
    protect {
        my $shard = shift;
        $log->msg_prefix("[shard $shard]");
        my ($min_pid, $max_pid) = get_one_line_array_sql(PPC(shard => $shard), "select min(pid), max(pid) from phrases");
        $log->out("Shard adgroups have pids from $min_pid to $max_pid");
        if ($shard_min_pids{$shard}) {
            $min_pid = $shard_min_pids{$shard};
            $log->out("Setting lower pid limit to $min_pid (from command line)");
        }
        for (my $chunk_start = $min_pid; $chunk_start <= $max_pid; $chunk_start += $CHUNK_SIZE) {
            my $chunk_end = $chunk_start + $CHUNK_SIZE;
            $log->out("Processing chunk $chunk_start-$chunk_end");
            relaxed times => $relaxed_times, sub {
                my $rows_affected = do_sql(PPC(shard => $shard), [
                    "update phrases g join campaigns c using (cid) join users u using (uid)
                     set g.group_name = CONCAT(IF(u.lang = 'ru', 'Группа', 'AdGroup'), ' №', g.pid), g.LastChange = g.LastChange
                     where (g.group_name is null or g.group_name = '')
                       and g.pid between ? and ?
                    "], $chunk_start, $chunk_end);
                $log->out("Chunk $chunk_start-$chunk_end done: @{[int($rows_affected)]} rows affected");
            };
        }
    }
);

unless (all {all {$_} @$_} values %$results) {
    $log->out("Some shards have failed:");
    $log->out($results);
}

$log->out('FINISH');
