#!/usr/bin/perl

=pod

=head1 NAME
    
    big-forecast.pl - массовый новый прогноз - бъёт фразы по кусочкам и считает общий прогноз в Xls.
    может работать для десятков тысяч ключевиков

=head1 DESCRIPTION

    big-forecast.pl -f FILE_WITH_PHRASES -o OUT_FILE

=cut

use strict;
use warnings;
use utf8;

use EV;
use Path::Tiny;
use Getopt::Long;
use JSON;

use Yandex::Retry;
use Yandex::ListUtils;
use Yandex::HTTP;
use Yandex::Shell;

use my_inc '../..', for => 'protected';
use DevTools;
use PrimitivesIds;
use Forecast;

my $FORECAST_QUERY = {
    cmd => 'ajaxDataForNewBudgetForecast',
    advanced_forecast => 'yes',
    period => 'month',
    period_num => 0,
    # phrases => 'Москва',
    #json_minus_words => '[]',
    # 0 - все, 213 - Москва, ...
    geo => 0,
    unglue =>  1,
    fixate_stopwords => 1,
    currency => 'RUB',
};
my $XLS_QUERY = {
    cmd => "advancedBudgetForecast",
    campaign_name => "Новая",
    pseudo_currency_id => "rub",
    pseudo_currency_rate => "30",
    pseudo_currency_name => "руб.",
    currency_rate => "30",
    geo => $FORECAST_QUERY->{geo},
    geo_text => "регионы",
    #json_minus_words => "[]",
    forecast_type => "budget-by-positions",
    separate_shows => "1",
    is_mobile => "0",
    currency => "RUB",
    periodType => "month",
    periodValue => "0",
    #json_phrases => "[...]",
    "show_period" => [
        "day",
        "week"
        ],
    "show_plan" => [
        "select",
        "10",
        "11",
        "13",
        "20",
        "23",
        ]
};

my ($FILE, $OUT, $N);
my $CHUNK_SIZE=1000;
GetOptions(
    "help" => \&usage,
    "file|f=s" => \$FILE,
    "chunk=s" => \$CHUNK_SIZE,
    "out|o=s" => \$OUT,
    "n=n" => \$N,
    ) or die "can't parse options";

if (!defined $OUT) {
    usage("Out file is not defined");
}
if (!defined $FILE) {
    usage("File is not defined");
} elsif (!-f $FILE) {
    usage("File $FILE not exists")
}

my $UID = PrimitivesIds::get_uid(login => 'yndx.zhur.superreader');
{
    no warnings 'once';
    $DoCmd::PARSE_JSON_FORM_ALWAYS = 1;
    $Yandex::Shell::PRINT_COMMANDS = 1;
    $Forecast::MAX_PHRASES_NUM = $CHUNK_SIZE;
}

no warnings;
$^W = 0;

my @phrases = grep {/\S/} map {s/^\s+|\s+$//gr} path($FILE)->lines_utf8;
if (!@phrases) {
    usage("File is empty");
}
my @todo;
while((!defined $N || @todo < $N) && @phrases) {
    my $phrase = shift @phrases;
    my $r = Forecast::test_phrases([$phrase]);
    if ($r->{error}) {
        print STDERR "WARN: skip incorrect phrase '$phrase': $r->{error}\n";
    } else {
        push @todo, $phrase;
    }
}

my @data;
my %pos2place = qw(
    10 first_premium
    11 second_premium
    13 fourth_premium
    20 first_place
    23 fourth_place
);
print "Phrases count: ".scalar(@todo).", chunk: $CHUNK_SIZE\n";
my $chunk_idx = 0;
for my $chunk (chunks(\@todo, $CHUNK_SIZE)) {
    print "Start process chunk ".(++$chunk_idx)."\n";
    retry tries => 3, sub {
        my $query = Yandex::HTTP::make_url('', {%$FORECAST_QUERY, phrases => join(',', @$chunk)});
        my $ret = DevTools::process_query($UID, $query);
        if ($ret->[0] != 200) {
            print STDERR "ERROR: ".to_json($ret);
            exit 1;
        }
        my $res = JSON->new->utf8->decode($ret->[2]->[0]);
        if ($res->{error}) {
            print "ERROR: $res->{error}\n";
            exit 1;
        }
    
        for my $pos (@{$res->{data_by_positions}}) {
            my %phrase = (
                "phrase" => $res->{key2phrase}->{$pos->{md5}},
                "position" => 20,
                "shows" => $pos->{shows},
                "sum" => undef,
                "amnesty_price" => undef,
                "bid_price" => undef,
                );
            while(my ($num, $place) = each %pos2place) {
                my $d = $pos->{positions}{$place};
                $phrase{$num} = {
                    "clicks" => $d->{clicks},
                    "sum" => $d->{budget},
                    "ctr" => $d->{ctr},
                    "amnesty_price" => int(($d->{sum}//0) / ($d->{clicks}||1)),
                    "shows" => $d->{shows},
                    "bid_price" => $d->{bid},
                };
            }
            push @data, \%phrase;
        }
    }
}

my $xls_query = {
    %$XLS_QUERY,
    json_phrases => JSON->new->utf8(1)->encode(\@data),
};

my $ret = DevTools::process_query($UID, Yandex::HTTP::make_url('', $xls_query));
if ($ret->[0] != 200) {
    print STDERR "ERROR: ".to_json($ret);
    exit 1;
}
path($OUT)->spew($ret->[2]->[0]);
print "SUCCESS\n";


sub usage {
    my $msg = shift;
    if ($msg) {
        print STDERR "ERROR: $msg\n";
    }
    system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit( $msg ? 1 : 0 );
}
