#!/usr/bin/perl
use Direct::Modern;

use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils;

use my_inc '../..', for => 'protected';
use BS::ResyncQueue;
use MailService;
use ScriptHelper 'Yandex::Log' => [ date_suf => '%Y%m%d', auto_rotate => 1, tee => $ENV{LOG_TEE}, lock => 1 ];
use Settings;
use ShardingTools;

my $CHUNK_SIZE = 100;
my $CHUNK_TIME = 60;
my $DRY_RUN;
my $CID_ARG = undef;
my $CAMPAIGNS_LIMIT_PER_SHARD = undef;
my $STOP_FILE_LOCATION = "$Settings::ROOT/protected/run/change-campaign-strategies-min-price-to-highest-position.stop";

extract_script_params(
    'dry-run' => \$DRY_RUN,
    'cid=i' => \$CID_ARG,
    'campaigns-limit-per-shard=i' => \$CAMPAIGNS_LIMIT_PER_SHARD,
);

if ( -e $STOP_FILE_LOCATION ) {
    $log->die("not running, found stop file at $STOP_FILE_LOCATION");
}

$log->out('start');
$log->out("to stop, touch $STOP_FILE_LOCATION");

foreach_shard_parallel shard => [ ppc_shards() ], sub {
    my ($shard) = @_;
    modify_campaigns_on_shard($shard);
};

$log->out('finish');

sub modify_campaigns_on_shard {
    my ($shard) = @_;

    my $shard_prefix_guard = $log->msg_prefix_guard("shard=$shard");
    $log->out('start');

    # SQL-запросы будут доставать кампании по порядку cid, следующий запрос будет искать кампании с cid больше этого
    my $cid_floor = 0;

    my $handled_campaigns = 0;

    while (1) {
        rate_limiting_sleep();

        my $campaign_rows = get_all_sql( PPC( shard => $shard ), [
            "SELECT cid, uid, strategy_min_price, strategy_no_premium",
            FROM => 'campaigns',
            WHERE => {
                cid__gt => $cid_floor,
                _OR => { strategy_min_price__ne => '', strategy_no_premium => 'min_price' },
                ( $CID_ARG ? ( cid => $CID_ARG ) : () ),
            },
            'ORDER BY cid',
            LIMIT => $CHUNK_SIZE,
        ] );

        $log->out( { got_rows => scalar(@$campaign_rows) } );
        last unless @$campaign_rows;

        $cid_floor = $campaign_rows->[-1]->{cid};

        modify_campaign_multi( PPC( shard => $shard ), $campaign_rows );

        $handled_campaigns += scalar(@$campaign_rows);
        if ( defined $CAMPAIGNS_LIMIT_PER_SHARD && $handled_campaigns >= $CAMPAIGNS_LIMIT_PER_SHARD ) {
            $log->out("terminating early, handled $handled_campaigns campaigns");
            last;
        }

        if ( -e $STOP_FILE_LOCATION ) {
            $log->out("terminating early, found stop file at $STOP_FILE_LOCATION");
            last;
        }
    }

    $log->out('finish');
}

sub rate_limiting_sleep {
    return if $CID_ARG;

    state $last_sleep_timestamp;

    if ( ! defined $last_sleep_timestamp ) {
        $last_sleep_timestamp = Time::HiRes::time;
        return;
    }

    my $time_to_sleep = $last_sleep_timestamp + $CHUNK_TIME - Time::HiRes::time;
    if ( $time_to_sleep > 0 ) {
        $log->out("sleeping for $time_to_sleep");
        Time::HiRes::sleep($time_to_sleep);
    }

    $last_sleep_timestamp = Time::HiRes::time;
}


sub modify_campaign_multi {
    my ( $db, $campaign_rows ) = @_;

    for my $campaign_row (@$campaign_rows) {
        modify_campaign( $db, $campaign_row );
    }

    my $resync_data = [ map { { cid => $_->{cid}, pid => 0, bid => 0, priority => BS::ResyncQueue::PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE } } @$campaign_rows ];
    $log->out( { resync_data => $resync_data } );
    unless ($DRY_RUN) {
        bs_resync($resync_data);
    }

    # uid => [ cid, cid, cid, ... ]
    my %cids_by_uids;
    for my $campaign_row (@$campaign_rows) {
        $cids_by_uids{ $campaign_row->{uid} } ||= [];
        push @{ $cids_by_uids{ $campaign_row->{uid} } }, $campaign_row->{cid};
    }

    for my $uid ( nsort keys %cids_by_uids ) {
        my $cids = $cids_by_uids{$uid};
        my @cids_head = @$cids[ 0 .. ( $#$cids > 4 ? 4 : $#$cids ) ];
        $log->out( "sending email to $uid about cids: " . join( ', ', @$cids ) );
        try {
            send_prepared_mail( 'campaign_strategy_min_price_deprecated', $uid, $Settings::NOTIFICATION_EMAIL_FROM, { cids => $cids, cids_head => \@cids_head } );
        } catch {
            $log->out("error sending email: $_");
        };
    }
}

sub modify_campaign {
    my ( $db, $campaign_row ) = @_;

    $log->out( { campaign_row => $campaign_row } );

    if ( $campaign_row->{strategy_min_price} && $campaign_row->{strategy_min_price} ne '' ) {
        $log->out('fixing strategy_min_price');
        unless ($DRY_RUN) {
            my $affected = do_sql( $db, [
                'UPDATE campaigns SET strategy_min_price = "", LastChange = NOW()',
                WHERE => { cid => $campaign_row->{cid}, strategy_min_price__ne => '' }
            ] );
            $log->out( { affected => $affected } );
        }
    }

    if ( $campaign_row->{strategy_no_premium} && $campaign_row->{strategy_no_premium} eq 'min_price' ) {
        $log->out('fixing strategy_no_premium');
        unless ($DRY_RUN) {
            my $affected = do_sql( $db, [
                'UPDATE campaigns SET strategy_no_premium = "highest_place", LastChange = NOW()',
                WHERE => { cid => $campaign_row->{cid}, strategy_no_premium => 'min_price' }
            ] );
            $log->out( { affected => $affected } );
        }
    }
}

