#!/usr/bin/perl
use Direct::Modern;

use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils;

use my_inc '../..', for => 'protected';
use BS::ResyncQueue;
use Campaign;
use JSON;
use MailService;
use Yandex::Retry qw/relaxed_guard/;
use ScriptHelper 'Yandex::Log' => [ date_suf => '%Y%m%d', auto_rotate => 1, tee => $ENV{LOG_TEE}, lock => 1 ];
use Settings;
use ShardingTools;

my $CHUNK_SIZE = 500;
my $CHUNK_TIME = 15;
my $DRY_RUN;
my $CID_ARG = undef;
my $ONLY_SHARD = undef;
my $FILE_NAME = undef;

extract_script_params(
    'dry-run' => \$DRY_RUN,
    'cid=i' => \$CID_ARG,
    'shard=i' => \$ONLY_SHARD,
    'resume_from_file=s' => \$FILE_NAME,
);

$log->out('start');

my $cids_for_resume;

if ($FILE_NAME){
    open( my $file, '<', $FILE_NAME) || die "Can't open $FILE_NAME";
    @$cids_for_resume = map {chomp $_; $_} <$file>;
    close($file);
}

foreach_shard_parallel shard => [ ppc_shards() ], sub {
    my ($shard) = @_;
    modify_campaigns_on_shard($shard, $cids_for_resume);
};

$log->out('finish');

sub modify_campaigns_on_shard {
    my ($shard, $cids_for_resume) = @_;

    return if $ONLY_SHARD && $shard != $ONLY_SHARD;

    my $shard_prefix_guard = $log->msg_prefix_guard("shard=$shard");
    $log->out('start');

    # SQL-запросы будут доставать кампании по порядку cid, следующий запрос будет искать кампании с cid больше этого
    my $cid_floor = 0;

    my $handled_campaigns = 0;

    while (1) {
        my $campaign_rows = get_all_sql( PPC( shard => $shard ), [
            "SELECT cid, uid, type, currency, strategy_data",
            FROM => 'campaigns',
            WHERE => {
                cid__gt => $cid_floor,
                #Если делаем докатку, и у кампании стоит уже не default, т.е. пользователь поменял стратегию после нашего апдейта - докатывать ее не будем
                ($cids_for_resume ? (cid => $cids_for_resume, strategy_name => 'default') 
                   : (
                       strategy_name => 'no_premium',
                       ( $CID_ARG ? ( cid => $CID_ARG ) : () )
                     ),
                )
            },
            'ORDER BY cid',
            LIMIT => $CHUNK_SIZE,
        ] );

        $log->out( { got_rows => scalar(@$campaign_rows) } );
        last unless @$campaign_rows;

        $cid_floor = $campaign_rows->[-1]->{cid};
        try {
            modify_campaign_multi( PPC( shard => $shard ), $campaign_rows );
        }
        catch {
            $log->out(@_);
        };
    }

    $log->out('finish');
}

sub modify_campaign_multi {
    my ( $db, $campaign_rows ) = @_;

    my $rg = relaxed_guard(times => $CHUNK_TIME);
    update_campaigns( $db, $campaign_rows );

    my $resync_data = [ map { { cid => $_->{cid}, pid => 0, bid => 0, priority => BS::ResyncQueue::PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE } } @$campaign_rows ];
    $log->out( { resync_data => $resync_data } );
    unless ($DRY_RUN) {
        bs_resync($resync_data);
    }

    # uid => [ cid, cid, cid, ... ]
    my %cids_by_uids;
    for my $campaign_row (@$campaign_rows) {
        $cids_by_uids{ $campaign_row->{uid} } ||= [];
        push @{ $cids_by_uids{ $campaign_row->{uid} } }, $campaign_row->{cid};
    }

    for my $uid ( nsort keys %cids_by_uids ) {
        my $cids = $cids_by_uids{$uid};
        my @cids_head = @$cids[ 0 .. ( $#$cids > 4 ? 4 : $#$cids ) ];
        $log->out( "sending email to $uid about cids: " . join( ', ', @$cids ) );
        try {
            send_prepared_mail( 'campaign_strategy_no_premium_deprecated', $uid, $Settings::NOTIFICATION_EMAIL_FROM, { cids => $cids, cids_head => \@cids_head } )
                unless $DRY_RUN;
        } catch {
            $log->out("error sending email: $_");
        };
    }
}

sub update_campaigns {
    my ( $db, $campaign_rows ) = @_;

    my $cids = [map {$_->{cid}} @$campaign_rows];
    $log->out( { campaigns => join(', ', @$cids) } );

    unless ($DRY_RUN) {
        my $new_strategy = {"name"=>"default", "version" => 1};
        #Если задан файл - значит делаем "докатку", апдейтить базу не нужно
        unless ($FILE_NAME) {
            my $affected = do_sql( $db, [
                q/UPDATE campaigns SET
                    strategy_name = 'default',
                    LastChange = NOW(),
                    strategy_data = '/.to_json($new_strategy)."'",
                WHERE => { cid => $cids, strategy_name =>'no_premium' }
            ] );
            $log->out( { affected => $affected } );
        }

        foreach my $camp (@$campaign_rows){
            my $strategy;
            eval {
                if ($FILE_NAME) {
                    #Если "докатываем" кампании из файла - стратегия у них уже переписана, восстанавливаем исходные данные - они у всех одинаковые.
                    $camp->{strategy_data} = '{"name": "no_premium", "place": "highest_place", "version": 1}';
                }
                $strategy = from_json($camp->{strategy_data})
            };
            next unless $strategy;
            $log->out({ 'mark strategy change for campaign' => $camp->{cid}});
            Campaign::mark_strategy_change(
                    $camp->{cid},
                    $camp->{uid},
                    Campaign::define_strategy({type => $camp->{type}, currency => $camp->{currency},  strategy_decoded => $new_strategy}),
                    Campaign::define_strategy({type => $camp->{type}, currency => $camp->{currency},  strategy_decoded => $strategy}),
                    0
            );
        }
    }
    
    return;
}

