#!/usr/bin/perl

=head1 DESCRIPTION

    Скрипт для изменения типа у фидов.

    Тип фида просто так менять нельзя, т.к. фильтры на фид в смарт/динамических группах могут содержать условия,
    специфичные для типа бизнеса и фида.
    Поэтому условия нужно либо сбросить во "Все товары", либо проверить, что все условия поддерживаются новым
    типом фида.

    После смены у всех затронутых фидов меняется статус обновления на New, чтобы переотправить их в BL
    и все связанные баннеры переотправляются в БК

    Доступные опции.
        --feed-ids 123,567,789          # список feed_id через запятую, обязательный
        --new-feed-type YandexCustom    # новый тип фида, обязательный
        --reset-filter-conditions       # удалить все условия в фильтрах, привязанных к фиду
        --dry-run                       # запуск без записи в базу

    --reset-filter-conditions сбрасывает условия во всех фильтрах, привязанных к фиду во "Все товары".
    В динамических группах при этом удаляются все кроме одного фильтра, т.к. в динамических группах все условия
    должны быть уникальными. Остается фильтр с наименьшим dyn_id

    Если у фида не нужно менять тип, он и привязанные к нему фильтры не затрагиваются

=head1 USAGE

    Запуск в холостом режиме, со сбросом условий в фильтрах
    LOG_TEE=1 ./protected/one-shot/change-feed-type.pl --dry-run --feed-ids 123,456,789 --new-feed-type YandexCustom --reset-filter-conditions

    Запуск в холостом режиме, без сбрасывания условий в фильтрах
    LOG_TEE=1 ./protected/one-shot/change-feed-type.pl --dry-run --feed-ids 123,456,789 --new-feed-type YandexCustom

=head1 FUNCTIONS

=cut

use Direct::Modern;

use JSON;
use List::MoreUtils qw/uniq/;

use Yandex::DBTools;

use my_inc '../..';

use Direct::AdGroups2::Performance;
use Direct::AdGroups2::Dynamic;
use Direct::DynamicConditions;
use Direct::Feeds;
use Direct::Model::Feed;
use Direct::PerformanceFilters;
use Direct::Validation::DynamicConditions;
use Direct::Validation::PerformanceFilters;
use Settings;
use ShardingTools;

use ScriptHelper;

# Какой from_tab нужно выставить фильтру, если его условие стало пустым
# подсмотрено в data3/desktop.blocks/i-utils/__feed-filter-data/i-utils__feed-filter-data.utils.js
my $EMPTY_CONDITION_TAB = {
    'YandexMarket'  => 'condition',
    'default'       => 'all-products',
};

my $DRY_RUN = 0;
extract_script_params(
    'dry-run' => \$DRY_RUN,
    'feed-ids=s' => \my $FEED_IDS_STR,
    'new-feed-type=s' => \my $NEW_FEED_TYPE,
    'reset-filter-conditions' => \my $RESET_FILTER_CONDITIONS,
);

if (!$FEED_IDS_STR || !$NEW_FEED_TYPE) {
    usage();
}
# этот список потом не бьется на чанки, т.к. пока подразумевается, что он будет маленьким
my @FEED_IDS = split(/,/, $FEED_IDS_STR);

validate_new_feed_type();

$log->out("START");
log_script_params();
for my $shard (ppc_shards()) {
    my $shard_prefix_guard = $log->msg_prefix_guard("[shard=$shard".($DRY_RUN ? ', dry-run' : '').']');

    fix_feeds($shard);
}
$log->out("FINISH");

=head2 validate_new_feed_type

    Проверить, что всем фидам из @FEED_IDS можно поставить тип фида $NEW_FEED_TYPE
    проверка идет по списку соответствий типов бизнеса и допустимых для них типов фидов

    Если не включено сбрасывание всех фильтров во "Все товары", проверяется, что
    условия в Смарт и динамических фильтрах подходят новому типу фида

=cut
sub validate_new_feed_type {
    my $id_to_business = get_hash_sql(PPC(shard => 'all'), ["
        SELECT feed_id, business_type
        FROM feeds",
        WHERE => [
            feed_id => \@FEED_IDS,
        ]
    ]);
    my $business_to_feed_type = Direct::Model::Feed::VALID_TYPES_FOR_BUSINESS();

    my @businesses = uniq values %$id_to_business;

    for my $business (@businesses) {
        if (!$business_to_feed_type->{$business}{$NEW_FEED_TYPE}) {
            my $filter_type = "${business}_$NEW_FEED_TYPE";
            my @feed_ids = grep { $id_to_business->{$_} eq $business } keys %$id_to_business;
            $log->die("invalid new filter type $filter_type for filters ".to_json(\@feed_ids));
        }
    }

    my $has_errors = 0;
    if (!$RESET_FILTER_CONDITIONS) {
        for my $shard (ppc_shards()) {
            $has_errors ||= validate_performance_filters($shard, $id_to_business);
            $has_errors ||= validate_dynamic_conditions($shard, $id_to_business);
        }
    }
    if ($has_errors) {
        $log->die("validation errors");
    }
}

=head2 validate_performance_filters($shard, $feed_id_to_business)

    Проверка, подходят ли условия Смарт фильтров к новому типу фида.
    В $feed_id_to_business лежит мапа feed_id => business_type

    Если есть ошибки валидации, пишет в лог и возвращает 1

=cut
sub validate_performance_filters {
    my ($shard, $feed_id_to_business) = @_;

    my $performance_filters = get_performance_filters($shard);
    my $has_errors = 0;
    for my $filter (values %$performance_filters) {
        next unless scalar @{$filter->condition};

        my $filter_type = $feed_id_to_business->{$filter->adgroup->feed_id()}.'_'.$NEW_FEED_TYPE;
        my $vr = Direct::Validation::PerformanceFilters::validate_performance_filter_rules($filter->condition, $filter_type);
        if (!$vr->is_valid || $vr->has_only_warnings) {
            $log->out("Validation error for performance fiter ".$filter->id().': '.$vr->get_first_error_description());
            $has_errors = 1;
        }
    }

    return $has_errors;
}


=head2 validate_dynamic_conditions($shard, $feed_id_to_business)

    Проверка, подходят ли условия динамических фильтров к новому типу фида.
    В $feed_id_to_business лежит мапа feed_id => business_type

    Если есть ошибки валидации, пишет в лог и возвращает 1

=cut
sub validate_dynamic_conditions {
    my ($shard, $feed_id_to_business) = @_;

    my $conditions = get_dynamic_conditions($shard);
    my $has_errors = 0;
    for my $dyn_cond (values %$conditions) {
        next unless scalar @{$dyn_cond->condition};

        my $filter_type = $feed_id_to_business->{$dyn_cond->adgroup->feed_id()}.'_'.$NEW_FEED_TYPE;
        my $vr = Direct::Validation::DynamicConditions::validate_dynamic_condition_rules($dyn_cond->condition, $filter_type);
        if (!$vr->is_valid || $vr->has_only_warnings) {
            $log->out("Validation error for dynamic condition ".$dyn_cond->id().': '.$vr->get_first_error_description());
            $has_errors = 1;
        }
    }

    return $has_errors;
}

sub log_script_params {
    $log->out("parameters: ".to_json({
        'dry-run' => $DRY_RUN,
        'feed-ids' => $FEED_IDS_STR,
        'new-feed-type' => $NEW_FEED_TYPE,
        'reset-filter-conditions' => $RESET_FILTER_CONDITIONS,
    }, {canonical => 1}));
}

=head2 fix_feeds($shard)

    Внутри транзакции сначала обрабатывает все фильтры, привязанные к фидам, потом меняет тип фида в таблице feeds

=cut
sub fix_feeds {
    my ($shard) = @_;

    # нужно ли как-то менять фильтры
    my $need_to_fix_filters = $RESET_FILTER_CONDITIONS;

    do_in_transaction sub {
        if ($need_to_fix_filters) {
            fix_performance_filters($shard);
            fix_dynamic_conditions($shard);
        }

        change_feed_type($shard);
    }
}

=head2 fix_performance_filters($shard)

    Обрабатывает все смарт фильтры, привязанные к фидам, перед тем как поменять им тип.
    Новые условия фильтра должны быть совместимы с $NEW_FEED_TYPE

    При запуске скрипта с параметром --reset-filter-conditions, условия фильтров очищаются.

    Если в результате преобразование условие окажется пустым, у фильтра сбрасывается параметр from_tab -
    он обозначает, на какой "вкладке" фильтра было создано условие.
    см $EMPTY_CONDITION_TAB

=cut
sub fix_performance_filters {
    my ($shard) = @_;

    my $filters = get_performance_filters($shard);

    for my $perf_filter_id (keys %$filters) {
        my $filter = $filters->{$perf_filter_id};

        $log->out("performance filter $perf_filter_id previous condition_json: ".$filter->_condition_json());
        $log->out("performance filter $perf_filter_id previous from_tab: ".$filter->from_tab());

        if ($RESET_FILTER_CONDITIONS) {
            $log->out("resetting condition of performance filter $perf_filter_id to {}");
            $filter->condition([]);
            # неявная часть condition_json
            $filter->available(0);
        }

        if ($filter->_condition_json() eq '{}') {
            $filter->from_tab(get_empty_condition_tab());
        }

        $log->out("performance filter $perf_filter_id new condition_json: ".$filter->_condition_json());
        $log->out("performance filter $perf_filter_id new from_tab: ".$filter->from_tab());
    }

    if (!$DRY_RUN) {
        Direct::PerformanceFilters->new([values %$filters])->update();
    }
}

=head2 get_performance_filters($shard)

    Получение всех Смарт фильтров, привзанных к фидам @FEED_IDS, у которых
    еще не прописан новый фид.
    У всех фильтров проинициализировано поле `adgroup`

=cut
sub get_performance_filters {
    my ($shard) = @_;

    my $rows = get_all_sql(PPC(shard => $shard), ["
        SELECT ap.pid, ap.feed_id
        FROM adgroups_performance ap
            JOIN feeds f ON f.feed_id = ap.feed_id",
        WHERE => [
            'ap.feed_id' => \@FEED_IDS,
            'f.feed_type__ne' => $NEW_FEED_TYPE,
        ]
    ]);
    my @pids = map { $_->{pid} } @$rows;

    my $filters = Direct::PerformanceFilters->get_by(
            adgroup_id => \@pids,
            # для выгрузки from_tab
            with_additional => 1,
        )->items_by('id');
    my $adgroups = Direct::AdGroups2::Performance->get_by(
            adgroup_id => \@pids
        )->items_by('id');

    for my $filter (values %$filters) {
        $filter->adgroup($adgroups->{$filter->adgroup_id()});
    }

    return $filters;
}

=head2 fix_dynamic_conditions($shard)

    Обрабатывает все динамические фильтры, привязанные к фидам @FEED_IDS
    Т.к. внутри одной динамической группы не может быть двух фильтров с одинаковыми условиями,
    перед сохранением обработанных фильтров происходит их дедупликация внутри группы.
    Фильтры, которые начали дублироваться, удаляются. Из двух одинаковых фильтров остается тот,
    у которого меньший dyn_id

    При запуске скрипта с параметром --reset-filter-conditions, условия фильтров очищаются.

    Если в результате преобразование условие окажется пустым, у фильтра сбрасывается параметр from_tab -
    он обозначает, на какой "вкладке" фильтра было создано условие.
    см $EMPTY_CONDITION_TAB

=cut
sub fix_dynamic_conditions {
    my ($shard) = @_;

    my $conditions = get_dynamic_conditions($shard);

    my %conditions_by_group;
    push @{$conditions_by_group{$_->adgroup_id()}}, $_ for values %$conditions;

    my @conditions_to_delete;
    my @conditions_to_update;
    for my $adgroup_id (keys %conditions_by_group) {
        my @group_conds = sort { $a->id() <=> $b->id() } @{$conditions_by_group{$adgroup_id}};

        my %seen_condition_hashes;
        for my $dyn_cond (@group_conds) {
            my $dyn_cond_id = $dyn_cond->id();

            $log->out("dynamic condition $dyn_cond_id previous condition_json: ".$dyn_cond->_condition_json());
            $log->out("dynamic condition $dyn_cond_id previous from_tab: ".$dyn_cond->from_tab());

            if ($RESET_FILTER_CONDITIONS) {
                $log->out("resetting condition of dynamic condition $dyn_cond_id to {}");
                $dyn_cond->condition([]);
                # неявная часть condition_json
                $dyn_cond->available(0);
            }

            if ($dyn_cond->_condition_json() eq '{}') {
                $dyn_cond->from_tab(get_empty_condition_tab());
            }

            $log->out("dynamic condition $dyn_cond_id new condition_json: ".$dyn_cond->_condition_json());
            $log->out("dynamic condition $dyn_cond_id new from_tab: ".$dyn_cond->from_tab());

            if (!$seen_condition_hashes{$dyn_cond->_condition_hash()}) {
                push @conditions_to_update, $dyn_cond;
                $seen_condition_hashes{$dyn_cond->_condition_hash()} = 1;
            } else {
                $log->out("deleting dynamic condition $dyn_cond_id, because it is duplicate in $adgroup_id now");
                $log->out($dyn_cond->to_hash());
                push @conditions_to_delete, $dyn_cond;
            }
        }
    }

    if (!$DRY_RUN) {
        Direct::DynamicConditions->new(\@conditions_to_delete)->delete();
        Direct::DynamicConditions->new(\@conditions_to_update)->update();
    }
}

=head2 get_dynamic_conditions($shard)

    Получение всех динамических фильтров, привязанных к фидам @FEED_IDS,
    у которых еще не выставлен тип $NEW_FEED_TYPE
    У всех фильтров проинициализировано поле `adgroup`

=cut
sub get_dynamic_conditions {
    my ($shard) = @_;

    my $rows = get_all_sql(PPC(shard => $shard), ["
        SELECT ad.pid, ad.feed_id
        FROM adgroups_dynamic ad
            JOIN feeds f ON f.feed_id = ad.feed_id",
        WHERE => [
            'ad.feed_id' => \@FEED_IDS,
            'f.feed_type__ne' => $NEW_FEED_TYPE,
        ]
    ]);
    my @pids = map { $_->{pid} } @$rows;

    my $conditions = Direct::DynamicConditions->get_by(
            adgroup_id => \@pids,
            # для выгрузки from_tab
            with_additional => 1,
        )->items_by('id');
    my $adgroups = Direct::AdGroups2::Dynamic->get_by(
            adgroup_id => \@pids
        )->items_by('id');

    for my $dyn_cond (values %$conditions) {
        $dyn_cond->adgroup($adgroups->{$dyn_cond->adgroup_id()});
    }

    return $conditions;
}

=head2 get_empty_condition_tab

    Какой from_tab нужно выставить фильтру, если его условие сбросилось в {}
    Зависит от $NEW_FEED_TYPE

=cut
sub get_empty_condition_tab {
    return $EMPTY_CONDITION_TAB->{$NEW_FEED_TYPE} || $EMPTY_CONDITION_TAB->{default};
}


=head2 change_feed_type($shard)

    Поменять всем фидам из @FEED_IDS тип фида на $NEW_FEED_TYPE, если нужно.

    При смене типа статус фида сбрасывается в New, чтобы фид переотправили в BL, и тот подтвердил новый тип фида
    Также все баннеры связанные с фидом ставятся на переотправку в БК, т.к. в BannerLandData отправляется тип фида.
    Он может повлиять на проверку несменяемости типа фида при генерации

=cut
sub change_feed_type {
    my ($shard) = @_;

    my $rows = get_all_sql(PPC(shard => $shard), ["
        SELECT feed_id, ClientID
        FROM feeds",
        WHERE => [
            feed_id => \@FEED_IDS,
            feed_type__ne => $NEW_FEED_TYPE,
        ]
    ]);
    if (!@$rows) {
        return;
    }
    my @client_ids = uniq map { $_->{ClientID} } @$rows;
    my @feed_ids = map { $_->{feed_id} } @$rows;

    my $feeds_collection = Direct::Feeds->get_by(\@client_ids, feed_id => \@feed_ids);

    for my $feed (@{$feeds_collection->items()}) {
        my $feed_id = $feed->id();
        $log->out("feed $feed_id old feed_type: ".$feed->feed_type());
        $log->out("changing feed_type of feed $feed_id to $NEW_FEED_TYPE, update_status to New and resyncing banners");
        $feed->feed_type($NEW_FEED_TYPE);
        $feed->update_status('New');
        $feed->do_bs_sync_banners(1);
    }

    if (!$DRY_RUN) {
        $feeds_collection->save();
    }
}
