#!/usr/bin/perl

use my_inc '../..';

=head1 NAME

change_agency_subclient_product_id.pl

=head1 DESCRIPTION

Скрипт меняет казахстанский продукт с валютного на квазивалютный для всех субклиентов указанных агентств

=head1 RUNNING

    Запускать с параметрами:
        --agency - логин агенства, клиентов которого добавляем на конвертацию
        --client-login - логин субклиента, которого хотим мигрировать, работает только вместе с параметром --agency
        --chunk_size - размер пачки кампаний для ручки Баланса Balance.GetOrdersInfo
        --timeout - таймаут для ручки Баланса Balance.GetOrdersInfo

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::Validate qw/is_valid_id/;
use Yandex::Balance;
use Yandex::ListUtils qw/chunks xisect xminus/;

use Primitives;
use PrimitivesIds;
use Client;
use Campaign;
use RBACElementary;
use RBAC2::Extended;
use RBACDirect;

use ScriptHelper;
use Settings;

use List::MoreUtils qw/any/;

my (@agency_logins, @client_logins, $chunk_size, $balance_call_timeout);
extract_script_params(
    'agency=s@' => \@agency_logins,
    'client-login=s@' => \@client_logins,
    'chunk_size=i' => \$chunk_size,
    'timeout=i' => \$balance_call_timeout,
);

my $OPERATOR_UID = 0;
my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID) or $log->die("Can't connect to RBAC: $@");

$log->out('START');

# ProductID могут различаться в тестинге и продакшене, поэтому берем их из базы
my $old_product_id = get_one_field_sql(PPCDICT, ["SELECT ProductID FROM products", where => {currency => 'KZT', type => 'Text', UnitName => 'Bucks'}]);
$log->die("Can't get valid old ProductID: $old_product_id") unless is_valid_id($old_product_id);
my $new_product_id = get_one_field_sql(PPCDICT, ["SELECT ProductID FROM products", where => {currency => 'KZT', type => 'Text', UnitName => 'QuasiCurrency'}]);
$log->die("Can't get valid new ProductID: $new_product_id") unless is_valid_id($new_product_id);

$log->out("Change ProductID from $old_product_id to $new_product_id");

$log->out('Got '. scalar @agency_logins .' agencies from params:', \@agency_logins);
foreach my $agency_login (@agency_logins) {
    my $msg_guard = $log->msg_prefix_guard("[agency=$agency_login]");
    my $agency_uid = get_uid_by_login($agency_login);
    unless (is_valid_id($agency_uid)) {
        $log->out("SKIP agency: got unvalid uid $agency_uid");
        next;
    }
    my $agency_chief_uid = rbac_get_chief_rep_of_agency_rep($agency_uid);
    unless (is_valid_id($agency_chief_uid)) {
        $log->out("SKIP agency: got unvalid chief uid $agency_chief_uid");
        next;
    }
    my $agency_id = rbac_get_agency_clientid_by_uid( $agency_chief_uid ) or $log->out("agency ClientID not found");
    unless (is_valid_id($agency_id)) {
        $log->out("SKIP agency: got unvalid ClientID $agency_id");
        next;
    }
    # Баланс может перевести субклиена на новый продукт только если НДС всегда был нулевым
    my $agency_nds = get_one_column_sql(PPC(ClientID => $agency_id), ["select nds from agency_nds", where => {ClientID => SHARD_IDS}]);
    if (any { $_ > 0 } @$agency_nds) {
        $log->out("SKIP agency (ClientID $agency_id): has non zero NDS");
        next;
    }

    my $clids = RBACDirect::rbac_get_subclients_clientids($rbac, $agency_chief_uid);
    $log->out('Got '. scalar @$clids .' subclients', $clids);
    # работаетм только с указанными субклиентами если их передали в параметрах
    if (@client_logins) {
        $log->out("working only with subclients:", \@client_logins);
        my $client_ids = get_clientids(login => \@client_logins);
        $clids = xisect $clids, $client_ids;
    }

    my $clids2data = mass_get_clients_data($clids, [qw/work_currency is_using_quasi_currency non_resident country_region_id chief_uid/]);
    my $chief_uid2login = get_uid2login(uid => [ map { $clids2data->{$_}->{chief_uid} } @$clids ]);

    foreach my $client_id (@$clids) {
        my $client_login = $chief_uid2login->{ $clids2data->{$client_id}->{chief_uid} };
        my $msg_guard = $log->msg_prefix_guard("[agency=$agency_login, agency_id=$agency_id, client=$client_login, client_id=$client_id]");
        # оставляем только субклиентов в тенге
        if ($clids2data->{$client_id}->{work_currency} ne 'KZT') {
            $log->out("SKIP subclient: bad currency $clids2data->{$client_id}->{work_currency}");
            next;
        }
        # нерезиденты используют собственный график НДС
        if ($clids2data->{$client_id}->{non_resident}) {
            $log->out("Non resident subclient");
            # Баланс может перевести субклиена на новый продукт только если НДС всегда был нулевым
            my $client_nds = get_one_column_sql(PPC(ClientID => $client_id), ["select nds from client_nds", where => {ClientID => SHARD_IDS}]);
            if (any { $_ > 0 } @$client_nds) {
                $log->out("SKIP subclient: has non zero NDS");
                next;
            }
        }

        if ($clids2data->{$client_id}->{is_using_quasi_currency} == 0) {
            # ставим клиенту признак использования квазивалюты is_using_quasi_currency
            $log->out("Set is_using_quasi_currency flag for client");
            create_update_client({client_data => {ClientID => $client_id, is_using_quasi_currency => 1}});
        }

        # находим все кампании клиента в Директе
        $log->out("Get campaigns...");
        my $camps_info = get_hashes_hash_sql(PPC(ClientID => $client_id),
            ["SELECT cid, uid, type, name, AgencyUID, ProductID FROM campaigns",
            where => {
                    ClientID => SHARD_IDS,
                    currency => 'KZT',
            }]);
        my @cids = keys %$camps_info;
        $log->out('Got '. scalar @cids .' campaigns:', @cids);
        # добавляем в данные по кампаниям дополнительные поля для Баланса
        my $managers = {};
        foreach my $cid (@cids) {
            my $agency_uid = $camps_info->{$cid}->{AgencyUID};
            my $camp_type = $camps_info->{$cid}->{type};

            my $manager = $managers->{$agency_uid // 0}->{$camp_type};
            if ($agency_uid && !is_valid_id($manager)) {
                # кэшируем менеджера, чтобы лишний раз не ходить в RBAC
                $manager = $managers->{$agency_uid}->{$camp_type} = rbac_get_manager_of_agency($rbac, $agency_uid, $camp_type);
            }
            if (is_valid_id($manager)) {
                $camps_info->{$cid}->{ManagerUID} = $manager;
            }

            $camps_info->{$cid}->{pay_method_flag} = 0;
            my $pay_method = check_method_pay($cid);
            if ($pay_method eq 'with_block') {
                $camps_info->{$cid}->{pay_method_flag} = 1;
            }
        }

        # находим все кампании клиента в Балансе
        my $balance_info = [];
        for my $cids_chunk (chunks [@cids], $chunk_size || 30) {
            my ($balance_info_chunk) = Yandex::Balance::balance_call('Balance.GetOrdersInfo', [[map {{ServiceID => 7, ServiceOrderID => $_}} @$cids_chunk]], {timeout => $balance_call_timeout || 300});
            push @$balance_info, @$balance_info_chunk;
        }
        $log->out('Got '. scalar @$balance_info .' campaigns from Balance', $balance_info);

        # обновляем продукт на кампаниях в Директе, которых нет в Балансе
        my $not_balance_cids = xminus(\@cids, [ map {$_->{ServiceOrderID}} @$balance_info ]);
        if (@$not_balance_cids) {
            $log->out('Update '. scalar @$not_balance_cids .' campaigns in Direct not found in Balance:', $not_balance_cids);
            my $updated = do_update_table(PPC(cid => $not_balance_cids), 'campaigns',
                {ProductID => $new_product_id, LastChange__dont_quote => 'LastChange'},
                where => {cid => SHARD_IDS, ProductID => $old_product_id}) + 0;
            $log->out('Updated '. $updated .' campaigns in Direct not found in Balance');
        }

        # обновляем продукт на кампаниях, которые есть и в Блансе и в Директе
        foreach my $row (@$balance_info) {
            my $msg_guard = $log->msg_prefix_guard("[agency=$agency_login, agency_id=$agency_id, client=$client_login, client_id=$client_id, cid=$row->{ServiceOrderID}]");
            my $is_updated_in_balance = 0;
            if ($row->{product_id} == $old_product_id) {
                # если в Балансе продукт старый - пробуем обновить
                my $update_orders_data = {
                    AgencyID => $agency_id,
                    ClientID => $client_id,
                    ServiceID => $row->{ServiceID},
                    ServiceOrderID => $row->{ServiceOrderID},
                    Text => $camps_info->{$row->{ServiceOrderID}}->{name},
                    ManagerUID => $camps_info->{$row->{ServiceOrderID}}->{ManagerUID} // 0,
                    RegionID => $clids2data->{$client_id}->{country_region_id},
                    GroupServiceOrderID => $row->{GroupServiceOrderID},
                    unmoderated => $camps_info->{$row->{ServiceOrderID}}->{pay_method_flag},
                    ProductID => $new_product_id,
                    ForceProductChange => 1
                    # is_ua_optimize здесь не указан, т.к. в create_campaigns_balance утверждается, что агентсткие кошельки - отключаемые
                };
                my $balance_response = balance_create_update_orders($OPERATOR_UID, [ $update_orders_data ]);
                if ($balance_response) {
                    $log->out('SUCCESS campaign in Balance updated');
                    $is_updated_in_balance = 1;
                } else {
                    $log->out('ERROR updating campaign in Balance');
                }
            } elsif ($row->{product_id} == $new_product_id) {
                $log->out('SUCCESS campaign already in new ProductID in Balance');
                $is_updated_in_balance = 1;
            } else {
                # если получили что-то странное, то тоже ничего не делаем
                $log->out("ERROR unexpected product_id from Balance: $row->{product_id}");
                next;
            }
            if ($is_updated_in_balance && $camps_info->{$row->{ServiceOrderID}}->{ProductID} == $old_product_id) {
                my $updated = do_update_table(PPC(cid => $row->{ServiceOrderID}), 'campaigns',
                    {ProductID => $new_product_id, LastChange__dont_quote => 'LastChange'},
                    where => {cid => SHARD_IDS, ProductID => $old_product_id}) + 0;
                $log->out('Campaign in Direct updated') if $updated;
            }
        }
    }
}

$log->out('FINISH');





