#!/usr/bin/perl

use Direct::Modern;

use Getopt::Long;
use List::Util qw/min/;
use POSIX qw/strftime/;
use Yandex::DBTools;

use my_inc '../..';
use Settings;
use ScriptHelper;

my @known_tables = qw/
    monitor_values_day
    monitor_values_hour
    monitor_values_sec
/;
my %time_delta_for_table = (
    monitor_values_sec => 3600 / 4,
    monitor_values_hour => 86400,
    monitor_values_day => 86400,
);
my $to_date = strftime("%Y-%m-%d", localtime(time() - 86400));    # вчера
my $from_date;
my $sleep_interval = 2;
my $tables_str = join(",", @known_tables);
my $dry_run = 0;
GetOptions(
    'from-date=s' => \$from_date,
    'to-date=s' => \$to_date,
    'sleep-interval=i' => \$sleep_interval,
    'tables=s' => \$tables_str,
    'time-delta-for-table=s%' => \%time_delta_for_table,
    'dry-run' => \$dry_run,
) || die "can't parse options, stop";
die "invalid --from-date, must be in format YYYY-MM-DD\n" unless !$from_date || $from_date =~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}$/;
die "invalid --to-date, must be in format YYYY-MM-DD\n" unless $to_date =~ /^[0-9]{4}-[0-9]{2}-[0-9]{2}$/;
die "invalid --tables value, mus be list of table names separated by ','\n" unless $tables_str =~ /^[a-z0-9_,]+$/;
my @tables = split /,/, $tables_str;
for my $table (@tables) {
    die "unknown table $table" if !grep { $_ eq $table } @known_tables;
}

$log->out('START');
for my $table (@tables) {
    $log->out("copying $table");
    my $min_ts;
    if ($from_date) {
        $min_ts = `date -d "$from_date" +%s`;
        chomp $min_ts;
    } else {
        $min_ts = get_one_field_sql(PPCLOG, "select unix_timestamp(min(measure_time)) from $table");
        $min_ts = $min_ts - ($min_ts % $time_delta_for_table{$table});
    }
    my $max_ts = `date -d "$to_date" +%s` + 86400;
    my $from_ts = $min_ts;
    while ($from_ts < $max_ts) {
        my $to_ts = min($from_ts + $time_delta_for_table{$table} - 1, $max_ts - 1);
        my $from_time = strftime("%Y-%m-%d %H:%M:%S", gmtime($from_ts));
        my $to_time = strftime("%Y-%m-%d %H:%M:%S", gmtime($to_ts));
        $log->out(sprintf "inserting into %s values from %s to %s", $table, $from_time, $to_time);
        my @recs = map {[$_->{target_id}, $_->{measure_time}, $_->{value}]} @{ get_all_sql(PPCLOG, ["SELECT target_id, measure_time, value FROM $table", where => {measure_time__ge => $from_time, measure_time__le => $to_time}]) };
        if ($dry_run) {
            $log->out(sprintf "skipping inserting %d records in dry-run mode", scalar @recs);
        } else {
            do_mass_insert_sql(MONITOR, "REPLACE INTO $table (target_id, measure_time, value) VALUES %s", \@recs);
            $log->out(sprintf "inserted %d records", scalar @recs);
        }
        $log->out(sprintf "sleeping for %d seconds", $sleep_interval);
        sleep($sleep_interval);
        $from_ts = $to_ts + 1;
    }
}
$log->out("FINISH");
