#!/usr/bin/perl
use strict;
use warnings;

=head1 NAME

create-ppclog-tables-in-advance.pl

=head1 SYNOPSIS

    # Написать в лог, что скрипт собирается делать:
    create-ppclog-tables-in-advance.pl --db ppclog --db ppcpricelog

    # На самом деле создать нужные таблицы
    create-ppclog-tables-in-advance.pl --db ppclog --db ppcpricelog --create

=head1 DESCRIPTION

Скрипт создаёт автоматически создаваемые каждый день или месяц таблицы на несколько
дней вперёд (например, на 2 недели вперёд). Он находит в каталогах в db_schema файлы
*.schema.sql с шаблонами типа YYYYMMDD в названии и создаёт таблицы в соответствующих
базах с соответствующими названиями.

С какими базами он умеет работать — см. ключи %DB_MAP. Если базу явно не указать, скрипт
не работает.

Какие опции поддерживаются:

--db        (можно указывать несколько раз): с какими БД работать, обязательно
--days      на сколько дней вперёд создавать таблицы, по умолчанию 14
--create    на самом деле создать таблицы, по умолчанию они не создаются
--help      показать справку

=cut

use my_inc '../..';

use DateTime;
use File::Spec;
use List::MoreUtils 'uniq';

use Yandex::DBSchema;
use Yandex::DBTools;

use ScriptHelper;
use Settings;

=head1 CONFIGURATION

=head2 %DB_MAP

=cut

my %DB_MAP = (
    ppclog => PPCLOG,
    ppcpricelog => PPCPRICELOG,
);

my $result = run();
exit $result;

=head1 SUBROUTINES/METHODS

=head2 run()

=cut

sub run {
    my @DBS;
    my $DAYS = 14;
    my $DRYRUN = 1;

    extract_script_params(
        'db=s'   => \@DBS,
        'days=i' => \$DAYS,
        'create' => sub { $DRYRUN = 0 },
    );

    die 'Missing required parameter: --db' unless @DBS;

    my @unknown_dbs = grep { ! exists $DB_MAP{$_} } @DBS;
    die "Unknown databases: " . join( ', ', @unknown_dbs ) if @unknown_dbs;

    $log->out('start');
    $log->out( { dbs => \@DBS, days => $DAYS, dryrun => $DRYRUN } );

    my @datetimes = map { DateTime->from_epoch( epoch => time + 86400 * $_ ) } ( 0 .. $DAYS );

    my $db_schema_root = $Settings::ROOT . '/db_schema';
    for my $filename ( map { glob("$db_schema_root/$_/*_YYYY*.schema.sql") } @DBS ) {
        my $relname = File::Spec->abs2rel( $filename, $db_schema_root );

        my ( $dbname, $table_pattern ) = split m{/}, $relname;
        $table_pattern =~ s/\.schema\.sql$//g;

        my @tables;

        my ( $time_format, $pattern );
        if ( $table_pattern =~ /YYYYMMDD/ ) {
            $time_format = '%Y%m%d';
        } elsif ( $table_pattern =~ /YYYYMM/ ) {
            $time_format = '%Y%m';
        } else {
            die "Unknown table pattern: $table_pattern";
        }

        for my $dt (@datetimes) {
            my $stamp = $dt->strftime($time_format);
            push @tables, $table_pattern =~ s/YYYY\w*/$stamp/gr;
        }

        @tables = uniq @tables;

        my $db = $DB_MAP{$dbname};
        @tables = grep { ! is_table_exists( $db, $_ ) } @tables;

        $log->out( "dbname=$dbname, table_pattern=$table_pattern, tables: " . join( ', ', @tables ) );

        unless ($DRYRUN) {
            for my $table (@tables) {
                $log->out("Creating $table like $dbname.$table_pattern");
                create_table_by_schema( $db, $table, like => "$dbname.$table_pattern", if_not_exists => 1);
            }
        }
    }

    $log->out('finish');

    return 0;
}
