#!/usr/bin/perl

use my_inc '../..';

=head1 DESCRIPTION

Исправляем geo после слияния городов.
Можно останавливать и перезапускать.

Список для подмнены берётся из %GeoTools::SUBSTITUTE_GEO_ID

    protected/one-shot/del_outdated_geo.pl


=cut

use warnings;
use strict;
use utf8;

use Settings;
use ScriptHelper;
use Yandex::DBTools;
use Yandex::DBShards;
use ShardingTools;
use Yandex::Retry;
use BS::ResyncQueue;
use GeoTools ();

my $SLEEP_COEF = 1;
my $SHARD;
my $CHUNK_SIZE = 100_000;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'shard-id=s' => \$SHARD,
    'cid=i@' => \my @CIDS,
);


my %shards = ($SHARD) ? ($SHARD => 1) : map { $_ => 1 } ppc_shards();
my $from_re = join q{|} => keys %GeoTools::SUBSTITUTE_GEO_ID;

$log->out('START');

foreach_shard_parallel_verbose(
    $log,
    sub {
        my ($shard) = @_;
        return unless $shards{$shard};
        my $log_guard = $log->msg_prefix_guard("[shard $shard]");

        iterate_adgroups($shard);
        iterate_campaigns($shard);
        iterate_mediaplans($shard);
        iterate_media_groups($shard);

        $log->out("Working");
    }
);

$log->out('FINISH');


sub iterate_adgroups {
    my $shard = shift;
    my $min = get_one_field_sql(PPC(shard => $shard), [
            "select min(pid) from phrases",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $max = get_one_field_sql(PPC(shard => $shard), [
            "select max(pid) from phrases",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $count = 0;
    while ($min <= $max) {
        my $relax = relaxed_guard(times => $SLEEP_COEF);
        my $batch = get_all_sql(PPC(shard => $shard), [
            "select pid, cid, geo from phrases",
            where => {
                (@CIDS ? (cid => \@CIDS) : ()),
                pid__between => [$min, $min + $CHUNK_SIZE],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            },
        ]);
        my @changed;
        for my $adgroup (@$batch) {
            my $new_geo = GeoTools::substitute_temporary_geo($adgroup->{geo});
            next if $new_geo eq $adgroup->{geo};
            $log->out(sprintf "try update geo on pid %d from %s to %s", $adgroup->{pid}, $adgroup->{geo}, $new_geo);
            $adgroup->{geo} = $new_geo;
            $adgroup->{priority} = -30;
            push @changed, $adgroup;
        }
        if (@changed) {
            my $affected = do_update_table(PPC(shard => $shard), 'phrases', {
                geo__dont_quote => sql_case('pid', {map { $_->{pid} => $_->{geo} } @changed})
            }, where => {
                pid => [ map { $_->{pid} } @changed ],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            });
            bs_resync(\@changed);
            $count += @changed;
            $log->out(sprintf "updated %d groups", $affected);
        }
        $min += $CHUNK_SIZE;
        $log->out(sprintf "looked for pid %d/%d", $min, $max);
    }
    $log->out("Changed $count adgroups");
}

sub iterate_campaigns {
    my $shard = shift;
    my $min = get_one_field_sql(PPC(shard => $shard), [
            "select min(cid) from campaigns",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $max = get_one_field_sql(PPC(shard => $shard), [
            "select max(cid) from campaigns",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $count = 0;
    while ($min <= $max) {
        my $relax = relaxed_guard(times => $SLEEP_COEF);
        my $batch = get_all_sql(PPC(shard => $shard), [
            "select cid, geo from campaigns",
            where => {
                (@CIDS ? (cid => \@CIDS) : ()),
                cid__between => [$min, $min + $CHUNK_SIZE],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            },
        ]);
        my @changed;
        for my $camp (@$batch) {
            my $new_geo = GeoTools::substitute_temporary_geo($camp->{geo});
            next if $new_geo eq $camp->{geo};
            $log->out(sprintf "try update geo on cid %d from %s to %s", $camp->{cid}, $camp->{geo}, $new_geo);
            $camp->{geo} = $new_geo;
            push @changed, $camp;
        }
        if (@changed) {
            my $affected = do_update_table(PPC(shard => $shard), 'campaigns', {
                geo__dont_quote => sql_case('cid', {map { $_->{cid} => $_->{geo} } @changed})
            }, where => {
                cid => [ map { $_->{cid} } @changed ],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            });
            $count += @changed;
            $log->out(sprintf "updated %d campaigns", $affected);
        }
        $min += $CHUNK_SIZE;
        $log->out(sprintf "looked for cid %d/%d", $min, $max);
    }
    $log->out("Changed $count campaigns");
}

sub iterate_mediaplans {
    my $shard = shift;
    my $min = get_one_field_sql(PPC(shard => $shard), [
            "select min(mbid) from mediaplan_banners",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $max = get_one_field_sql(PPC(shard => $shard), [
            "select max(mbid) from mediaplan_banners",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $count = 0;
    while ($min <= $max) {
        my $relax = relaxed_guard(times => $SLEEP_COEF);
        my $batch = get_all_sql(PPC(shard => $shard), [
            "select mbid, geo from mediaplan_banners",
            where => {
                (@CIDS ? (cid => \@CIDS) : ()),
                mbid__between => [$min, $min + $CHUNK_SIZE],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            },
        ]);
        my %changes;
        for my $banner (@$batch) {
            my $new_geo = GeoTools::substitute_temporary_geo($banner->{geo});
            next if $new_geo eq $banner->{geo};
            $log->out(sprintf "try update geo on mbid %d from %s to %s", $banner->{mbid}, $banner->{geo}, $new_geo);
            $changes{$banner->{mbid}} = $new_geo;
            $count++;
        }
        if (%changes) {
            my $affected = do_update_table(PPC(shard => $shard), 'mediaplan_banners', {
                geo__dont_quote => sql_case(mbid => \%changes)
            },
            where => {
                mbid => [keys %changes],
                geo__rlike => "(^|,)-?($from_re)(,|\$)",
            });
            $log->out(sprintf "updated %d mediaplan_banners", $affected);
        }
        $min += $CHUNK_SIZE;
        $log->out(sprintf "looked for mbid %d/%d", $min, $max);
    }
    $log->out("Changed $count mediaplan banners");
}

sub iterate_media_groups {
    my $shard = shift;
    my $min = get_one_field_sql(PPC(shard => $shard), [
            "select min(mgid) from media_groups",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $max = get_one_field_sql(PPC(shard => $shard), [
            "select max(mgid) from media_groups",
            @CIDS ? (where => {cid => \@CIDS}) : (),
        ]) // 0;
    my $count = 0;
    while ($min <= $max) {
        my $relax = relaxed_guard(times => $SLEEP_COEF);
        my $batch = get_all_sql(PPC(shard => $shard), [
            "select mgid, geo from media_groups",
            where => {
                (@CIDS ? (cid => \@CIDS) : ()),
                mgid__between => [$min, $min + $CHUNK_SIZE],
                geo__rlike => "(^|,)-?($from_re)(,|\$)"
            },
        ]);
        my %changes;
        for my $group (@$batch) {
            my $new_geo = GeoTools::substitute_temporary_geo($group->{geo});
            next if $new_geo eq $group->{geo};
            $log->out(sprintf "try update geo on mgid %d from %s to %s", $group->{mgid}, $group->{geo}, $new_geo);
            $changes{$group->{mgid}} = $new_geo;
            $count++;
        }
        if (%changes) {
            my $affected = do_update_table(PPC(shard => $shard), 'media_groups', {
                geo__dont_quote => sql_case(mgid => \%changes),
                statusBsSynced => 'No',
            }, where => {
                mgid => [keys %changes],
                geo__rlike => "(^|,)-?($from_re)(,|\$)",
            });
            $log->out(sprintf "updated %d media_groups", $affected);
        }
        $min += $CHUNK_SIZE;
        $log->out(sprintf "looked for mgid %d/%d", $min, $max);
    }
    $log->out("Changed $count media groups");
}


